/* glpapi/glp_create_item.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_create_item - create new row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_create_item(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_create_item routine creates new row with
-- the given name. Initially the created row has GLP_FX type (i.e. the
-- auxiliary variable is fixed) with both lower and upper bounds equal
-- to zero (such row corresponds to the linear equality constraint with
-- zero right-hand side). Being created new row becomes the current row.
--
-- If what = GLP_COL, the glp_create_item routine creates new column
-- with the given name. Initially the created column has GLP_LO type
-- with lower bound equal to zero (such column corresponds to the
-- non-negative structural variable). Being created new column becomes
-- the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - row/column with the given name already exists;
-- 2 - the parameter what or the given name is invalid. */

int glp_create_item(int what, char *name)
{     if (what == GLP_ROW)
      {  /* create new row */
         STR *row_name;
         GLPITEM *row;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->row_tab, row_name) != NULL)
         {  /* row already exists */
            delete_str(row_name);
            return 1;
         }
         row = get_atom(glp->item_pool);
         row->name = row_name;
         row->type = GLP_FX;
         row->lb = row->ub = 0.0;
         row->ptr = NULL;
         row->seqn = 0;
         row->tagx = GLP_BS;
         row->valx = row->dx = 0.0;
         row->next = NULL;
         /* add to the symbol table */
         node = insert_by_key(glp->row_tab, row->name);
         node->link = row;
         /* add to the end of the linked list */
         if (glp->first_row == NULL)
            glp->first_row = row;
         else
            glp->last_row->next = row;
         glp->last_row = row;
         /* new row becomes the current row */
         glp->this_row = row;
      }
      else if (what == GLP_COL)
      {  /* create new column */
         STR *col_name;
         GLPITEM *col;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->col_tab, col_name) != NULL)
         {  /* column already exists */
            delete_str(col_name);
            return 1;
         }
         col = get_atom(glp->item_pool);
         col->name = col_name;
         col->type = GLP_LO;
         col->lb = col->ub = 0.0;
         col->ptr = NULL;
         col->seqn = 0;
         col->tagx = GLP_BS;
         col->valx = col->dx = 0.0;
         col->next = NULL;
         /* add to the symbol table */
         node = insert_by_key(glp->col_tab, col->name);
         node->link = col;
         /* add to the end of the linked list */
         if (glp->first_col == NULL)
            glp->first_col = col;
         else
            glp->last_col->next = col;
         glp->last_col = col;
         /* new column becomes the current column */
         glp->this_col = col;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/* eof */
