/* glpmat/scrape_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- scrape_mat - remove tiny elements from sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- int scrape_mat(MAT *A, double eps);
--
-- *Description*
--
-- The scrape_mat routine removes from sparse matrix A all elements,
-- whose magnitudes are not greater than eps.
--
-- If the parameter eps is equal to 0.0, only zero elements will be
-- removed from the matrix.
--
-- *Returns*
--
-- The scrape_mat routine returns the total number of elements, which
-- have been removed from the matrix. */

int scrape_mat(MAT *A, double eps)
{     ELEM *e, *ee;
      int i, j, count = 0;
      /* nullify all tiny elements */
      for (i = 1; i <= A->m; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
            if (fabs(e->val) < eps) e->val = 0.0;
      }
      /* remove zero elements from row linked lists */
      for (i = 1; i <= A->m; i++)
      {  ee = NULL;
         while (A->row[i] != NULL)
         {  e = A->row[i], A->row[i] = e->row;
            if (e->val == 0.0)
               count++;
            else
               e->row = ee, ee = e;
         }
         A->row[i] = ee;
      }
      /* remove zero elements from column linked lists */
      for (j = 1; j <= A->n; j++)
      {  ee = NULL;
         while (A->col[j] != NULL)
         {  e = A->col[j], A->col[j] = e->col;
            if (e->val == 0.0)
               free_atom(A->pool, e);
            else
               e->col = ee, ee = e;
         }
         A->col[j] = ee;
      }
      return count;
}

/* eof */
