/* glpmat/mprd_numb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- mprd_numb - multiply matrices (C := A * B); numeric phase
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *mprd_numb(MAT *C, MAT *A, MAT *B, double _work[]);
--
-- *Description*
--
-- The mprd_numb routine computes the product C = A * B numerically,
-- assuming that the pattern of the matrix C (i.e. distribution of its
-- non-zero elements) is known. (Symbolic phase of this operation can
-- be performed by means of the mprd_symb routine.)
--
-- Input matrices A and B remain unchanged on exit. Number of columns of
-- the matrix A should be equal to number of rows of the matrix B.
--
-- The resultant matrix C should not coincide neither with matrix A nor
-- with matrix B. Number of rows of C should be equal number of rows of
-- A, number of columns of C should be equal number of columns of B. On
-- entry the distribution of non-zero elements of the matrix C should be
-- correct, however, numeric values of elements are not essential. On
-- exit the matrix C has the same pattern (note that some elements may
-- be equal to zero as result of numeric cancellation, i.e. the matrix C
-- may contain explicit zeros).
--
-- The auxiliary array work should contain at least 1+n elements, where
-- n is number of columns of the matrix A. If this parameter is NULL,
-- the mprd_numb routine automatically allocates and frees the working
-- array.
--
-- *Returns*
--
-- The mprd_symb routine returns a pointer to the matrix C. */

MAT *mprd_numb(MAT *C, MAT *A, MAT *B, double _work[])
{     ELEM *e, *ee;
      int i, j;
      double *work = _work;
      if (C == A || C == B)
         fault("mprd_numb: invalid specification of resultant matrix");
      if (!(C->m == A->m && A->n == B->m && B->n == C->n))
         fault("mprd_numb: inconsistent dimension; product undefined");
      if (_work == NULL) work = ucalloc(1+A->n, sizeof(double));
      for (j = 1; j <= A->n; j++) work[j] = 0;
      for (i = 1; i <= C->m; i++)
      {  /* work := i-th row of A */
         for (e = A->row[i]; e != NULL; e = e->row) work[e->j] = e->val;
         /* compute i-th row of C */
         for (e = C->row[i]; e != NULL; e = e->row)
         {  /* c[i,j] = a[i,*] * b[*,j] */
            double sum = 0.0;
            for (ee = B->col[e->j]; ee != NULL; ee = ee->col)
               sum += work[ee->i] * ee->val;
            e->val = sum;
         }
         /* clear working array */
         for (e = A->row[i]; e != NULL; e = e->row) work[e->j] = 0.0;
      }
      if (_work == NULL) ufree(work);
      return C;
}

/* eof */
