/* glphbsm/hbsm_to_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glphbsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- hbsm_to_mat - convert sparse matrix from HBSM to MAT.
--
-- *Synopsis*
--
-- #include "glphbsm.h"
-- MAT *hbsm_to_mat(HBSM *hbsm);
--
-- *Description*
--
-- The hbsm_to_mat routine converts the sparse matrix from the object of
-- HBSM type (Harwell-Boeing format) to the object of MAT type (standard
-- format).
--
-- In case of symmetric (or skew symmetric) matrices the hbsm_to_mat
-- routine forms both lower and upper traingles of the result matrices.
-- In case of matrices that have pattern only the hbsm_to_mat routine
-- set all non-zero elements of the result matrices to 1.
--
-- This version of the hbsm_to_mat routine can't convert complex and
-- elemental (unassembled) matrices.
--
-- *Returns*
--
-- If the conversion was successful the hbsm_to_mat routine returns a
-- pointer to an object of MAT type that presents the result matrix in
-- standard format. Otherwise the routine returns NULL. */

MAT *hbsm_to_mat(HBSM *hbsm)
{     MAT *A = NULL;
      int i, j, k;
      double val;
      if (hbsm->nrow < 1)
      {  error("hbsm_to_mat: invalid number of rows");
         goto fail;
      }
      if (hbsm->ncol < 1)
      {  error("hbsm_to_mat: invalid number of columns");
         goto fail;
      }
      if (strcmp(hbsm->mxtype, "RSA") == 0)
      {  /* real symmetric assembled */
         if (hbsm->nrow != hbsm->ncol)
err1:    {  error("hbsm_to_mat: number of rows not equal to number of c"
               "olumns");
            goto fail;
         }
         if (hbsm->colptr == NULL)
err2:    {  error("hbsm_to_mat: array `colptr' not allocated");
            goto fail;
         }
         if (hbsm->rowind == NULL)
err3:    {  error("hbsm_to_mat: array `rowind' not allocated");
            goto fail;
         }
         if (hbsm->values == NULL)
err4:    {  error("hbsm_to_mat: array `values' not allocated");
            goto fail;
         }
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero))
err5:       {  error("hbsm_to_mat: invalid column pointer");
               goto fail;
            }
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow))
err6:       {  error("hbsm_to_mat: invalid row index");
               goto fail;
            }
            if (i < j)
err7:       {  error("hbsm_to_mat: invalid matrix structure");
               goto fail;
            }
            val = hbsm->values[k];
            if (val != 0.0)
            {  new_elem(A, i, j, val);
               if (i != j) new_elem(A, j, i, val);
            }
         }
      }
      else if (strcmp(hbsm->mxtype, "RUA") == 0)
      {  /* real unsymmetric assembled */
         if (hbsm->nrow != hbsm->ncol) goto err1;
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         if (hbsm->values == NULL) goto err4;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            val = hbsm->values[k];
            if (val != 0.0) new_elem(A, i, j, val);
         }
      }
      else if (strcmp(hbsm->mxtype, "RZA") == 0)
      {  /* real skew symmetric assembled */
         if (hbsm->nrow != hbsm->ncol) goto err1;
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         if (hbsm->values == NULL) goto err4;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            if (i <= j) goto err7;
            val = hbsm->values[k];
            if (val != 0.0)
            {  new_elem(A, i, j, +val);
               new_elem(A, j, i, -val);
            }
         }
      }
      else if (strcmp(hbsm->mxtype, "RRA") == 0)
      {  /* real rectangular assembled */
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         if (hbsm->values == NULL) goto err4;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            val = hbsm->values[k];
            if (val != 0.0) new_elem(A, i, j, val);
         }
      }
      else if (strcmp(hbsm->mxtype, "PSA") == 0)
      {  /* pattern symmetric assembled */
         if (hbsm->nrow != hbsm->ncol) goto err1;
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            if (i < j) goto err7;
            val = 1.0;
            new_elem(A, i, j, val);
            if (i != j) new_elem(A, j, i, val);
         }
      }
      else if (strcmp(hbsm->mxtype, "PUA") == 0)
      {  /* pattern unsymmetric assembled */
         if (hbsm->nrow != hbsm->ncol) goto err1;
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            val = 1.0;
            new_elem(A, i, j, val);
         }
      }
      else if (strcmp(hbsm->mxtype, "PRA") == 0)
      {  /* pattern rectangular assembled */
         if (hbsm->colptr == NULL) goto err2;
         if (hbsm->rowind == NULL) goto err3;
         A = create_mat(hbsm->nrow, hbsm->ncol);
         for (j = 1; j <= hbsm->ncol; j++)
         for (k = hbsm->colptr[j]; k < hbsm->colptr[j+1]; k++)
         {  if (!(1 <= k && k <= hbsm->nnzero)) goto err5;
            i = hbsm->rowind[k];
            if (!(1 <= i && i <= hbsm->nrow)) goto err6;
            val = 1.0;
            new_elem(A, i, j, val);
         }
      }
      else
      {  error("hbsm_to_mat: can't convert matrix of type `%s'",
            hbsm->mxtype);
         goto fail;
      }
      return A;
fail: if (A != NULL) delete_mat(A);
      return NULL;
}

/* eof */
