/* glpgel/gel_ft.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>
#include <stddef.h>
#include "glpgel.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- gel_ft - sparse gaussian elimination (Forrest & Tomlin technique).
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- int gel_ft(PER *P, MAT *V, PER *Q, int k1, int k2,
--    void (*func)(int i, int k, double f), double eps, int *Unz,
--    double work[]);
--
-- *Description*
--
-- The gel_ft routine uses Forrest & Tomlin version of the gaussian
-- elimination technique to transform the matrix U = P*V*Q to the upper
-- triangular form, where V is the given sparse matrix, P and Q are the
-- given permutation matrices.
--
-- (The matrix U has implicit representation, therefore all operations
-- are actually performed on the matrix V, that differs from U only in
-- the order of rows and columns determining by the permutation matrices
-- P and Q.)
--
-- It is assumed that on entry the matrix U has the following special
-- structure:
-- It is assumed that on entry the matrix U has the following special
-- structure:
--
--     1   k1       k2   n
-- 1   x x x x x x x x x x
--     . x x x x x x x x x
-- k1  . . * * * * * * * *
--     . . * * * * * * * *
--     . . * . * * * * * *
--     . . * . . * * * * *
--     . . * . . . * * * *
-- k2  . . * . . . . * * *
--     . . . . . . . . x x
-- n   . . . . . . . . . x
--
-- Hence only the submatrix with rows k1, k1+1, ..., k2 and columns
-- k1, k1+1, ..., n should be transformed (elements of this submatrix
-- are marked by '*', all other elements of the matrix U are marked by
-- 'x').
--
-- The result of the transformation is the matrices P', V', and Q' that
-- define the upper triangular matrix U' = P'*V'*Q' (these new matrices
-- are stored instead the source matrices P, V, and Q).
--
-- To eliminate subdiagonal elements of the matrix U the routine applies
-- to this matrix elementary transformations of the following types: row
-- permutation, column permutation, and subtraction one row multiplied
-- by a number from other row (gaussian transformation). Each time when
-- gaussian transformation is applied to the matrix U (in order to
-- eliminate one subdiagonal element), the gel_bg routine calls the func
-- routine in order that the calling routine would have possibility to
-- accumulate all information about these transformations (information
-- about row and column permutations are accumulated in the matrices P
-- and Q by the gel_bg routine).
--
-- The point where the func routine is called, and the meaning of the
-- parameters passed to this routine can be explained by means of the
-- following general scheme of elimination process.
--
-- Before elimination the routine shifts columns k1+1, k1+2, ..., k2 of
-- the matrix U by one position to the left and upwards and also moves
-- k1-th row and k1-th column to the position k2. As a result of such
-- symmetric permutation of rows and columns the matrix U becomes the
-- following:
--
--     1   k1       k2   n
-- 1   x x x x x x x x x x
--     . x x x x x x x x x
-- k1  . . * * * * * * * *
--     . . . * * * * * * *
--     . . . . * * * * * *
--     . . . . . * * * * *
--     . . . . . . * * * *
-- k2  . . * * * * * * * *
--     . . . . . . . . x x
-- n   . . . . . . . . . x
--
-- Then the routine uses gaussian elimination to nullify all subdiagonal
-- elements of the matrix U.
--
-- Forrest & Tomlin technique assumes no pivoting. On k-th elimination
-- step (k = k1, k1+1, ..., k2-1) the diagonal element u[k,k] is always
-- used as the pivot. It's obvious that k-th step consists of nullifying
-- the element u[k2,k] (if this element differs from zero).
--
-- Forrest & Tomlin technique involves no filling in, that is the
-- important advantage of this technique, but from the other hand this
-- technique in many cases is less numerically stable because of there
-- is no pivoting.
--
-- So, the general scheme of the elimination process is the following:
--
-- (* permute symmetrically rows and columns of the matrix U *)
-- ...;
-- for (k = k1; k < k2; k++)
-- {  (* k-th elimination step starts here *)
--    (* the pivot element is always u[k,k] *)
--    assert(u[k,k] != 0.0);
--    (* eliminate non-zero subdiagonal element u[k2,k] *)
--    i = k2;
--    if (u[i,k] == 0.0) continue;
--    (* compute gaussian multiplier *)
--    f = u[i,k] / u[k,k];
--    (* (i-th row of U) := (i-th row of U) - f * (k-th row of U) *)
--    ...;
--    assert(u[i,k] == 0.0);
--    (* elementary transformation of i-th row of the matrix U has been
--       performed *)
--    func(i, k, f);
-- }
-- assert(u[k2,k2] != 0.0);
--
-- Should note that row numbers passed to the func routine correspond to
-- the matrix U (not to V!). The correspondence between row and column
-- numbers of the matrices U = P*V*Q and V is determined by the row
-- permutation matrix P and the column permutation matrix Q. So, if an
-- element u[i,j] of the matrix U corresponds to an element v[i',j'] of
-- the matrix V, the following formulae take a place:
--
-- i' = P->row[i], j' = Q->col[j], i = P->col[i'], j = Q->row[j'].
--
-- Since the using Forrest and Tomlin technique changes only one row of
-- the matrix U (which has the number k2), this routine uses simplified
-- checking for the "quality" of the transformation. The transformation
-- is considered as satisfactory (from the numerical point of view) if
-- after the transformation has been finished the following condition is
-- true:
--
-- |u'[k2,k2]| >= eps * big > 0,
--
-- where eps is the given relative tolerance, big is the largest of
-- absolute values of elements which appeared in the transformed row
-- during all eliminating process. (This condition gives some guarantee
-- that the solution of the system with the matrix U' will be relatively
-- accurate.)
--
-- Should note that the routine checks the element u[k,k] only if the
-- element u[k2,k] is not zero, i.e. if gaussian transformation should
-- be actually applied.
--
-- On entry the variable Unz should define the total number of non-zeros
-- in the source matrix U. On exit this variable will define the total
-- number of non-zeros in the transformed matrix U'. (If on entry Unz is
-- set to zero, on exit Unz will define the difference between total
-- numbers of non-zeros in the matrices U' and U.)
--
-- The auxiliary array work should have at least n elements, where n is
-- the order of the matrix U.
--
-- *Returns*
--
-- The gel_ft routine returns one of the following error codes:
--
--  0 - no errors;
-- -k - if k1 <= k < k2, on k-th elimination step the pivot element
--      u[k,k] is found equal to zero; if k = k2, after the last step
--      the element u[k2,k2] is found equal to zero;
-- +k - in this case always k = k2; this means that for the diagonal
--      element u[k2,k2] the numerical stability condition (see above)
--      has been violated.
--
-- Should note that if the matrix U passed to the gel_ft routine is
-- already a result of preceding transformations of some initial matrix
-- U0, the appearence of numerical errors in this routine is usually
-- connected with all sequence of transformations. Hence in such case
-- instead this routine the general gaussian elimination should be
-- applied directly to the initial matrix U0.
--
-- *Reference*
--
-- Forrest J.J.H, Tomlin J.A. Updating triangular factors of the basis
-- to maintain sparsity in the product-form simplex method. Math.Prog.,
-- 2 (1972), pp. 263-78. */

#define iU(i) (P->col[i])
/* converts row number of V to row number of U */

#define iV(i) (P->row[i])
/* converts row number of U to row number of V */

#define jU(j) (Q->row[j])
/* converts column number of V to column number of U */

#define jV(j) (Q->col[j])
/* converts column number of U to column number of V */

int gel_ft(PER *P, MAT *V, PER *Q, int k1, int k2,
      void (*func)(int i, int k, double f), double eps, int *Unz,
      double work[])
{     ELEM *vpq, *e, *ee, *en;
      int n = V->m, i, j, k, p, q, ret = 0;
      double f, big, drop = 1e-15;
      if (V->m != V->n)
         fault("gel_ft: transformed matrix is not quadratic");
      if (!(P->n == n && Q->n == n))
         fault("gel_ft: permutation matrices have invalid order");
      if (!(1 <= k1 && k1 <= k2 && k2 <= n))
         fault("gel_ft: initial active submatrix has invalid position");
      /* shift rows and columns k1+1, k1+2, ..., k2 of the matrix U by
         one position to the left and upwards, and move the row and the
         column having number k1 to the position k2 */
      for (k = k1; k < k2; k++)
      {  int t1, t2;
         /* permute k-th and k+1-th rows of the matrix U */
         t1 = iV(k); t2 = iV(k+1);
         iV(k)   = t2; iU(t2) = k;
         iV(k+1) = t1; iU(t1) = k+1;
         /* permute k-th and k+1-th columns of the matrix U */
         t1 = jV(k); t2 = jV(k+1);
         jV(k)   = t2; jU(t2) = k;
         jV(k+1) = t1; jU(t1) = k+1;
      }
      /* clear auxiliary array */
      for (j = 1; j <= n; j++) work[j] = 0.0;
      /* k2-th row of U = i-th row of V */
      i = iV(k2);
      /* work := (i-th row of V) */
      big = 0.0; /* = max|v[i,*]| */
      for (e = V->row[i]; e != NULL; e = e->row)
      {  if (jU(e->j) < k1 || e->val == 0.0)
err:        fault("gel_ft: transformed matrix has invalid pattern");
         work[e->j] = e->val;
         if (big < fabs(e->val)) big = fabs(e->val);
      }
      /* main loop of gaussian elimination */
      for (k = k1; k < k2; k++)
      {  /* the goal of k-th step is to nullify the subdiagonal element
            u[k2,k] */
         p = iV(k); /* k-h row of U     = p-th row of V */
         q = jV(k); /* k-th column of U = q-th column of V */
         /* if u[k2,k] = 0, no elimination is needed */
         if (work[q] == 0.0) continue;
         /* search for u[k,k] = v[p,q] */
         vpq = NULL;
         for (e = V->row[p]; e != NULL; e = e->row)
         {  if (jU(e->j) < k || e->val == 0.0) goto err;
            if (e->j == q) vpq = e;
         }
         if (vpq == NULL)
         {  /* the diagonal element u[k,k] is equal to zero */
            ret = -k;
            goto done;
         }
         /* compute gaussian multiplier f = v[i,q]/v[p,q] */
         f = work[q] / vpq->val;
         /* work := work - f * (p-th row of V) */
         for (e = V->row[p]; e != NULL; e = e->row)
         {  work[e->j] -= f * e->val;
            /* if |v[i,j]| < drop * max|v[i,*]|, where drop is relative
               drop tolerance, the element v[i,j] can be replaced by
               exact zero; this slightly decreases accuracy but allows
               to improve sparsity */
            if (fabs(work[e->j]) < drop * big) work[e->j] = 0.0;
            if (big < fabs(work[e->j])) big = fabs(work[e->j]);
         }
         /* u[k2,k] = v[i,q] should be exact zero */
         work[q] = 0.0;
         /* elementary gaussian transformation of i-th row of the matrix
            V has been performed */
         func(iU(i) /* = k2 */, k, f);
         /* k-th elimination step is finished */
      }
      /* check the numerical stability condition for the last diagonal
         element u[k2,k2] */
      j = jV(k2);
      if (work[j] == 0.0)
      {  ret = -k2;
         goto done;
      }
      if (fabs(work[j]) < eps * big)
      {  ret = +k2;
         goto done;
      }
      /* replace existing elements of i-th row of the matrix V */
      for (e = V->row[i]; e != NULL; e = e->row)
      {  j = e->j;
         e->val = work[j], work[j] = 0.0;
      }
      /* create new elements, which appeared in i-th row as a result
         of elimination */
      for (j = 1; j <= n; j++)
         if (work[j] != 0.0) new_elem(V, i, j, work[j]), (*Unz)++;
      /* remove zero elements, which appeared in i-th row as a result
         of numerical cancellation */
      for (e = V->row[i], V->row[i] = NULL; e != NULL; e = en)
      {  en = e->row;
         if (e->val != 0.0)
         {  /* v[i,j] != 0, keep this element */
            e->row = V->row[i], V->row[i] = e;
            continue;
         }
         /* v[i,j] = 0; remove this element from the list of j-th
            column of the matrix V */
         j = e->j;
         if (V->col[j] == e)
            V->col[j] = e->col;
         else
         {  for (ee = V->col[j]; ee != NULL; ee = ee->col)
              if (ee->col == e) break;
            assert(ee != NULL);
            ee->col = e->col;
         }
         /* return the element v[i,j] to the memory pool */
         free_atom(V->pool, e), (*Unz)--;
      }
done: /* return to the calling program */
      return ret;
}

/* eof */
