/*
 * StopWatch.cs - Implementation of the "StopWatch" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.StopWatch"/> class provides
/// a facility to measure the elapsed time of a benchmark.</para>
/// </summary>
public class StopWatch
{
	// Internal state.
	private DateTime start;
	private DateTime stop;
	private bool running;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.StopWatch"/>
	/// instance.</para>
	/// </summary>
	public StopWatch()
	{
		stop = DateTime.Now;
		start = stop;
		running = false;
	}

	/// <summary>
	/// <para>Start the stop-watch running.</para>
	/// </summary>
	public void Start()
	{
		if(!running)
		{
			stop = DateTime.Now;
			start = stop;
			running = true;
		}
	}

	/// <summary>
	/// <para>Stop the stop-watch running.</para>
	/// </summary>
	public void Stop()
	{
		if(running)
		{
			stop = DateTime.Now;
			running = false;
		}
	}

	/// <summary>
	/// <para>Clear the stop-watch state.  If the stop-watch is running,
	/// then it will start measuring again from the current time.</para>
	/// </summary>
	public void Clear()
	{
		stop = DateTime.Now;
		start = stop;
	}

	/// <summary>
	/// <para>Get the elapsed time.  The value will be only as accurate as
	/// the "Now" property in <see cref="T:System.DateTime"/>.  The accuracy
	/// of that value is system-dependent.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The elapsed time in ticks.</para>
	/// </value>
	public TimeSpan Elapsed
	{
		get
		{
			if(running)
			{
				stop = DateTime.Now;
			}
			return stop - start;
		}
	}

} // class StopWatch

} // namespace PNetMark
