/*
 * DrawingContext.cs - Context for drawing into a curses window.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

public class DrawingContext
{
	// Internal state.
	private Window window;
	private IntPtr handle;
	private Attributes attrs;
	private static short[] colorPairs;
	private static short nextPair;

	/// <summary>
	/// <para>Constructs a new <see cref="T:Curses.DrawingContext"/> instance,
	/// and attached it to a specific <see cref="T:Curses.Window"/>.</para>
	/// </summary>
	///
	/// <param name="handle">
	/// <para>The handle for the window being drawn into.</para>
	/// </param>
	internal DrawingContext(Window window, IntPtr handle)
			{
				this.window = window;
				this.handle = handle;
				this.attrs = Attributes.A_NORMAL;
				if(handle != (IntPtr)0)
				{
					Curses.Native.wattrset(handle, (int)(Attributes.A_NORMAL));
				}
			}

	/// <summary>
	/// <para>Get the window that corresponds to this drawing context.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The window corresponding to this drawing context.</para>
	/// </value>
	public Curses.Window Window
			{
				get
				{
					return window;
				}
			}

	/// <summary>
	/// <para>Move the cursor position.</para>
	/// </summary>
	///
	/// <param name="x">
	/// <para>The X co-ordinate for the new cursor position.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate for the new cursor position.</para>
	/// </param>
	public void Move(int x, int y)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wmove(handle, y, x);
				}
			}

	/// <summary>
	/// <para>Add a single character at the cursor position, using
	/// the current attributes.</para>
	/// </summary>
	///
	/// <param name="ch">
	/// <para>The character to be added.</para>
	/// </param>
	public void Add(char ch)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.waddch(handle, ((uint)ch) | ((uint)attrs));
				}
			}

	/// <summary>
	/// <para>Add a single character at the cursor position,
	/// using a specific set of attributes.</para>
	/// </summary>
	///
	/// <param name="ch">
	/// <para>The character to be added.</para>
	/// </param>
	///
	/// <param name="attrs">
	/// <para>The attributes to be used.</para>
	/// </param>
	public void Add(char ch, Attributes attrs)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.waddch(handle, ((uint)ch) | ((uint)attrs));
				}
			}

	/// <summary>
	/// <para>Add a string at the cursor position, using the current
	/// attributes.</para>
	/// </summary>
	///
	/// <param name="str">
	/// <para>The string to be added.</para>
	/// </param>
	public void Add(String str)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.waddstr(handle, str);
				}
			}

	/// <summary>
	/// <para>Add a string at the cursor position, using a specific
	/// set of attributes.</para>
	/// </summary>
	///
	/// <param name="str">
	/// <para>The string to be added.</para>
	/// </param>
	///
	/// <param name="attrs">
	/// <para>The attributes to be used.</para>
	/// </param>
	public void Add(String str, Attributes attrs)
			{
				if(handle != (IntPtr)0)
				{
					if(attrs != this.attrs)
					{
						Curses.Native.wattrset(handle, (int)attrs);
					}
					Curses.Native.waddstr(handle, str);
					if(attrs != this.attrs)
					{
						Curses.Native.wattrset(handle, (int)(this.attrs));
					}
				}
			}

	/// <summary>
	/// <para>Add a line drawing character at the cursor position.</para>
	/// </summary>
	///
	/// <param name="ch">
	/// <para>The character to be added.</para>
	/// </param>
	public void Add(LineDrawingChar ch)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.CursesHelpAddLineChar(handle, (int)ch);
				}
			}

	/// <summary>
	/// <para>Echo a single character to the cursor position, and
	/// then immediately refresh the window.</para>
	/// </summary>
	///
	/// <param name="ch">
	/// <para>The character to be echo'ed.</para>
	/// </param>
	public void Echo(char ch)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wechochar(handle, (uint)ch);
				}
			}

	/// <summary>
	/// <para>Echo a string to the cursor position, and
	/// then immediately refresh the window.</para>
	/// </summary>
	///
	/// <param name="str">
	/// <para>The string to be echo'ed.</para>
	/// </param>
	public void Echo(String str)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.waddstr(handle, str);
					Curses.Native.wrefresh(handle);
				}
			}

	/// <summary>
	/// <para>Refresh the contents of the window associated with
	/// this drawing context.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method must be called to update the actual screen
	/// with any changes that have been made to the window's contents.</para>
	/// </remarks>
	public void Refresh()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wrefresh(handle);
				}
			}

	/// <summary>
	/// <para>Change a specific set of attributes to enable or
	/// disable them.</para>
	/// </summary>
	///
	/// <param name="attrs">
	/// <para>The attributes to be enabled or disabled.</para>
	/// </param>
	///
	/// <param name="enable">
	/// <para><see langword="true"/> to enable the attributes, or
	/// <see langword="false"/> to disable them.</para>
	/// </param>
	public void ChangeAttrs(Attributes attrs, bool enable)
			{
				Attributes newAttrs;
				if(enable)
				{
					newAttrs = (this.attrs | attrs);
				}
				else
				{
					newAttrs = (this.attrs & ~attrs);
				}
				if(newAttrs != this.attrs && handle != (IntPtr)0)
				{
					Curses.Native.wattrset(handle, (int)newAttrs);
				}
				this.attrs = newAttrs;
			}

	/// <summary>
	/// <para>Get or set "stand out" mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para><see langword="true"/> if "stand out" mode is enabled,
	/// or <see langword="false"/> otherwise.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>Setting this property to <see langword="false"/> will
	/// disable all attributes and return to "normal" mode.</para>
	/// </remarks>
	public bool StandOut
			{
				get
				{
					return ((attrs & Attributes.A_STANDOUT) != 0);
				}
				set
				{
					if(value)
					{
						ChangeAttrs(Attributes.A_STANDOUT, true);
					}
					else
					{
						ChangeAttrs(attrs, false);
					}
				}
			}

	/// <summary>
	/// <para>Allocate a pair of foreground and background colors.</para>
	/// </summary>
	///
	/// <param name="fg">
	/// <para>The foreground color.</para>
	/// </param>
	///
	/// <param name="bg">
	/// <para>The background color.</para>
	/// </param>
	///
	/// <returns>
	/// <para>The pair value.</para>
	/// </returns>
	internal static short AllocColorPair(Color fg, Color bg)
			{
				if(Curses.Native.has_colors() == 0)
				{
					// Cannot set terminal colors, so use the default pair.
					return (short)0;
				}
				if(fg == Color.Default && bg == Color.Default)
				{
					// Pair 0 always contains the defaults.
					return (short)0;
				}
				else
				{
					// See if we already have a pair for the color.
					int index;
					short pair;
					if(colorPairs == null)
					{
						colorPairs = new short [81];
						for(index = 0; index < 81; ++index)
						{
							colorPairs[index] = -1;
						}
						nextPair = 1;
					}
					index = (((int)fg) + 1) + (((int)bg) + 1) * 9;
					pair = colorPairs[index];
					if(pair == -1)
					{
						// Allocate a new color pair.
						if(nextPair < Curses.Native.CursesHelpGetColorPairs())
						{
							// Set this pair, and record it for later.
							Curses.Native.init_pair(nextPair,
													(short)fg, (short)bg);
							colorPairs[index] = nextPair;
							pair = nextPair++;
						}
						else
						{
							// No more pairs available, so use the default.
							pair = 0;
						}
					}
					return pair;
				}
			}

	/// <summary>
	/// <para>Get the color contents of a specific pair.</para>
	/// </summary>
	///
	/// <param name="pair">
	/// <para>The curses pair value to be inspected.</para>
	/// </param>
	///
	/// <param name="fg">
	/// <para>The foreground color within the color pair.</para>
	/// </param>
	///
	/// <param name="bg">
	/// <para>The background color within the color pair.</para>
	/// </param>
	internal static void PairContent(short pair, out Color fg, out Color bg)
			{
				short sfg, sbg;
				Curses.Native.pair_content(pair, out sfg, out sbg);
				if((sfg & 255) == 255)
				{
					fg = Color.Default;
				}
				else
				{
					fg = (Color)sfg;
				}
				if((sbg & 255) == 255)
				{
					bg = Color.Default;
				}
				else
				{
					bg = (Color)sbg;
				}
			}

	/// <summary>
	/// <para>Set a pair of foreground and background colors.</para>
	/// </summary>
	///
	/// <param name="fg">
	/// <para>The foreground color.</para>
	/// </param>
	///
	/// <param name="bg">
	/// <para>The background color.</para>
	/// </param>
	public void SetColorPair(Color fg, Color bg)
			{
				Attributes newAttrs;
				newAttrs = (attrs & ~(Attributes.A_COLOR));
				newAttrs |= (Attributes)(((int)(AllocColorPair(fg, bg))) << 8);
				if(newAttrs != attrs && handle != (IntPtr)0)
				{
					Curses.Native.wattrset(handle, (int)newAttrs);
				}
				attrs = newAttrs;
			}

	/// <summary>
	/// <para>Get or set the current foreground color.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The color value.</para>
	/// </value>
	public Color Foreground
			{
				get
				{
					if(Curses.Native.has_colors() != 0)
					{
						short pair =
							(short)((((int)(attrs & Attributes.A_COLOR)) >> 8));
						Color fg, bg;
						PairContent(pair, out fg, out bg);
						return fg;
					}
					else
					{
						return Color.Default;
					}
				}
				set
				{
					if(Curses.Native.has_colors() != 0)
					{
						short pair =
							(short)((((int)(attrs & Attributes.A_COLOR)) >> 8));
						Color fg, bg;
						PairContent(pair, out fg, out bg);
						if(value != (Color)fg)
						{
							SetColorPair(value, bg);
						}
					}
				}
			}

	/// <summary>
	/// <para>Get or set the current background color.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The color value.</para>
	/// </value>
	public Color Background
			{
				get
				{
					if(Curses.Native.has_colors() != 0)
					{
						short pair =
							(short)((((int)(attrs & Attributes.A_COLOR)) >> 8));
						Color fg, bg;
						PairContent(pair, out fg, out bg);
						return bg;
					}
					else
					{
						return Color.Default;
					}
				}
				set
				{
					if(Curses.Native.has_colors() != 0)
					{
						short pair =
							(short)((((int)(attrs & Attributes.A_COLOR)) >> 8));
						Color fg, bg;
						PairContent(pair, out fg, out bg);
						if(value != bg)
						{
							SetColorPair(fg, value);
						}
					}
				}
			}

	/// <summary>
	/// <para>Return to a normal attribute mode.</para>
	/// </summary>
	public void Normal()
			{
				if(attrs != Attributes.A_NORMAL && handle != (IntPtr)0)
				{
					Curses.Native.wattrset(handle, (int)0);
				}
				attrs = Attributes.A_NORMAL;
			}

	/// <summary>
	/// <para>Determine if the underlying curses implementation
	/// supports colors or not.</para>
	/// </summary>
	///
	/// <value>
	/// <para><see langword="true"/> if colors are supported.</para>
	/// </value>
	public static bool HasColors
			{
				get
				{
					if(Setup.Initialize())
					{
						return (Curses.Native.has_colors() != 0);
					}
					else
					{
						return false;
					}
				}
			}

	/// <summary>
	/// <para>Clear the entire contents of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is normally followed by a call to <c>Refresh</c>.</para>
	/// </remarks>
	public virtual void Clear()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wclear(handle);
				}
			}

	/// <summary>
	/// <para>Erase the entire contents of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This does not mark the contents of the window for update,
	/// as <c>Clear</c> does.</para>
	/// </remarks>
	public virtual void Erase()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.werase(handle);
				}
			}

	/// <summary>
	/// <para>Clear the window from the current cursor position to
	/// the bottom of the window.</para>
	/// </summary>
	public virtual void ClearToBottom()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wclrtobot(handle);
				}
			}

	/// <summary>
	/// <para>Clear the window from the current cursor position to
	/// the end of the current line.</para>
	/// </summary>
	public virtual void ClearToEOL()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wclrtoeol(handle);
				}
			}

	/// <summary>
	/// <para>Draw a horizontal line across the window.</para>
	/// </summary>
	///
	/// <param name="x1">
	/// <para>The X co-ordinate for the first character to be drawn.</para>
	/// </param>
	///
	/// <param name="x2">
	/// <para>The X co-ordinate for the last character to be drawn.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate for the line.</para>
	/// </param>
	public virtual void HLine(int x1, int x2, int y)
			{
				if(handle == (IntPtr)0)
				{
					return;
				}
				Curses.Native.wmove(handle, y, x1);
				Curses.Native.whline(handle, '-', (x2 - x1) + 1);
			}

	/// <summary>
	/// <para>Draw a vertical line down the window.</para>
	/// </summary>
	///
	/// <param name="x">
	/// <para>The X co-ordinate for the line.</para>
	/// </param>
	///
	/// <param name="y1">
	/// <para>The Y co-ordinate for the first character to be drawn.</para>
	/// </param>
	///
	/// <param name="y2">
	/// <para>The Y co-ordinate for the last character to be drawn.</para>
	/// </param>
	public virtual void VLine(int x, int y1, int y2)
			{
				if(handle == (IntPtr)0)
				{
					return;
				}
				Curses.Native.wmove(handle, y1, x);
				Curses.Native.wvline(handle, '|', (y2 - y1) + 1);
			}

	/// <summary>
	/// <para>Insert a character at the current position.</para>
	/// </summary>
	///
	/// <param name="ch">
	/// <para>The character to be inserted.</para>
	/// </param>
	public virtual void InsertChar(char ch)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.winsch(handle, ((uint)ch) | ((uint)attrs));
				}
			}

	/// <summary>
	/// <para>Delete a character at the current position.</para>
	/// </summary>
	public virtual void DeleteChar()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wdelch(handle);
				}
			}

} // struct DrawingContext

} // namespace Curses
