/*
 * Frog.cs - Object that represents a "frog" game piece.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Frogger
{

using System;
using Curses;

public class Frog : GamePiece
{
	// Size of the frog.
	public const int FrogWidth  = 5;
	public const int FrogHeight = 4;

	// Kind of frog to draw.
	private int kind;
	private const int FrogKind_Normal = 0;
	private const int FrogKind_OnLog  = 1;

	// Current log line where the frog is sitting.
	private int logLine;

	// Flag that is set when the frog is dead.
	private bool dead;

	// Flag that is set when the frog is safe.
	private bool safe;

	// Timeout object that is tracking how long the frog has left to live.
	private FrogTimeout timeout;

	// Create a new frog game piece at an initial position.
	public Frog(MainBoard board, DrawingContext dc, int x, int y)
			: base(board, dc, x, y, FrogWidth, FrogHeight)
			{
				kind = FrogKind_Normal;
				logLine = -1;
				dead = false;
				safe = false;
				timeout = new FrogTimeout(this, 10);
			}

	// Draw this game piece.
	public override void Draw()
			{
				DrawingContext dc = this.dc;
				if(kind == FrogKind_Normal)
				{
					dc.Foreground = Color.Green;
					dc.Move(x, y);
					dc.Add("  ^  ");
					dc.Move(x, y + 1);
					dc.Add("\\/ \\/");
					dc.Move(x, y + 2);
					dc.Add(" \\ / ");
					dc.Move(x, y + 3);
					dc.Add("_/ \\_");
				}
				else
				{
					dc.Foreground = Color.Yellow;
					dc.Move(x, y);
					dc.Add("--");
					dc.Foreground = Color.Green;
					dc.Add('^');
					dc.Foreground = Color.Yellow;
					dc.Add("--");
					dc.Foreground = Color.Green;
					dc.Move(x, y + 1);
					dc.Add("\\/ \\/");
					dc.Move(x, y + 2);
					dc.Add(" \\ / ");
					dc.Move(x, y + 3);
					dc.Add("_/");
					dc.Foreground = Color.Yellow;
					dc.Add('-');
					dc.Foreground = Color.Green;
					dc.Add("\\_");
				}
			}

	// Erase this game piece.
	public void Erase()
			{
				DrawingContext dc = this.dc;
				if(kind == FrogKind_Normal)
				{
					dc.Move(x, y);
					dc.Add("     ");
					dc.Move(x, y + 1);
					dc.Add("     ");
					dc.Move(x, y + 2);
					dc.Add("     ");
					dc.Move(x, y + 3);
					dc.Add("     ");
				}
				else
				{
					dc.Foreground = Color.Yellow;
					dc.Move(x, y);
					dc.Add("-----");
					dc.Move(x, y + 1);
					dc.Add("     ");
					dc.Move(x, y + 2);
					dc.Add("     ");
					dc.Move(x, y + 3);
					dc.Add("-----");
				}
			}

	// Draw a collision between the frog and some other object.
	public void DrawCollision()
			{
				DrawingContext dc = this.dc;
				dc.Foreground = Color.Red;
				dc.Move(x, y);
				dc.Add("  *  ");
				dc.Move(x, y + 1);
				dc.Add("** **");
				dc.Move(x, y + 2);
				dc.Add(" * * ");
				dc.Move(x, y + 3);
				dc.Add("** **");
			}

	// Move the frog as a result of a log that it is sitting on
	// moving underneath it.
	public void MoveWithLog(int delta)
			{
				x += delta;
				if(x <= 0 || (x + FrogWidth) >= dc.Window.Width - 1)
				{
					DrawCollision();
					dead = true;
				}
			}

	// Move the frog to a new log line.
	private void MoveToLogLine(int newLine)
			{
				LogLine line;

				// Erase the frog from its current position.
				Erase();

				// Remove the frog from the log it is sitting on.
				if(logLine >= 0)
				{
					line = board.GetLogLine(logLine);
					line.SittingFrog = null;
				}

				// Move the frog to the new log.
				logLine = newLine;
				line = board.GetLogLine(logLine);
				line.SittingFrog = this;
				y = line.Y;
				kind = FrogKind_OnLog;
				Draw();

				// Determine if the frog landed in the water.
				if(!line.IntersectsWith(this))
				{
					DrawCollision();
					dead = true;
				}
			}
	
	// Move the frog to the top bank.
	private void MoveToTopBank()
			{
				LogLine line;

				// Erase the frog from its current position.
				Erase();

				// Remove the frog from the log it is sitting on.
				if(logLine >= 0)
				{
					line = board.GetLogLine(logLine);
					line.SittingFrog = null;
				}

				// Move the frog to the top bank.
				logLine = -2;
				kind = FrogKind_Normal;
				y = 0;
				Draw();
				dc.Refresh();

				// Stop everything and start a new level.
				safe = true;
			}
	
	// Move the frog to the bottom bank.
	private void MoveToBottomBank()
			{
				LogLine line;

				// Erase the frog from its current position.
				Erase();

				// Remove the frog from the log it is sitting on.
				if(logLine >= 0)
				{
					line = board.GetLogLine(logLine);
					line.SittingFrog = null;
				}

				// Move the frog to the top bank.
				logLine = -1;
				kind = FrogKind_Normal;
				y = dc.Window.Height - FrogHeight;
				Draw();
			}

	// Move the frog sideways by a delta amount.
	private void MoveSideWays(int delta)
			{
				Erase();
				x += delta;
				if(x < 0)
				{
					x = 0;
				}
				else if((x + FrogWidth) > dc.Window.Width)
				{
					x = dc.Window.Width - FrogWidth;
				}
				Draw();

				// Did the frog fall off the end of the log,
				// or go too far on the riverbank?
				if(logLine >= 0)
				{
					if(!board.GetLogLine(logLine).IntersectsWith(this))
					{
						DrawCollision();
						dead = true;
					}
				}
				else if(logLine == -1)
				{
					if(x <= 0 || (x + FrogWidth) >= dc.Window.Width)
					{
						DrawCollision();
						dead = true;
					}
				}
			}

	public enum PlayResult
	{
		Dead,
		NextLevel,
		Quit
	}

	// Play the game.
	public PlayResult Play()
			{
				int ch;
				dead = false;
				safe = false;
				while(!dead && !safe)
				{
					ch = Input.GetNextChar();
					if(ch == 0x1B || ch == 'q' || ch == 'Q')
					{
						break;
					}
					else if(ch == KeyCode.KEY_UP || ch == 'k' || ch == 'K')
					{
						Misc.Beep();
						if(logLine == -1)
						{
							MoveToLogLine(board.NumLogLines - 1);
						}
						else if(logLine > 0)
						{
							MoveToLogLine(logLine - 1);
						}
						else if(logLine != -2)
						{
							MoveToTopBank();
						}
					}
					else if(ch == KeyCode.KEY_DOWN || ch == 'j' || ch == 'J')
					{
						Misc.Beep();
						if(logLine != -1)
						{
							if(logLine == (board.NumLogLines - 1))
							{
								MoveToBottomBank();
							}
							else if(logLine != -2)
							{
								MoveToLogLine(logLine + 1);
							}
							else
							{
								MoveToLogLine(0);
							}
						}
					}
					else if(ch == KeyCode.KEY_LEFT || ch == 'h' || ch == 'H')
					{
						Misc.Beep();
						MoveSideWays(-(FrogWidth / 2));
					}
					else if(ch == KeyCode.KEY_RIGHT || ch == 'l' || ch == 'L')
					{
						Misc.Beep();
						MoveSideWays(FrogWidth / 2);
					}
					else if(ch == 's' || ch == 'S')
					{
						Misc.SoundEnabled = !Misc.SoundEnabled;
					}
				}
				if(dead)
				{
					return PlayResult.Dead;
				}
				else if(safe)
				{
					return PlayResult.NextLevel;
				}
				else
				{
					return PlayResult.Quit;
				}
			}

	// Stop all asynchronous events associated with this game piece.
	public override void StopEvents()
			{
				timeout.Remove();
			}

	// Update the time that the frog has left to cross the river.
	public void UpdateTime()
			{
				int timeToLive = timeout.TimeToLive;
				dc.Move(dc.Window.Width - 15, 1);
				dc.Foreground = Color.White;
				dc.Add("Time Left: ");
				if(timeToLive < 10)
				{
					dc.Add(' ');
				}
				else
				{
					dc.Add((char)('0' + (timeToLive / 10)));
				}
				dc.Add((char)('0' + (timeToLive % 10)));
			}

	// Timeout class that manages the frog's total time to live.
	private sealed class FrogTimeout : Timeout
	{
		// Internal state.
		private Frog frog;
		private int timeToLive;

		// Constructor.
		public FrogTimeout(Frog frog, int timeToLive)
			: base(new TimeSpan(TimeSpan.TicksPerSecond),
			       new TimeSpan(TimeSpan.TicksPerSecond))
			{
				this.frog = frog;
				this.timeToLive = timeToLive;
			}

		// Handle a timeout.
		protected override void OnFired(Object sender, EventArgs args)
			{
				if(timeToLive > 0)
				{
					--timeToLive;
					frog.UpdateTime();
					if(timeToLive <= 0)
					{
						Remove();
						frog.DrawCollision();
						frog.dead = true;
					}
				}
			}

		// Get the number of seconds that the frog has left to live.
		public int TimeToLive
			{
				get
				{
					return timeToLive;
				}
			}

	}

} // class Frog

} // namespace Frogger
