// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"
#include "ioports.h"

int VPBThread::offset = 0;
int VPBThread::cards = 0;
int VPBThread::count = 0;

static ostream &vpblog(slog_level_t level, unsigned port, int id)
{
	return slog(level) << "vpb(" << hex << port << dec << "," << id << "): ";
}

VPBThread::VPBThread() :
Semaphore(0), Thread((Semaphore *)this, keythreads.priService()), Mutex()
{
	int j;
	struct vpbreg *preg;

	id = ++count;
	active = false;
	reg = (void *)(new char[sizeof(struct vpbreg)]);
	preg = (struct vpbreg *)reg;
	control = 0;
	pipmask = 0;
	port = vpbivr.VPBConfig::getFirst() + offset;
	offset += 16;
	setControl(0);

	preg->numch = vpbivr.getChannels();
	preg->szmess = vpbivr.getMsgSize();
	for(j = 0; j < preg->numch; ++j)
	{
		preg->szrxdf[j] = vpbivr.getRxSize();
		preg->sztxdf[j] = vpbivr.getTxSize();
	}
	preg->lsf = vpbivr.getFrameSize();
	szRelayBuf = vpbivr.getRelaySize();

	dspReset();
	if(!dspTest(1000))
	{
		vpblog(SLOG_ERROR, port, id) << "card failure" << endl;	
		return;
	}
	++cards;
	vpblog(SLOG_INFO, port, id) << "initializing firmware" << endl;
}

VPBThread::~VPBThread()
{
	if(!active)
		Post();
	Terminate();
	if(reg)
		delete (char *)reg;
	offset -= 16;
}

void VPBThread::setControl(unsigned char bits)
{
	control |= bits;
	ioset(port + VPB_CONTROL, control);
}

void VPBThread::clrControl(unsigned char bits)
{
	control &= ~bits;
	ioset(port + VPB_CONTROL, control);
}

void VPBThread::dspWrite(unsigned short addr, unsigned short *data, unsigned len)
{
	setControl(VPB_ENABLE_FLAG | VPB_UPDOWN_FLAG);
	ioset(port + VPB_ADDRLO, pipmask + (addr & 0xff));
	ioset(port + VPB_ADDRHI, (addr>>8) & 0xff);
	iowrite(port + VPB_DATA, data, len);
}

void VPBThread::dspRead(unsigned short addr, unsigned short *data, unsigned len)
{
	setControl(VPB_ENABLE_FLAG | VPB_UPDOWN_FLAG);
	ioset(port + VPB_ADDRLO, pipmask + (addr & 0xff));
	ioset(port + VPB_ADDRHI, (addr >> 8) & 0xff);
	ioread(port + VPB_DATA, data, len);
}

void VPBThread::dspReset(void)
{
	setControl(VPB_RESET_FLAG);
}

void VPBThread::dspEnable(void)
{
	clrControl(VPB_RESET_FLAG);
}

bool VPBThread::dspTest(unsigned length)
{
	int i;
	unsigned short *buffer = new unsigned short[length];

	for(i = 0; i < length; ++i)
		buffer[i] = i;

	dspWrite(0, buffer, length);
	memset(buffer, 0, sizeof(unsigned short) * length);
	dspRead(0, buffer, length);

	for(i = 0; i < length; ++i)
		if(buffer[i] != i)
			return false;

	return true;
}
	

struct vpbreg *VPBThread::getRegistry(void)
{
	return (struct vpbreg *)reg;
}

void	VPBThread::Run(void)
{
	vpblog(SLOG_INFO, port, id) << "running..." << endl;

	if(!active)
		Sleep(10000);

	dspEnable();
	 
	for(;;)
		Sleep(10000);
}

