// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "server.h"
#include <ctype.h>

Scheduler::Scheduler() :
Thread(NULL, keythreads.priScheduler()), ThreadLock()
{
	char path[65];

	interval = 0;

	if(canModify(keypaths.getRunfiles()))
		sprintf(path, "%s/bayonne.sched", keypaths.getRunfiles());
	else
		sprintf(path, "%s/.bayonne.sched", getenv("HOME"));

	if(isFile(path))
		rtmp = open(path, O_WRONLY | O_TRUNC);
	else
		rtmp = creat(path, 0664);
	sched.close();
}

Scheduler::~Scheduler(void)
{
	Terminate();
	close(rtmp);
	sched.close();
}

void Scheduler::Initial(void)
{
	if(sched.is_open())
		sched.close();

	sched.open("/etc/bayonne.sched");
	if(!sched)
		slog(SLOG_WARNING) << "/etc/bayonne.sched: cannot access" << endl;
	else
	{
		slog(SLOG_NOTICE) << "scheduler started" << endl;
		interval = Load();
		Update();
	}
}

void Scheduler::Stop(void)
{
	slog(SLOG_DEBUG) << "scheduler stopped" << endl;
	Terminate();
	sched.close();
	close(rtmp);
}

void Scheduler::Run(void)
{
	Sync *sync;
	time_t now;

	Update();
	for(;;)
	{
		if(statmon)
			statmon->Refresh(60);
		else
			Sleep(60000);

		time(&now);
		sync = Sync::first;
		while(sync)
		{
			if((sync->runtime + 60 * sync->getInterval()) < now)
			{
				time(&sync->runtime);
				if(sync->isScheduled())
				{
					slog(SLOG_INFO) << "scheduler: " << sync->getName() << " updated" << endl;
					sync->Schedule();
				}
			}
			sync = sync->next;
		}
		
		if(!sched)
			Initial();

		if(!sched.is_open() || interval > 0)
			--interval;
		else if(interval < 1)
		{
			interval = Load();
			Update();
		}
	}
}

void Scheduler::Update(void)
{
	schedtmp tmp;

	TrunkGroup *grp = TrunkGroup::first;
	if(rtmp < 0)
		return;

	lseek(rtmp, 0l, SEEK_SET);
	while(grp)
	{
		memset(&tmp, 0, sizeof(tmp));
		strncpy(tmp.grp, grp->getName(), sizeof(tmp.grp));
		grp->getSchedule(tmp.scr);
		write(rtmp, (char *)&tmp, sizeof(tmp));
		grp = grp->next;
	} 
}

int Scheduler::Load(void)
{
	static char *days[7] = 
		{"su", "mo", "tu", "we", "th", "fr", "sa"};

	TrunkGroup *group;
	time_t now;
	struct tm *dt;
	char lbuf[256];
	char *name, *day, *start, *cp;
	char *s, *p;
	int hr, min;
	int lhr = 24, lmin = 0;
	int id, t1, t2, td;
	char *sp;
	struct tm tbuf;

	time(&now);
	dt = localtime_r(&now, &tbuf);

	if(!sched.is_open())
	{
		WriteLock();
		group = TrunkGroup::first;
		while(group)
		{
			group->setSchedule(NULL);
			group = group->next;
		}
		Unlock();
		return keythreads.getInterval();
	}
		
	sched.seekg(0);
	sched.clear();

	WriteLock();
	for(;;)
	{
		sched.getline(lbuf, 255);

		if(sched.eof())  
			break;
	
		name = lbuf;
		while(*name && isspace(*name))
			++name;

		if(!*name || *name == '#' || *name == ';')
			continue;

		name = strtok_r(name, " \t\n", &sp);
		day = strtok_r(NULL, " \t\n", &sp);
		start = strtok_r(NULL, " \t\n", &sp);
		cp = strtok_r(NULL, " \t\n", &sp);
		if(!cp || !day || ! start)
		{
			slog(SLOG_ERROR) << "error in schedule" << endl;
			continue;
		}

		if(stricmp(day, days[dt->tm_wday]) && stricmp(day, "*"))
			continue;		
		
		if (!strcmp(start,"*" )) 
			hr = min = 0;
		else 
		{
			hr = atoi(start);
			min = atoi(strchr(start, ':') + 1);	
		}			
	
		if(hr < lhr || (hr == lhr && min < lmin))
			if(hr > dt->tm_hour || (hr == dt->tm_hour && min >= dt->tm_min))
			{
				lhr = hr;
				lmin = min;
			}

		if(hr > dt->tm_hour || (hr == dt->tm_hour && min > dt->tm_min))
			continue;
		
		if(!strcmp(name, "*"))
		{
			if ( !( group = TrunkGroup::first ) ) 
				slog( SLOG_WARNING ) << "scheduler could not find any group" << endl;
			while(group)
			{
				group->setSchedule(cp);
				group = group->next;
			}
			continue;
		}

		p = strtok_r(name, ",;", &sp);	
		while(p)
		{
			group = keyserver.getGroup(p);
			if(!group) 
			{ 
				slog( SLOG_WARNING ) << "unknown group '" << p << "' refereced in schedule" << endl;
				continue;
			} 
			else
				group->setSchedule(cp);
			p = strtok_r(NULL, ",;", &sp);
		}
	}
	Unlock();

	sched.clear(); // Probably we've hit an EOF - so we must clear fail()

	td = (lhr * 60) + lmin - (dt->tm_hour * 60) - dt->tm_min;
	if(td > keythreads.getInterval())
		td = keythreads.getInterval();
	if(td < 1)
		td = 1;
	return td;
}
					
