// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <bayonne.h>
#include <ctype.h>

class EnglishTranslator : public Translator
{
private:
	char *getName(void)
		{return "english";};
public:
	EnglishTranslator();
	char *Speak(Trunk *trunk);
} english;

EnglishTranslator::EnglishTranslator() :
Translator("/bayonne/english", keylocal.getLast("english"))
{
	slog(SLOG_DEBUG) << "TTS: loading english phrasebooks..." << endl;
}

static void spell(strstream *s, char *text);

static void order(strstream *s, int num)
{
	char *low[] = { "th",
		"1st", "2nd", "3rd", "4th", "5th",
		"6th", "7th", "8th", "9th", "10th",
		"11th", "12th", "13th", "14th", "15th",
		"16th", "17th", "18th", "19th"};

	if(num > 100)
	{
		if(num % 100)
			*s << ','  << num / 100;
		else
			*s << ',' << num / 100 << "th";
		num %= 100;
	}
	if(num > 19)
	{
		if(num % 10)
			*s << ',' << num / 10;
		else
			*s << ',' << num / 10 << "th";
		num %= 10;
	}
	if(num)
		*s << ',' << low[num];
}

static void lownumber(strstream *s, int num)
{
	if(num >= 100)
	{
		*s << ',' << num / 100 << ",hundred";
		num %= 100;
		if(!num)
			return;
	}
	if(num < 20)
	{
		*s << ',' << num;
		return;
	}
	*s << ',' << (num / 10) * 10;
	if(num % 10)
		*s << "," << num % 10;
}

static bool number(strstream *s, char *text)
{
	unsigned long num;
	bool zero = false;

	if(!text)
		return true;

	if(!*text)
		return true;

	if(*text == '-')
	{
		*s << ",negative";
		++text; 
	}
	num = atol(text);
	if(num > 999999999)
	{
		zero = true;
		lownumber(s, num / 1000000000);
		*s << ",billion";
		num %= 1000000000;
	}
	if(num > 999999)
	{
		zero = true;
		lownumber(s, num / 1000000);
		*s << ",million";
		num %= 1000000;
	}
	if(num > 999)
	{
		zero = true;
		lownumber(s, num / 1000);
		*s << ",thousand";
		num %= 1000;
	}
	if(num || zero)
		lownumber(s, num);

	if(num)
		zero = false;

	text = strchr(text, '.');
	if(!text)
		return zero;
	*s << ",point";
	spell(s, ++text);
	return false;
}

static void spell(strstream *s, char *text)
{
	char ch;
	if(!text)
		return;


	while((ch = *(text++)))
	{
		if(ch >= 'A' && ch <= 'Z')
			ch += 32;

		if(ch >= '0' && ch <= '9')
			*s << ',' << ch;
		else if(ch >= 'a' && ch <= 'z')
			*s << ',' << ch;
		else if(ch == '.')
			*s << ",point";
	}
}

static void saydate(strstream *s, char *text)
{
	static char *months[] = {
		"january", "february", "march", "april",
		"may", "june", "july", "august",
		"september", "october", "november", "december"};

	time_t now;
	struct tm *dt;
	time(&now);
	dt = localtime(&now);
	int month, day, year;
	char buf[5];

	if(text[2] == '/')
		sscanf(text, "%02d/%02d/%04d", &month, &day, &year);
	else if(text[2] == '.')
		sscanf(text, "%02d.%02d.%04d", &day, &month, &year);
	else
		sscanf(text, "%04d%02d%02d", &year, &month, &day);

	--month;
	if(dt->tm_year < 1000)
		dt->tm_year += 1900;
	if(month == dt->tm_mon && day == dt->tm_mday && year == dt->tm_year)
	{
		*s << ",today";
		return;
	}	

	*s << ',' << months[month];
	order(s, day);
	if(dt->tm_year == year)
		return;

	if(year != 2000)
	{
		sprintf(buf, "%d", year / 100);
		number(s, buf);
		if(year % 100)
		{
			if(year % 100 < 10)
				*s << ",o";
			sprintf(buf, "%d", year % 100);
			number(s, buf);
		}
		else
			*s << ",hundred";
	}
	else
	{
		*s << ",2,thousand";
		if(year % 10)
			*s << ',' << year % 10;
	}
}

static char *getchange(char *text, int dec, char *ch)
{
	int len;
	text = strchr(text, '.');
	if(!text)
		return NULL;

	++text;
	memset(ch, '0', dec);
	ch[dec] = 0;
	len = strlen(text);
	if(len > dec)
		len = dec;
	strncpy(ch, text, len);
	return ch;
}
	
static void sayvalue(strstream *s, char *cu, char *ch, int dec, char *text)
{
	char chbuf[dec + 1];
	int currency = atoi(text);
	char cbuf[11];
	char *change = getchange(text, dec, chbuf);
	char *and = keylocal.getLast("currencyand");

	if(!currency && !change)
	{
		*s << ",0," << cu << 's';
	}

	if(currency)
	{
		sprintf(cbuf, "%d", currency);
		number(s, cbuf);
		if(currency > 1)
			*s << ',' << cu << 's';
		else
			*s << ',' << cu;

		if(change && and)
			*s << ',' << and;
	}
	if(change)
	{
		number(s, change);
		if(atoi(change) > 1)
			*s << ',' << ch << 's';
		else
			*s << ',' << ch;
	}
}			

static bool sayprimary(strstream *s, char *text)
{
	char *cu = keylocal.getLast("primarycurrency");
	char *ch = keylocal.getLast("primarychange");
	char *d = keylocal.getLast("primarydecimal");
	if(!d)
		d = "2";

	if(!cu)
		return false;

	sayvalue(s, cu, ch, atoi(d), text);
	return true;
}

static bool saylocal(strstream *s, double v, bool or)
{
	char tbuf[33];
	char *cu = keylocal.getLast("localcurrency");
	char *ch = keylocal.getLast("localchange");
	char *d = keylocal.getLast("localdecimal");
	char *cvt = keylocal.getLast("convertcurrency");
	char *os = keylocal.getLast("currencyor");

	if(!d)
		d = "2";

	if(!cu)
		return true;

	if(cvt)
	{
		v *= atof(cvt);
		sprintf(tbuf, "%d", v);
	}
	else
		sprintf(tbuf, "%d", v);	
	if(!v)
		return false;
	if(or)
		*s << ',' << *os;
	sayvalue(s, cu, ch, atoi(d), tbuf);
	return true;
}

static void saycurrency(strstream *s, char *text)
{
	saylocal(s, atof(text), sayprimary(s, text));
}
			
static void saytime(strstream *s, char *text)
{
	int hour, minute;
	bool pm = false;
	char buf[3];

	if(text[2] == '%')
		sscanf(text, "%02d:%02d", &hour, &minute);
	else
		sscanf(text, "%02d%02d", &hour, &minute);

	if(hour > 11)
	{
		pm = true;
		hour -= 12;
	}
	if(!hour)
		hour = 12;

	sprintf(buf, "%d", hour);
	number(s, buf);
	if(minute)
	{
		sprintf(buf, "%d", minute);
		number(s, buf);
	}
	if(pm)
		*s << ",p,m";
	else
		*s << ",a,m";
}

char *EnglishTranslator::Speak(Trunk *trunk)
{
	char *pbuf = getPlayBuffer(trunk);
	strstream str(pbuf, 254);
	char rules[256];
	char *rule = NULL;
	char *phrase;
	bool skip = false;
	long lastnum = 0;
	char *text;

	while(NULL != (rule = trunk->getOption(NULL)))
	{
		if(*rule != '&')
		{
			str << ',' << rule;
			continue;
		}
		phrase = getLast(rule + 1);
		if(!phrase)
			phrase = rule;

		strncpy(rules, phrase, 254);
		rules[255] = 0;
		rule = strtok(rules, " ;,\t\n");
		while(rule)
		{
			if(!stricmp(rule, "&exit"))
			{
				if(skip)
					return NULL;
			}
			else if(!stricmp(rule, "&ignore"))
			{
				trunk->getOption(NULL);
			}
			else if(!stricmp(rule, "&use"))
			{
				text = trunk->getOption(NULL);
				if(text)
					str << ',' << text;
			}
			else if(!stricmp(rule, "&skip"))
			{
				if(skip)
					trunk->getOption(NULL);
			}
			else if(!stricmp(rule, "&single") || !stricmp(rule, "&singular"))
			{
				text = trunk->getOption(NULL);
				if(lastnum == 1 && !skip && text)
					str << ',' << text;
			}
			else if(!stricmp(rule, "&plural"))
			{
				text = trunk->getOption(NULL);
				if(lastnum != 1 && !skip && text)
					str << ',' << text;
			}
			else if(!stricmp(rule, "&spell"))
			{
				skip = false;
				spell(&str, trunk->getOption(""));
			}
			else if(!stricmp(rule, "&number"))
			{
				text = trunk->getOption(NULL);
				if(text)
				{
					lastnum = atol(text);	
					skip = number(&str, text);
				}
			}
			else if(!stricmp(rule, "&order"))
			{
				skip = false;
				text = trunk->getOption(NULL);
				if(text)
					order(&str, atoi(text));
			}
			else if(!stricmp(rule, "&time"))
			{
				skip = false;
				text = trunk->getOption(NULL);
				if(text)
					saytime(&str, text);
			}
			else if(!stricmp(rule, "&date"))
			{
				skip = false;
				text = trunk->getOption(NULL);
				if(text)
					saydate(&str, text);
			}
			else if(!stricmp(rule, "&unit"))
			{
				skip = false;
				text = trunk->getOption(NULL);
				if(text)
				{
					lastnum = atol(text);
					if(number(&str, text))
						str << ",0";
				}
			}
			else if(!stricmp(rule, "&zero"))
			{
				skip = false;
				text = trunk->getOption(NULL);
				if(!lastnum && text)
					str << ',' << text;
			}
			else if(!stricmp(rule, "&currency"))
			{
				skip = false;
				lastnum = 1;
				text = trunk->getOption(NULL);
				if(text)
				{
					if(atof(text))
						saycurrency(&str, text);
					else
						lastnum = 0;
				}
			}
			else if(!stricmp(rule, "&primary"))
			{
				skip = false;
				lastnum = 1;
				text = trunk->getOption(NULL);
				if(text)
				{
					if(atof(text))
						sayprimary(&str, text);
					else
						lastnum = 0;
				}
			}
			else if(!stricmp(rule, "&local"))
			{
				skip = false;
				lastnum = 1;
				text = trunk->getOption(NULL);
				if(text)
				{
					if(!saylocal(&str, atof(text), false))
						lastnum = 0;
				}
			}
			else 
			{
				skip = false;
				str << ',' << rule;
			}
			rule = strtok(NULL, " ;,\t\n");
		}	
	}

	str << ends;
	pbuf[255] = 0;
	while(*pbuf)
	{
		*pbuf = tolower(*pbuf);
		++pbuf;
	}
	return NULL;
}
	

	
	

