// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <sys/ioctl.h>
#include "driver.h"

char TrunkDriver::status[240];

TrunkDriver::TrunkDriver(int fd, int ts) :
Trunk(ts), TimerPort()
{
	TrunkEvent event;
	char *cp;
	struct phone_capability *cap;

	next = prev = NULL;
	service = NULL;
	handler = NULL;
	lastring = 0;

	dev = fd;

#ifdef	IXJ_PHONEDEV
	driver = ixj_driver;
#else
	driver = generic_driver;
#endif

	cap_count = ioctl(dev, PHONE_CAPABILITIES);
	cap_list = new struct phone_capability[cap_count];
	ioctl(dev, PHONE_CAPABILITIES_LIST, cap_list);

	cap = getCapability(vendor);
	if(cap)
		switch(cap->cap)
		{
#ifdef	IXJ_PHONEDEV
		case PHONE_VENDOR_IXJ:
			driver = ixj_driver;
			break;
#endif
		}

	ioctl(dev, PHONE_WINK_DURATION, ivr.getWinkDuration());

	handler = &TrunkDriver::hangupHandler;
	event.id = TRUNK_ENTER_STATE;

	switch(driver)
	{
#ifdef	IXJ_PHONEDEV
	case ixj_driver:
		status[id] = 't';
		ioctl(dev, IXJCTL_PORT, PORT_PSTN);

		cpatone[CPA_DIALTONE].filter = 0;
		cpatone[CPA_DIALTONE].freq = f440_480;
		cpatone[CPA_DIALTONE].enable = 0;
		ioctl(dev, IXJCTL_SET_FILTER, &cpatone[0]);
		
		slog(SLOG_INFO) << "phone" << id << ": Quicknet driver started" << endl;
		if(!ioctl(dev, IXJCTL_PSTN_LINETEST))
		{
			slog(SLOG_WARNING) << "phone" << id << ": linetest failure" << endl;
			handler = &TrunkDriver::busyHandler;
		}
		break;
#endif
	default:
		slog(SLOG_INFO) << "phone" << id << ": generic driver started" << endl;
	}
	service = ivr.getService();		
	service->Attach(this);
	
	(this->*handler)(&event);
}

TrunkDriver::~TrunkDriver()
{
	handler = NULL;
	if(service)
		service->Detach(this);

	endTimer();
	
	setHookState(false);
	switch(driver)
	{
#ifdef	IXJ_PHONEDEV
	case ixj_driver:
		ioctl(dev, IXJCTL_PORT, PORT_POTS);
		break;
#endif
	}
	close(dev);
	slog(SLOG_INFO) << "phone" << id << ": device stopped" << endl;
}

void TrunkDriver::setCPADetect(cpatone_t tone, bool mode)
{
	switch(driver)
	{
#ifdef	IXJ_PHONEDEV
	case ixj_driver:
		if(mode && cpatone[tone].enable)
			break;
		if(!mode && !cpatone[tone].enable)
			break;
		cpatone[tone].enable = mode;
		ioctl(dev, IXJCTL_SET_FILTER, &cpatone[tone]);
#endif
	}
}
void TrunkDriver::getName(char *buffer)
{
	sprintf(buffer, "phone%d", id);
}

void TrunkDriver::Exit(void)
{
	handler = &TrunkDriver::hangupHandler;
}

bool TrunkDriver::postEvent(TrunkEvent *event)
{
	bool rtn = true;
	trunkhandler_t prior;

	EnterMutex();
	switch(event->id)
	{
	case TRUNK_TIMER_EXPIRED:
		if(!getTimer())
			rtn = false;
		break;
	case TRUNK_DTMF_KEYUP:
		if(!flags.dtmf)
			rtn = false;
		break;
	}	
	if(!rtn)
	{
		LeaveMutex();
		return false;
	}

	if(!handler)
	{
		slog(SLOG_WARNING) << "phone" << id;
		slog() << ": no handler active; event=" << event->id << endl;
		LeaveMutex();
		return false;
	}

retry:
	debug->DebugEvent(this, event);
	prior = handler;
	rtn = (this->*handler)(event);
	if(rtn)
	{
		if(handler != prior)
		{
			event->id = TRUNK_ENTER_STATE;
			goto retry;
		}
		LeaveMutex();
		return true;
	}

	// default handler

	rtn = true;
	switch(event->id)
	{
	case TRUNK_RINGING_ON:
		++rings;
		break;
	case TRUNK_ENTER_STATE:
		if(flags.offhook)
			setDTMFDetect();
		else
			setDTMFDetect(false);
		endTimer();
		break;
	case TRUNK_CPA_DIALTONE:
	case TRUNK_LINE_WINK:
		TrunkSignal(TRUNK_SIGNAL_HANGUP);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_TIMER_EXPIRED:
		TrunkSignal(TRUNK_SIGNAL_TIMEOUT);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_DTMF_KEYUP:
		if(digits < 32)
			dtmf.bin.data[digits++] = digit[event->parm.dtmf.digit];
		dtmf.bin.data[digits] = 0;
		if(TrunkSignal((trunksignal_t)(event->parm.dtmf.digit + TRUNK_SIGNAL_0)))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_EXIT_SHELL:
		tgi.pid = 0;
		break;
	case TRUNK_STOP_STATE:
		endTimer();
		handler = &TrunkDriver::stepHandler;
		break;
	case TRUNK_EXIT_STATE:
		break;
	case TRUNK_MAKE_BUSY:
		handler = &TrunkDriver::busyHandler;
		break;
	case TRUNK_MAKE_IDLE:
		handler = &TrunkDriver::idleHandler;
		break;
	default:
		rtn = false;
	}
	if(handler != prior)
	{
		event->id = TRUNK_ENTER_STATE;
		goto retry;
	}
	LeaveMutex();
	return rtn;
}

void TrunkDriver::getEvents(void)
{
	union telephony_exception exc;
	char digit;
	TrunkEvent event;
	cid_t cid;
	char buf[81];
	time_t now;
	int ringid;
	char *cp;

	exc.bytes = ioctl(dev, PHONE_EXCEPTION);
	if(exc.bits.pstn_ring && rings < 2)
	{
		time(&now);
		if(now - lastring < 4)
		{
			setSymbol("ringid", 4);
			cp = getSymbol("ringid");
			if(cp)
				ringid = atoi(cp);
			else
				ringid = 0;
			sprintf(buf, "%d", ++ringid);
			setSymbol("ringid", buf);
			exc.bits.pstn_ring = 0;
//			exc.bits.caller_id = 1;
		}
	}
	if(exc.bits.pstn_ring)
	{
		event.id = TRUNK_RINGING_ON;
		cp = getSymbol("ringid");
		if(cp)
			event.parm.ring.digit = atoi(cp);
		else
			event.parm.ring.digit = 0;
		event.parm.ring.duration = 1000;
		postEvent(&event);
		event.id = TRUNK_RINGING_OFF;
		postEvent(&event);
		time(&lastring);
	}
	if(exc.bits.pstn_wink)
	{
		event.id = TRUNK_LINE_WINK;
		event.parm.duration = 0;
		postEvent(&event);
	}
	if(exc.bits.hookstate)
	{
		switch(ioctl(dev, PHONE_HOOKSTATE))
		{
		case PSTN_ON_HOOK:
//			slog(SLOG_DEBUG) << "onhook" << endl;
			event.id = TRUNK_ON_HOOK;
			postEvent(&event);
			break;
		case PSTN_RINGING:
//			slog(SLOG_DEBUG) << "ringing" << endl;
			break;
		case PSTN_OFF_HOOK:
//			slog(SLOG_DEBUG) << "offhook" << endl;
			event,id = TRUNK_OFF_HOOK;
			postEvent(&event);
			break;
//		case PSTN_PULSE_DIAL:
//			slog(SLOG_DEBUG) << "pulse dial" << endl;
		}
	}
	if(exc.bits.dtmf_ready)
	{
		digit = ioctl(dev, PHONE_GET_DTMF_ASCII);
		event.id = TRUNK_DTMF_KEYUP;
		switch(digit)
		{
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			event.parm.dtmf.digit = digit - '0';
			break;
		case '*':
			event.parm.dtmf.digit = 10;
			break;
		case '#':
			event.parm.dtmf.digit = 11;
			break;
		}
		event.parm.dtmf.duration = 80;
		event.parm.dtmf.e1 = event.parm.dtmf.e2 = 0;
		postEvent(&event);
	}
	if(exc.bits.f0)
	{
		switch(driver)
		{
#ifdef	IXJ_PHONEDEV
		case ixj_driver:
			if(ioctl(dev, IXJCTL_GET_FILTER_HIST, 0) & 1)
			{
				event.id = TRUNK_CPA_DIALTONE;
				postEvent(&event);
			}
			break;
#endif
		}
	}
	if(exc.bits.f1)
	{
		switch(driver)
		{
#ifdef	IXJ_PHONEDEV
		case ixj_driver:
			if(ioctl(dev, IXJCTL_GET_FILTER_HIST, 0) & 1)
			{
				event.id = TRUNK_CPA_BUSYTONE;
				postEvent(&event);
			}
			break;
#endif
		}
	}
	if(exc.bits.caller_id && flags.dsp == DSP_MODE_CALLERID)
	{

		event.id = TRUNK_CALLER_ID;
		memset(&cid, 0, sizeof(cid));
		switch(driver)
		{
#ifdef IXJ_PHONEDEV
		case ixj_driver:
			ioctl(dev, IXJCTL_CID, &cid.ixjcid);
			strncpy(buf, cid.ixjcid.number, cid.ixjcid.numlen);
			buf[cid.ixjcid.numlen] = 0;
			if(cid.ixjcid.numlen)
			{
				setSymbol("callertype", 4);
				setSymbol("callertype", "cid");
				setSymbol("caller", 11);
				setSymbol("caller", buf);
				setConstant("callerid", buf);
			}
			strncpy(buf, cid.ixjcid.name, cid.ixjcid.namelen);
			buf[cid.ixjcid.namelen] = 0;
			if(cid.ixjcid.namelen)
			{
				setSymbol("callername", 80);
				setSymbol("callername", buf);
			}
			if(cid.ixjcid.numlen)
				flags.dsp = DSP_MODE_INACTIVE;
			break;
#endif
		}
	}		
}
		
void TrunkDriver::setTimer(timeout_t ptimer)
{
	TimerPort::setTimer(ptimer);
	if(!service->isThread())
		service->Update();
}

void TrunkDriver::incTimer(timeout_t ptimer)
{
	TimerPort::incTimer(ptimer);
	service->Update();
}

unsigned long TrunkDriver::getIdleTime(void)
{
	time_t now;

	time(&now);
	if(handler == &TrunkDriver::idleHandler)
		return now - idle;

	return 0;
}

void TrunkDriver::setHookState(bool offhook)
{
	if(offhook == flags.offhook)
		return;

	if(offhook)
		ioctl(dev, PHONE_PSTN_SET_STATE, PSTN_OFF_HOOK);
	else
		ioctl(dev, PHONE_PSTN_SET_STATE, PSTN_ON_HOOK);

	flags.offhook = offhook;
}

struct phone_capability *TrunkDriver::getCapability(phone_cap cid, int sub)
{
	int cap;

	for(cap = 0; cap < cap_count; ++cap)
	{
		if(cap_list[cap].captype != cid)
			continue;

		if(sub == -1 || cap_list[cap].cap)
			return &cap_list[cap];
	}
	return NULL;
}


