/************************************************************************/
/* File		datafile.h						*/
/*									*/
/* Purpose	This C++ program file contains the implementation for	*/
/*		the DataFile class. The DataFile class manages a data	*/
/*		file and an index file. The data file contains records	*/
/*		of DataRecord type and the index file indexes the	*/
/*		records in the data file.				*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, June 30, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include "datafile.h"

/* The following variable is the error message array that is specific	*/
/* to the DataFile class. The default UtilityModule error values are	*/
/* overwritten by the DataFile object constructor.			*/

static char *errormessages[] =
   {
       "No error",				// NoError
       "DataFile object not prepared",		// NoPrep
       "DataFile object is open",		// IsOpen
       "DataFile object is closed",		// IsClosed
       "Could not create data file",		// MakeData
       "Could not create index file",		// MakeIndex
       "Could not open data file",		// OpenData
       "Could not open index file",		// OpenIndex
       "Could not close data file",		// CloseData
       "Could not close index file",		// CloseIndex
       "No search key",				// NoKey
       "Error searching for record",		// BadFind
       "Error writing data record",		// BadWrite
       "Error reading data record",		// BadRead
       "Error changing index value",		// BadChange
       "Error adding record to data file",	// BadAddData
       "Error adding key to index file",	// BadAddIndex
       "Error removing record from data file",	// BadDelData
       "Error removing key from index file",	// BadDelIndex
   };

/************************************************************************/
/* Function	DataFile()						*/
/*									*/
/* Purpose	This is the default constructor for a DataFile object.	*/
/*		The default constructor initializes the private member	*/
/*		variables to show that the object is empty.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The DataFile object will be initialized but the object	*/
/*		will still need to be prepared for use before it can	*/
/*		be used. (See Prepare_DataFile).			*/
/************************************************************************/

DataFile::DataFile()
   {
      itsmodulename	= "DataFile";
      itsmaxerror	= DataFileErrors;
      itserrormessages	= errormessages;
      itscanopenflag	= false;
      itsisopenflag	= false;
   }

/************************************************************************/
/* Function	DataFile(const String& filename, const int elements,	*/
/*		   const DataElementType* list)				*/
/*									*/
/* Purpose	This is an alternate DataFile constructor. This		*/
/*		constructor will initialize a DataFile object AND	*/
/*		prepare the object for use (See Prepare_DataFile).	*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain a complete path for the data file to operate	*/
/*		on. The variable 'elements' must contain the number of	*/
/*		data elements in the data file's record. Finally, the	*/
/*		variable 'list' must contain an array of element types,	*/
/*		one for each element in the data file's record.		*/
/*									*/
/* Output	This alternate DataFile constructor will intialize the	*/
/*		DataFile object and prepare it to be used. The object	*/
/*		is prepared for use but the files must still be opened	*/
/*		(See Open_Files).					*/
/************************************************************************/

DataFile::DataFile(const String& filename, const int elements,
   const DataElementType* list)
   {
      itsmodulename	= "DataFile";
      itsmaxerror	= DataFileErrors;
      itserrormessages	= errormessages;
      itscanopenflag	= false;
      itsisopenflag	= false;
      Prepare_DataFile(filename, elements, list);
   }

/************************************************************************/
/* Function	~DataFile()						*/
/*									*/
/* Purpose	This is the default destructor for a DataFile object.	*/
/*		The destructor will close the files if they are open.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If the data file and the index file were open, they	*/
/*		will be closed.						*/
/************************************************************************/

DataFile::~DataFile()
   {
      if (itsisopenflag == true)
         {
	    Close_Files();
	 }
   }

/************************************************************************/
/* Function	int Record_Size(void)					*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. The	*/
/*		function returns the number of bytes that are required	*/
/*		to write the object to a file.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the number of bytes that are	*/
/*		required to write the object to a file.			*/
/************************************************************************/

int DataFile::Record_Size(void)
   {
      return(sizeof(int) + itsdatafilename.Length() + 1 +
         itsdatarecord.Record_Size());
   }

/************************************************************************/
/* Function	status Read_From_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will read a DataFile object from a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		reading the object from the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataFile object is to be read from	*/
/*									*/
/* Output	If this function is able to read in a DataFile object	*/
/*		from 'file' then this function will return OK and the	*/
/*		DataFile object will contain the contents that were	*/
/*		read in from the file. However, the DataFile object	*/
/*		cannot be used until the files have been opened. If	*/
/*		this function is not able to read in a DataFile object	*/
/*		from 'file' then this function will return ERROR.  All	*/
/*		errors by this function are handled by the ObjectFile	*/
/*		class.							*/
/************************************************************************/

status DataFile::Read_From_File(File& file, const int filehandle)
   {
      status		result		= OK;

      itscanopenflag	= false;
      itsisopenflag	= false;
      if (Read_String_From_ObjectFile(file, filehandle, itsdatafilename)
         == ERROR)
	 {
	    /* Could not read DataFile filename from file.		*/

	    file.Set_Error_Info_String(
	       "Could not read DataFile's file name from file");
	    result	= ERROR;
	 }
      else if (itsdatarecord.Read_From_File(file, filehandle) == ERROR)
         {
	    /* Could not read DataRecord from file.			*/

	    file.Set_Error_Info_String(
	       "Could not read DataFile's DataRecord from file");
	    result	= ERROR;
	 }
      else
         {
	    itscanopenflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_From_File(File& file, const int filehandle)*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will write a DataFile object to a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		writing the object to the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataFile object is to be written to.	*/
/*									*/
/* Output	If this function is able to write the DataFile object	*/
/*		to 'file' then this function will return OK. If this	*/
/*		function is not able to write a DataFile object to	*/
/*		'file' then this function will return ERROR. All errors	*/
/*		by this function are handled by the ObjectFile class.	*/
/************************************************************************/

status DataFile::Write_To_File(File& file, const int filehandle)
   {
      status		result		= OK;

      if (Write_String_To_ObjectFile(file, filehandle, itsdatafilename)
         == ERROR)
	 {
	    /* Could not write DataFile filename to file.		*/

	    file.Set_Error_Info_String(
	       "Could not write DataFile's file name to file");
	    result	= ERROR;
	 }
      else if (itsdatarecord.Write_To_File(file, filehandle) == ERROR)
         {
	    /* Could not write DataRecord from file.			*/

	    file.Set_Error_Info_String(
	       "Could not write DataFile's DataRecord to file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Prepare_DataFile(const String& filename,		*/
/*		   const int elements, const DataElementType* list)	*/
/*									*/
/* Purpose	This function can be used to prepare a DataFile object	*/
/*		to perform file operations. A DataFile object must be	*/
/*		prepared before you can perform file operations on a	*/
/*		data file. After the DataFile object has been prepared,	*/
/*		you must open the data file. Then you can perform file	*/
/*		operations on it.					*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the name of the data file to operate on. The	*/
/*		filename should not have any path information included	*/
/*		with it. The variable 'elements' must contain the	*/
/*		number of DataElement elements that are a part of this	*/
/*		DataFile file's DataRecord. The variable 'list' must	*/
/*		contain the list of element types, one for each		*/
/*		element, that are a part of this DataFile file's	*/
/*		DataRecord.						*/
/*									*/
/* Output	The DataFile object will be prepared to perform file	*/
/*		operations. However, you must still open the file	*/
/*		before you can actually perform file operations.	*/
/************************************************************************/

void DataFile::Prepare_DataFile(const String& filename,
   const int elements, const DataElementType* list)
   {
      itsdatafilename	= filename;
      itsdatarecord.Set_Record_Elements(elements, list);
      itscanopenflag	= true;
   }

/************************************************************************/
/* Function	void Format_Record(DataRecord& record)			*/
/*									*/
/* Purpose	This function can be used to format a DataRecord object	*/
/*		to read and write records to and from the DataFile	*/
/*		object.							*/
/*									*/
/* Input	This function expects the variable 'record' to be a	*/
/*		reference to the DataRecord that is to be formatted.	*/
/*		Any data that the record contains will be lost.		*/
/*									*/
/* Output	The variable 'record' will be formatted with the same	*/
/*		list of elements as the data record used by the		*/
/*		DataFile object. Any data that the record previously	*/
/*		contained will be deleted.				*/
/************************************************************************/

void DataFile::Format_Record(DataRecord& record)
   {
      record.Copy_Record(itsdatarecord);
   }

/************************************************************************/
/* Function	status Create_Files(const String& directory)		*/
/*									*/
/* Purpose	This function will create the data file and the index	*/
/*		file that are managed by the DataFile object. The files	*/
/*		will be created in the directory that is specified in	*/
/*		the variable 'directory'.				*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the directory to create the files in.		*/
/*									*/
/* Output	If this function is able to create the data and index	*/
/*		files that are managed by the DataFile object then this	*/
/*		function will return OK and the files will exist in the	*/
/*		directory that was specified in the variable		*/
/*		'directory'. If this function was not able to create	*/
/*		the files then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataFile::Create_Files(const String& directory)
   {
      status		result		= OK;
      String		datapath;
      String		indexpath;

      if (itscanopenflag == false)
         {
	    /* DataFile object has not been initialized.		*/

	    itserrorinfo	= "Attempted to create ";
	    itserrorinfo	+= itsdatafilename;
	    itserrorinfo	+= " in directory ";
	    itserrorinfo	+= directory;
	    itserror		= DataFileNoPrep;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsisopenflag == true)
         {
	    /* DataFile object has open files.				*/

	    itserrorinfo	= "Attempted to create ";
	    itserrorinfo	+= itsdatafilename;
	    itserrorinfo	+= " in directory ";
	    itserrorinfo	+= directory;
	    itserror		= DataFileIsOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Create the complete path for the data file and for the	*/
	    /* index file.						*/

	    datapath	= directory;

	    /* Make sure the directory path ends with '/'.		*/

	    if (datapath.Data()[datapath.Length() - 1] != '/')
	       {
	          datapath	+= "/";
	       }
	    datapath	+= itsdatafilename;
	    indexpath	= datapath;
	    indexpath	+= ".ndx";

	    /* Create the data file and the index file.			*/

	    if (itsdatafile.Create_ObjectFile_File(datapath) == ERROR)
	       {
	          /* Could not create data file.			*/

	          itserrorinfo	= "Attempted to create ";
	          itserrorinfo	+= itsdatafilename;
	          itserrorinfo	+= " in directory ";
	          itserrorinfo	+= directory;
	          itserror	= DataFileMakeData;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (itsindexfile.Create_DataIndex_File(indexpath)
	       == ERROR)
	       {
	          /* Could not create index file.			*/

	          itserrorinfo	= "Attempted to create ";
	          itserrorinfo	+= itsdatafilename;
	          itserrorinfo	+= " in directory ";
	          itserrorinfo	+= directory;
	          itserror	= DataFileMakeIndex;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_Files(const String& directory)		*/
/*									*/
/* Purpose	This function will open the data file and the index	*/
/*		file that are managed by the DataFile object. The files	*/
/*		will be opened in the directory that is specified in	*/
/*		the variable 'directory'.				*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the directory to open the files in.		*/
/*									*/
/* Output	If this function is able to open the data and index	*/
/*		files that are managed by the DataFile object then this	*/
/*		function will return OK. If this function was not able	*/
/*		to open the files then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status DataFile::Open_Files(const String& directory)
   {
      status		result		= OK;
      String		datapath;
      String		indexpath;

      if (itscanopenflag == false)
         {
	    /* DataFile object has not been initialized.		*/

	    itserrorinfo	= "Attempted to open ";
	    itserrorinfo	+= itsdatafilename;
	    itserrorinfo	+= " in directory ";
	    itserrorinfo	+= directory;
	    itserror		= DataFileNoPrep;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsisopenflag == true)
         {
	    /* DataFile object has open files.				*/

	    itserrorinfo	= "Attempted to open ";
	    itserrorinfo	+= itsdatafilename;
	    itserrorinfo	+= " in directory ";
	    itserrorinfo	+= directory;
	    itserror		= DataFileIsOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Create the complete path for the data file and for the	*/
	    /* index file.						*/

	    datapath	= directory;

	    /* Make sure the directory path ends with '/'.		*/

	    if (datapath.Data()[datapath.Length() - 1] != '/')
	       {
	          datapath	+= "/";
	       }
	    datapath	+= itsdatafilename;
	    indexpath	= datapath;
	    indexpath	+= ".ndx";

	    /* Open the data file and the index file.			*/

	    if (itsdatafile.Open_ObjectFile_File(datapath) == ERROR)
	       {
	          /* Could not open data file.				*/

	          itserrorinfo	= "Attempted to open ";
	          itserrorinfo	+= itsdatafilename;
	          itserrorinfo	+= " in directory ";
	          itserrorinfo	+= directory;
	          itserror	= DataFileOpenData;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (itsindexfile.Open_DataIndex_File(indexpath) == ERROR)
	       {
	          /* Could not open index file.				*/

	          itserrorinfo	= "Attempted to open ";
	          itserrorinfo	+= itsdatafilename;
	          itserrorinfo	+= " in directory ";
	          itserrorinfo	+= directory;
	          itserror	= DataFileOpenIndex;
	          result	= ERROR;
	          Report_Error();
	       }
	    else
	       {
	          itsisopenflag	= true;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Close_Files(void)				*/
/*									*/
/* Purpose	This function will close the data file and the index	*/
/*		file that are managed by the DataFile object.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function was able to close the data file and	*/
/*		the index file that are managed by the DataFile object	*/
/*		then this function will return OK. If this function is	*/
/*		not able to close the files then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataFile::Close_Files(void)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataFile is not open.					*/

	    itserrorinfo	= "Attempted to close ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsdatafile.Close_ObjectFile_File() == ERROR)
	 {
	    /* Could not close data file.				*/

	    itserrorinfo	= "Attempted to close ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileCloseData;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsindexfile.Close_DataIndex_File() == ERROR)
	 {
	    /* Could not close index file.				*/

	    itserrorinfo	= "Attempted to close ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileCloseIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      itsisopenflag	= false;
      return(result);
   }

/************************************************************************/
/* Function	status Store_Record(DataRecord& record)			*/
/*									*/
/* Purpose	This function will write a record to a DataFile object.	*/
/*		The first element in the record is used as the search	*/
/*		key. If the record does not already exist in the file	*/
/*		then the record will be added to the file. If the	*/
/*		record does exist in the file then the old record will	*/
/*		be replaced with the new record.			*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a record that is formatted for the DataFile that it	*/
/*		contains data for.					*/
/*									*/
/* Output	If this function is able to add the record to the file,	*/
/*		or if this function is able to replace an old record	*/
/*		with the new record, then this function will return OK.	*/
/*		If this function is not able to store the record in the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataFile::Store_Record(DataRecord& record)
   {
      status			result		= OK;
      String			key;
      FilePointer		pointer;
      FilePointer		pointer2;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to store record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Use the first element in the data file's record to get	*/
	    /* the key that is used to search for the record.		*/

	    record.Get_Key(key);

	    /* Make sure that there is a key to search for.		*/

	    if (key.Length() == 0)
	       {
	          /* No key specified or error getting key.		*/

	          itserrorinfo	= "Attempted to store record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileNoKey;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Get the data file record pointer by searching for the	*/
	    /* key.							*/

	    else if (itsindexfile.Find_Key(key, pointer) == ERROR)
	       {
	          /* Report error */

	          itserrorinfo	= "Attempted to store record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the record already exists in the index file then we	*/
	    /* need to write the new record over the old record.	*/

	    else if (pointer != 0)
	       {
	          /* The key is already in the index so save the	*/
		  /* current record pointer in case it changes.		*/

		  pointer2	= pointer;

		  /* Write the new record over the old record.		*/

		  if (itsdatafile.Write_ObjectFile_Record(record,
		     pointer) == ERROR)
		     {
	                /* Report error */

	                itserrorinfo	= "Attempted to store record in ";
	                itserrorinfo	+= itsdatafilename;
	                itserror	= DataFileBadWrite;
	                result		= ERROR;
	                Report_Error();
		     }

		  /* If writing the new record over the old record	*/
		  /* changed the record's data file pointer then we	*/
		  /* need to record this change in the index file.	*/

		  else if (pointer2 != pointer)
		     {
		        /* Change the record's data file pointer.	*/

		        if (itsindexfile.Change_Value(key, pointer)
			   == ERROR)
			   {
	                      /* Report error */

	                      itserrorinfo	= "Attempted to store "
			      			  "record in ";
	                      itserrorinfo	+= itsdatafilename;
	                      itserror		= DataFileBadChange;
	                      result		= ERROR;
	                      Report_Error();
			   }
		     }
	       }

	    /* If we made it here then the record needs to be added to	*/
	    /* the data file.						*/

	    else if (itsdatafile.Add_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to store record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadAddData;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Add the key and the data file pointer for the record to	*/
	    /* the index file.						*/

	    else if (itsindexfile.Add_Key(key, pointer) == ERROR)
	       {
	          itserrorinfo	= "Attempted to store record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadAddKey;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_Record(DataRecord& record)		*/
/*									*/
/* Purpose	This function will remove a record from a DataFile	*/
/*		object. The first element in the record is used as the	*/
/*		search key.						*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a record that is formatted for the DataFile that it	*/
/*		contains data for.					*/
/*									*/
/* Output	If this function is able to remove the record from the	*/
/*		file then this function will return OK. If this		*/
/*		function is not able to remove the record from the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataFile::Remove_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to remove record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Use the first element in the data file's record to get	*/
	    /* the key that is used to search for the record.		*/

	    record.Get_Key(key);

	    /* Make sure that there is a key to search for.		*/

	    if (key.Length() == 0)
	       {
	          /* No key specified or error getting key.		*/

	          itserrorinfo	= "Attempted to remove record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileNoKey;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Get the data file record pointer by searching for the	*/
	    /* key.							*/

	    else if (itsindexfile.Find_Key(key, pointer) == ERROR)
	       {
	          /* Report error */

	          itserrorinfo	= "Attempted to remove record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Make sure that the record exists in the data file.	*/

	    else if (pointer == 0)
	       {
	          /* Key does not exist in the index file.		*/

	          itserrorinfo	= "Attempted to remove record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileNoFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Remove the record from the data file.			*/

	    else if (itsdatafile.Remove_ObjectFile_Record(pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to remove record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadDelData;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Remove the key from the index file.			*/

	    else if (itsindexfile.Remove_Key(key) == ERROR)
	       {
	          itserrorinfo	= "Attempted to remove record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadDelIndex;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Record(DataRecord& record)			*/
/*									*/
/* Purpose	This function can be used to return a record that is in	*/
/*		the DataFile file. The first element of the variable	*/
/*		'record' will be used as the search key.		*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a DataRecord object. The record does not	*/
/*		need to be formatted as its contents will be deleted.	*/
/*		However, the first element in the record must contain	*/
/*		the key to use when searching for the record.		*/
/*									*/
/* Output	If this function is able to read in the record from the	*/
/*		the file then this function will return OK and the	*/
/*		variable 'record' will contain the contents that were	*/
/*		read in from the file. If this function was not able	*/
/*		to read in the record from the file then this function	*/
/*		will return ERROR. All errors by this function are	*/
/*		reported to stderr.					*/
/************************************************************************/

status DataFile::Get_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to get record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Use the first element in the data file's record to get	*/
	    /* the key that is used to search for the record.		*/

	    record.Get_Key(key);

	    /* Make sure that there is a key to search for.		*/

	    if (key.Length() == 0)
	       {
	          /* No key specified or error getting key.		*/

	          itserrorinfo	= "Attempted to get record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileNoKey;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Get the data file record pointer by searching for the	*/
	    /* key.							*/

	    else if (itsindexfile.Find_Key(key, pointer) == ERROR)
	       {
	          /* Report error */

	          itserrorinfo	= "Attempted to get record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Make sure that the record exists in the data file.	*/

	    else if (pointer == 0)
	       {
	          /* Key does not exist in the index file.		*/

		  record.Clear_Record();
	       }

	    /* Read the record from the data file.			*/

	    else if (itsdatafile.Read_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to get record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadRead;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_First_Record(DataRecord& record)		*/
/*									*/
/* Purpose	This function can be used to return the first record	*/
/*		that is in the DataFile file.				*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a DataRecord object. The record does not	*/
/*		need to be formatted as its contents will be deleted.	*/
/*									*/
/* Output	If this function is able to read in the first record	*/
/*		from the file then this function will return OK and the	*/
/*		variable 'record' will contain the contents that were	*/
/*		read in from the file. If this function was not able	*/
/*		to read in the first record from the file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataFile::Get_First_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to get first record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Get the pointer to the first record in the data file.	*/

	    if (itsindexfile.Get_First_Key(key, pointer) == ERROR)
	       {
	          itserrorinfo	= "Attempted to get first record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* See if a record pointer was returned.			*/

	    else if (pointer == 0)
	       {
	          /* No records.					*/

		  record.Clear_Record();
	       }

	    /* Read in the record from the data file.			*/

	    else if (itsdatafile.Read_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to get first record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadRead;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Next_Record(DataRecord& record)		*/
/*									*/
/* Purpose	This function can be used to return the next record	*/
/*		that is in the DataFile file.				*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a DataRecord object. The record does not	*/
/*		need to be formatted as its contents will be deleted.	*/
/*									*/
/* Output	If this function is able to read in the next record	*/
/*		from the file then this function will return OK and the	*/
/*		variable 'record' will contain the contents that were	*/
/*		read in from the file. If this function was not able	*/
/*		to read in the next record from the file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataFile::Get_Next_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to get next record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Get the pointer to the next record in the data file.	*/

	    if (itsindexfile.Get_Next_Key(key, pointer) == ERROR)
	       {
	          itserrorinfo	= "Attempted to get next record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* See if a record pointer was returned.			*/

	    else if (pointer == 0)
	       {
	          /* No more records.					*/

		  record.Clear_Record();
	       }

	    /* Read in the record from the data file.			*/

	    else if (itsdatafile.Read_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to get next record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadRead;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Last_Record(DataRecord& record)		*/
/*									*/
/* Purpose	This function can be used to return the last record	*/
/*		that is in the DataFile file.				*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a DataRecord object. The record does not	*/
/*		need to be formatted as its contents will be deleted.	*/
/*									*/
/* Output	If this function is able to read in the last record	*/
/*		from the file then this function will return OK and the	*/
/*		variable 'record' will contain the contents that were	*/
/*		read in from the file. If this function was not able	*/
/*		to read in the last record from the file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataFile::Get_Last_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to get last record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Get the pointer to the last record in the data file.	*/

	    if (itsindexfile.Get_Last_Key(key, pointer) == ERROR)
	       {
	          itserrorinfo	= "Attempted to get last record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* See if a record pointer was returned.			*/

	    else if (pointer == 0)
	       {
	          /* No records.					*/

		  record.Clear_Record();
	       }

	    /* Read in the record from the data file.			*/

	    else if (itsdatafile.Read_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to get last record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadRead;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Previous_Record(DataRecord& record)		*/
/*									*/
/* Purpose	This function can be used to return the previous record	*/
/*		that is in the DataFile file.				*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a DataRecord object. The record does not	*/
/*		need to be formatted as its contents will be deleted.	*/
/*									*/
/* Output	If this function is able to read in the previous record	*/
/*		from the file then this function will return OK and the	*/
/*		variable 'record' will contain the contents that were	*/
/*		read in from the file. If this function was not able	*/
/*		to read in the previous record from the file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataFile::Get_Previous_Record(DataRecord& record)
   {
      status		result		= OK;
      String		key;
      FilePointer	pointer;

      if (itsisopenflag == false)
         {
	    /* Database is not open.					*/

	    itserrorinfo	= "Attempted to get previous record in ";
	    itserrorinfo	+= itsdatafilename;
	    itserror		= DataFileIsClosed;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Get the pointer to the previous record in the data file.	*/

	    if (itsindexfile.Get_Previous_Key(key, pointer) == ERROR)
	       {
	          itserrorinfo	= "Attempted to get previous record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadFind;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* See if a record pointer was returned.			*/

	    else if (pointer == 0)
	       {
	          /* No more records.					*/

		  record.Clear_Record();
	       }

	    /* Read in the record from the data file.			*/

	    else if (itsdatafile.Read_ObjectFile_Record(record, pointer)
	       == ERROR)
	       {
	          itserrorinfo	= "Attempted to get previous record in ";
	          itserrorinfo	+= itsdatafilename;
	          itserror	= DataFileBadRead;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }
