/************************************************************************/
/* File		webpublishpublish.cpp					*/
/*									*/
/* Purpose	This C++ program file contains the private WebPublish	*/
/*		member functions that process the --publish command	*/
/*		line option. The --publish command will publish all or	*/
/*		part of an account's website to the account's server.	*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	0.1.0 (Prototype)					*/
/*									*/
/* Date		Sunday, July 7, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <sys/types.h>			// Directory functions.
#include <sys/stat.h>			// stat functions.
#include <unistd.h>			// Unix standards.
#include <dirent.h>			// Directory functions.
#include <errno.h>			// errno.
#include <time.h>			// strftime.
#include <stdlib.h>			// atoi.
#include "webpublish.h"			// The WebPublish class.

/************************************************************************/
/* Function	condition is_publish_path_excluded(const String& path)	*/
/*									*/
/* Purpose	This function can be used to see if a given path is	*/
/*		listed in the database as one of the paths that is to	*/
/*		be excluded from being published.			*/
/*									*/
/* Input	This function expects the variable 'path' to contain	*/
/*		the complete file path that is to be searched for in	*/
/*		the database.						*/
/*									*/
/* Output	If the given path is listed in the database as one of	*/
/*		the paths that is to be excluded from being published	*/
/*		then this function will return true. Otherwise, this	*/
/*		function will return false.				*/
/************************************************************************/

condition WebPublish::is_publish_path_excluded(const String& path)
   {
      condition		result;
      String		key;
      DataRecord	record;

      key		= itsaccount;
      key		+= path;

      itsdatabase.Format_Record(WebPublishPublish, record);
      record.Set_Data(0, key);
      itsdatabase.Get_Record(WebPublishPublish, record);
      record.Get_Key(key);
      if (key.Length() != 0)
         {
	    result	= true;
	 }
      else
         {
	    result	= false;
	 }

      return(result);
   }

/************************************************************************/
/* Function	status transfer_file(DataRecord& account,		*/
/*		   const String& localpath, const String& serverpath)	*/
/*									*/
/* Purpose	This function is responsible for transferring a file	*/
/*		to an account's remote server. If the extension for the	*/
/*		file that is being transferred is associated with a	*/
/*		shell program then the shell program will be called	*/
/*		before the file is actually transferred.		*/
/*									*/
/* Input	This function expects the variable 'account' to contain	*/
/*		a formatted record containing the information for the	*/
/*		account that is having a file transferred. The variable	*/
/*		'localpath' must contain the complete path of the local	*/
/*		copy of the file that is to be transferred. Finally,	*/
/*		the variable 'serverpath' must contain the complete	*/
/*		serverpath (ralative to the server's base directory)	*/
/*		for the file on the server.				*/
/*									*/
/* Output	If this function is able to transfer the file to the	*/
/*		server then this function will return OK. If this	*/
/*		function is not able to transfer the file to the server	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status WebPublish::transfer_file(DataRecord& account,
   const String& localpath, const String& serverpath)
   {
      status		result		= OK;
      String		workpath	= localpath;
      String		newpath		= serverpath;
      FTPMode		mode		= FTPBinary;
      int		tempmode;		// Temporary file mode.
      int		systemresult;		// Shell return value.
      register int	index;			// Duh!
      String		key;			// Record key.
      String		shell;			// Account's shell.
      String		website;		// Account's website.
      String		server;			// Account's server.
      String		user;			// Account's user.
      String		directory;		// Account's directory.
      String		extension;		// File's extension.
      String		filename;		// Isolated filename.
      DataRecord	record;			// Database record.
      char		buffer[2048];		// Character buffer.

      /* Write a message to stdout if the verbose flag is true.		*/

      if (itsverboseflag == true)
         {
	    std::cout << "Transferring " << serverpath.Data();
	 }

      /* Beginning at the end of the file path...			*/

      index		= workpath.Length();

      /* Search backwards to isolate the file's extension.		*/

      while (workpath.Data()[--index] != '.' && index > 0);

      /* If a file extension was isolated, then we need to check the	*/
      /* file's extension to see if a shell program needs to be called	*/
      /* before the file is transferred.				*/

      if (workpath.Data()[index] == '.' && index > 0)
         {
	    /* Build the key to use when searching for a shell program.	*/

	    key		= itsaccount;
	    extension	= workpath.Data() + index + 1;
	    key		+= extension;

	    /* Search for a shell program for the given extension.	*/

	    itsdatabase.Format_Record(WebPublishShell, record);
	    record.Set_Data(0, key);
	    if (itsdatabase.Get_Record(WebPublishShell, record) == ERROR)
	       {
	          /* Database error.					*/

		  itserrorinfo	= "Attempted to transfer ";
		  itserrorinfo	+= serverpath;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }
	    else
	       {
	          /* If the returned record's key length is NOT zero	*/
		  /* then this extension is associated with a shell	*/
		  /* program.						*/

	          record.Get_Key(key);
		  if (key.Length() != 0)
		     {
		        /* Get the account's information.		*/

			record.Get_Data(3, shell);
			account.Get_Data(1, website);
			account.Get_Data(2, server);
			account.Get_Data(3, user);
			account.Get_Data(5, directory);

			/* If this account includes an offset directory	*/
			/* then the system call needs to include the	*/
			/* directory in the command line.		*/

			if (directory.Length() == 0)
			   {
			      sprintf(buffer, "%s -a %s -w %s -s %s -u "
			         "%s -e %s < %s > %s", shell.Data(),
			         itsaccount.Data(), website.Data(),
			         server.Data(), user.Data(),
			         extension.Data(), localpath.Data(),
			         itstempfile.Data());
			   }

			/* Else, the command line does not include a	*/
			/* directory.					*/

			else
			   {
			      sprintf(buffer, "%s -a %s -w %s -s %s -u "
			         "%s -d %s -e %s < %s > %s",
				 shell.Data(), itsaccount.Data(),
				 website.Data(), server.Data(),
				 user.Data(), directory.Data(),
			         extension.Data(), localpath.Data(),
			         itstempfile.Data());
			   }

			/* Call the shell program.			*/

			systemresult	= system(buffer);
		        if ((systemresult == 127 || systemresult == 32512)
		           && errno != 0)
		           {
			      /* System call crashed and burned.	*/

			      itserrorinfo	= "Attempted to "
						  "transfer ";
			      itserrorinfo	+= serverpath;
			      itserror		= WebPublishBadShell;
			      result		= ERROR;
			      Report_Error();
			   }
			else
			   {
			      /* Determine what the shell program did.	*/

			      switch(systemresult)
			         {
				    /* Shell program encountered an	*/
				    /* error.				*/

				    case 1:	// Shell program error.
				    case 256:	// reversed byte order.
				       /* The shell program returned	*/
				       /* an error.			*/

				       itserrorinfo = "Attempted to "
						      "transfer ";
				       itserrorinfo += serverpath;
				       itserror	    = WebPublishShellVal;
				       result	    = ERROR;
				       Report_Error();
				       break;

				    /* Shell program did not make any	*/
				    /* changes to the file.		*/

				    case 2:	// No changes made.
				    case 512:	// reversed byte order.
				       break;

				    /* Shell program made changes to	*/
				    /* the file.			*/

				    case 3:	// Changes made.
				    case 768:	// reversed byte order.
				       workpath	= itstempfile;
				       break;

				    /* All other values are an error.	*/

				    default:
				       /* Unknown return result.	*/

				       itserrorinfo = "Attempted to "
						      "transfer ";
				       itserrorinfo += serverpath;
				       itserror	    = WebPublishNotShell;
				       result	    = ERROR;
				       Report_Error();
				 }

			      /* Change the transfer mode to Text since	*/
			      /* we had to call a shell program.	*/

			      mode	= FTPText;
			   }
		     }
	       }
	 }

      /* If there have been no errors then we should be ready to	*/
      /* transfer the file to the server.				*/

      if (result == OK)
         {
	    /* Search backwards looking for '/' to isolate the		*/
	    /* filename.						*/

	    index	= newpath.Length();
	    while (newpath.Data()[--index] != '/' && index > 0);

	    /* If '/' was found then we have isolated the filename.	*/

	    if (*(newpath.Data() + index) == '/' && index > 0)
	       {
	          *(newpath.Data() + index)	= '\0';
	          filename	= newpath.Data() + index + 1;
	       }

	    /* Else, the string consists only of a filename.		*/

	    else
	       {
		  filename	= newpath;
		  newpath	= "";
	       }

	    /* Determine if we need to override the transfer mode.	*/

	    index	= filename.Length();
	    while (filename.Data()[--index] != '.' && index > 0);

	    /* If we have isolated a file extension, use it as the key.	*/

	    if (filename.Data()[index] == '.')
	       {
	          extension	= filename.Data() + index + 1;
		  itsdatabase.Format_Record(WebPublishModes, record);
		  record.Set_Data(0, extension);

		  /* Try to get the record from the database.		*/

		  if (itsdatabase.Get_Record(WebPublishModes, record)
		     == ERROR)
		     {
		        /* Could not get database record.		*/

			itserrorinfo	= "Attempted to transfer ";
			itserrorinfo	+= serverpath;
			itserror	= WebPublishBadGet;
			result	= ERROR;
			Report_Error();
		     }
		  else
		     {
		        /* If the record contains a key then we have	*/
			/* a transfer mode for this file extension.	*/

		        record.Get_Key(extension);
			if (extension.Length() != 0)
			   {
			      /* Override the default transfer mode.	*/

			      record.Get_Data(1, tempmode);
			      mode		= (FTPMode)tempmode;
			   }
		     }
	       }
	    if (itsverboseflag == true)
	       {
	          std::cout << " in " << (mode == FTPText ? "TEXT" : "BINARY")
		     << " mode.\n";
		  fflush(stdout);
	       }

	    /* Change the server's directory...				*/

	    if (itsftp.Change_Work_Directory(newpath) == ERROR)
	       {
		  itserrorinfo	= "Attempted to transfer ";
		  itserrorinfo	+= serverpath;
		  itserror	= WebPublishChangeDir;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* ...and transfer the file to the server.			*/

	    else if (itsftp.Put_File(workpath, filename, mode) == ERROR)
	       {
	          /* Could not transfer file.				*/

		  itserrorinfo	= "Attempted to transfer ";
		  itserrorinfo	+= serverpath;
		  itserror	= WebPublishPutFile;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status publish_file(DataRecord& account,		*/
/*		   const String& localpath, const String& filepath)	*/
/*									*/
/* Purpose	This function will publish one single file from the	*/
/*		specified account's website directory to the specified	*/
/*		account's server.					*/
/*									*/
/*		This function will only transfer a file if one of the	*/
/*		following conditions is true:				*/
/*									*/
/*		Condition		Description			*/
/*									*/
/*		force			--force command line option was	*/
/*					included on the command line.	*/
/*									*/
/*		Not on server		The file does not already exist	*/
/*					on the server.			*/
/*									*/
/*		Modification date	The modification date of the	*/
/*					file on the server is older	*/
/*					than the local copy of the	*/
/*					file.				*/
/*									*/
/* Input	This function expects the variable 'account' to contain	*/
/*		a DataRecord with the account information. The variable	*/
/* 		'localpath' must contain the full file path for the	*/
/*		file in the local directory. The variable 'filepath'	*/
/*		must contain the name of the file that is to be		*/
/*		published.						*/
/*									*/
/* Output	If this function was able to publish the file to the	*/
/*		server then this function will return OK. If this	*/
/*		function was not able to publish the file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status WebPublish::publish_file(DataRecord& account,
   const String& localpath, const String& filepath)
   {
      status		result		= OK;
      condition		publish		= false;
      char		buffer[1024];
      time_t		timevalue;
      String		comparevalue;
      struct stat	entryinfo;

      /* Get the file's stats to verify that the file exists and also	*/
      /* to get the file's modification time from the stat information.	*/

      if (stat(localpath.Data(), &entryinfo) == -1)
         {
            /* Could not get stats */

	    itserrorinfo	= "Attempted to publish ";
	    itserrorinfo	+= localpath;
	    itserror		= WebPublishBadStat;
	    result		= ERROR;
	    Report_Error();
         }

      /* Find out if this file is listed in the database as a path	*/
      /* that is to be ignored by the publishing function.		*/

      if (is_publish_path_excluded(localpath) == true)
         {
	    if (itsverboseflag == true)
	       {
	          std::cout << "Excluding    : " << filepath.Data()
		     << std::endl;
		  fflush(stdout);
	       }
	 }

      /* If the --force switch was specified on the command line...	*/

      else if (itsforceflag == true)
         {
	    publish	= true;
	 }

      /* ...or if the file does not exist on the server...		*/

      else if (itsftp.Does_File_Exist(filepath) == false)
         {
	    publish	= true;
	 }

      /* ...or if the modification time of the local copy of the file	*/
      /* is more recent than the copy of the file on the server.	*/

      else
         {
	    /* Create a character string out of the local file's	*/
	    /* modification time.					*/

	    timevalue	= entryinfo.st_mtime;
	    strftime(buffer, 1024, "%Y%m%d%H%M%S",
	       gmtime(&timevalue));

	    /* Get the modification time from the copy of the file	*/
	    /* that is on the server.					*/

	    if (itsftp.Get_Modification_Time(filepath,
	       comparevalue) == ERROR)
	       {
	          /* Could not get mod time.				*/

		  itserrorinfo	= "Attempted to publish ";
		  itserrorinfo	+= localpath;
		  itserror	= WebPublishNoMod;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* Compare the two strings to determine the most recent.	*/

	    else if (comparevalue.Compare(buffer) != MORE)
	       {
		  publish	= true;
	       }
	 }

      /* If there are no errors and the file has been selected to be	*/
      /* published then...						*/

      if (result == OK && publish == true)
         {
	    /* Transfer the file to the server.				*/

	    result	= transfer_file(account, localpath, filepath);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status publish_files(DataRecord& account)		*/
/*									*/
/* Purpose	This function will publish a list of files that were	*/
/*		included on the command line. The list is stored in the	*/
/*		private member variable 'itssingles'.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itssingles' to contain a list of files that are to be	*/
/*		transferred to an account's server. The private member	*/
/*		variable 'itssinglecount' must contain the number of	*/
/*		files in the list.					*/
/*									*/
/* Output	If this function is able to transfer the list of files	*/
/*		to an account's server then this function will return	*/
/*		OK. If this function was not able to transfer the list	*/
/*		of files then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::publish_files(DataRecord& account)
   {
      status		result		= OK;
      register int	index;
      String		filepath;
      String		website;
      String		serverpath;

      /* This loop will transfer each file in the list.			*/

      for (index = 0; index < itssinglecount && result == OK; index++)
         {
	    /* Using the account's data, build the full server path.	*/

            account.Get_Data(1, website);

	    /* If the file that is being transferred begins with '/'...	*/

	    if (itssingles[index].Data()[0] != '/')
	       {
	          /* The server file = the file being transferred and	*/
		  /* the local file path = the account's website +	*/
		  /* the file being transferred.			*/

		  filepath	= itssingles[index];
	          website	+= "/";
		  website	+= itssingles[index];
	       }
	    else
	       {
	          /* Isolate the server path from the file being	*/
		  /* transferred by subtracting the account's website.	*/
		  /* The local path = the file being transferred.	*/

		  filepath	= itssingles[index].Data() +
				  website.Length();
		  website	= itssingles[index];
	       }

	    /* Find out if the path of the file on the server needs to	*/
	    /* be appended to an offset directory.			*/

	    account.Get_Data(5, serverpath);
	    if (serverpath.Length() != 0)
	       {
	          if (serverpath.Data()[serverpath.Length() - 1] != '/')
		     {
		        serverpath	+= "/";
		     }
		  serverpath	+= filepath;
	       }
	    else
	       {
	          serverpath	= filepath;
	       }

	    /* Publish the file to the server.				*/

	    result	= publish_file(account, website, serverpath);
	 }				// index < itssinglecount
      return(result);
   }

/************************************************************************/
/* Function	status publish_directory(DataRecord& account,		*/
/*		   const String& directory, const int depth)		*/
/*									*/
/* Purpose	This function is responsible for publishing a local	*/
/*		copy of a directory to an account's server. When this	*/
/*		function needs to publish a subdirectory, this function	*/
/*		will call itself.					*/
/*									*/
/* Input	This function expects the variable 'account' to contain	*/
/*		a formatted account record containing the account's	*/
/*		information. The variable 'directory' must contain the	*/
/*		local directory path that is to be published to the	*/
/*		account's server. Finally, the variable 'depth' must	*/
/*		contain the number of directory levels to traverse.	*/
/*									*/
/* Output	If this function is able to publish a local directory	*/
/*		to an account's server then this function will return	*/
/*		OK. If this function is not able to publish a local	*/
/*		directory then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::publish_directory(DataRecord& account,
   const String& directory, const int depth)
   {
      status		result		= OK;
      register int	index;			// Index.
      String		extension;		// File extension.
      String		website;		// Website directory.
      String		newpath;		// New path.
      String		serverpath;		// Server path.
      DIR*		workdir;		// Temporary directory.
      struct dirent*	entry;			// Directory entry.
      struct stat	entryinfo;		// Directory information.

      if (itsverboseflag == true)
	 {
	    std::cout << "Publishing directory " << directory.Data() << std::endl;
	    fflush(stdout);
	 }

      /* Open the directory so that we can read in the entries.		*/

      workdir		= opendir(directory.Data());
      if (workdir == (DIR*)0)
         {
	    /* Could not open directory.				*/

	    itserrorinfo	= "Attempted to publish directory ";
	    itserrorinfo	+= directory;
	    itserror		= WebPublishDirOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Get the website's local directory.			*/

	    account.Get_Data(1, website);
	    if (website.Data()[website.Length() - 1] != '/')
	       {
	          website	+= "/";
	       }
	 }

      /* This loop will read in each directory entry and process it.	*/

      while (result == OK)
         {
	    /* Get a directory entry from the directory.		*/

	    errno	= 0;
	    entry	= readdir(workdir);
	    if (entry == (struct dirent*)0)
	       {
	          if (errno != 0)
		     {
	                /* Could not read directory entry.		*/

			itserrorinfo	= "Attempted to publish "
					  "directory ";
			itserrorinfo	+= directory;
			itserror	= WebPublishDirRead;
			result		= ERROR;
			Report_Error();
		     }
		  break;
	       }

	    /* Ignore all hidden files.					*/

	    if (*entry->d_name == '.')
	       {
	          continue;
	       }

	    /* Begin to build the complete path for the entry.		*/

	    newpath	= directory;
	    if (newpath.Data()[newpath.Length() - 1] != '/')
	       {
	          newpath	+= "/";
	       }
	    newpath	+= entry->d_name;

	    /* Get the stats for this path to see if it is a file.	*/

	    if (stat(newpath.Data(), &entryinfo) == -1)
	       {
	          /* Could not get stat for directory entry.		*/

		  itserrorinfo	= "Attempted to publish directory ";
		  itserrorinfo	+= directory;
		  itserror	= WebPublishBadStat;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Exclude this entry if it is listed in the database as	*/
	    /* a path that should not be published.			*/

	    if (is_publish_path_excluded(newpath) == true)
	       {
	          if (itsverboseflag == true)
		     {
		        std::cout << "Excluding : " <<
			   newpath.Data() + website.Length() << std::endl;
			fflush(stdout);
		     }
	          continue;
	       }

	    /* Build the server's path for this entry.			*/

	    account.Get_Data(5, serverpath);
	    if (serverpath.Length() != 0)
	       {
	          if (serverpath.Data()[serverpath.Length() - 1] != '/')
	             {
	                serverpath	+= "/";
	             }
	       }
	    serverpath	+= newpath.Data() + website.Length();

	    /* If the directory entry is another directory...		*/

	    if (S_ISDIR(entryinfo.st_mode))
	       {
	          /* This is another directory. Do not publish it if	*/
		  /* --depth has been reached.				*/

	          if (depth == -1 || depth - 1 > 0)
		     {

			/* Find out if this directory exists on the	*/
			/* server.					*/

			if (itsftp.Does_Directory_Exist(serverpath)
			   == false)
			   {
			      /* The directory does not exist.		*/

			      if (itsverboseflag == true)
			         {
				    std::cout << "Creating directory " <<
				       serverpath.Data() << std::endl;
				    fflush(stdout);
				 }

			      /* Create the directory on the server.	*/

			      if (itsftp.Create_Recursive_Directory(
			         serverpath) == ERROR)
				 {
				    /* Could not create directory.	*/

				    itserrorinfo = "Attempted to publish "
						   "directory ";
				    itserrorinfo += directory;
				    itserror	 = WebPublishDirMake;
				    result	 = ERROR;
				    Report_Error();
				 }
			   }

			/* If we do not have any errors then...		*/

			if (result == OK)
			   {
			      /* If --depth was NOT specified on the	*/
			      /* command line then publish everything.	*/

			      if (depth == -1)
			         {
				    result	= publish_directory(
						  account, newpath,
						  -1);
				 }

			      /* Else, decrement --depth.		*/

			      else
			         {
		                    result	= publish_directory(
						  account, newpath,
						  depth - 1);
			         }
			   }
		     }
	       }

	    /* Else, if the directory entry is a file...		*/

	    else if (S_ISREG(entryinfo.st_mode))
	       {
	          /* This is a file so find out if the --extension	*/
		  /* qualifier was included on the command line.	*/

	          if (itsextension.Length() != 0)
		     {

		        /* The --extension qualifier was included on	*/
			/* the command line so make sure that this file	*/
			/* has the same extension.			*/

		        index	= newpath.Length();
			while (newpath.Data()[--index] != '.' && index > 0);
			if (newpath.Data()[index] == '.')
			   {
			      /* The extension has been isolated so	*/
			      /* compare the file's extension with the	*/
			      /* --extension qualifier.			*/

			      index++;
			      if (itsextension.Compare(newpath.Data()
			         + index) == EQUAL)
				 {
				    /* Publish the qualified files.	*/

				    result	= publish_file(account,
						  newpath,
						  serverpath.Data());
				 }
			   }
		     }

		  /* Else, the --extension qualifier was NOT included	*/
		  /* on the command line.				*/

		  else
		     {
		        /* Publish the file to the server.		*/

	      	        result	= publish_file(account, newpath,
				  serverpath.Data());
		     }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status publish_account(DataRecord& account)		*/
/*									*/
/* Purpose	This function is responsible for calling the recursive	*/
/*		function that will actually publish an account.		*/
/*									*/
/* Input	This function expects the variable 'account' to contain	*/
/*		a formatted account record containing the account's	*/
/*		information.						*/
/*									*/
/* Output	If this function is able to publish the account then	*/
/*		this function will return OK. If this function is not	*/
/*		able to publish the account then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status WebPublish::publish_account(DataRecord& account)
   {
      String		website;
      String		directory;

      /* Get the account's local website directory...			*/

      account.Get_Data(1, website);
      directory		= website;

      /* If --directory was specified on the command line...		*/

      if (itsdirectory.Length() != 0)
         {
	    /* Make sure that our directory ends with '/' and add the	*/
	    /* specified directory path to the full path.		*/

	    if (directory.Data()[directory.Length() - 1] != '/')
	       {
	          directory	+= "/";
	       }
	    directory	+= itsdirectory;
	 }

      /* Call the recursive function that will publish the directory.	*/

      return(publish_directory(account, directory, itsdepth));
   }

/************************************************************************/
/* Function	status publish(void)					*/
/*									*/
/* Purpose	This function is responsible for publishing all or part	*/
/*		of an accounts website to the account's server. This	*/
/*		function locates the account's record and uses that	*/
/*		information to establish a connection to the remote	*/
/*		server. After it has established a connection, this	*/
/*		function will transfer the files that need to be	*/
/*		transferred.						*/
/*									*/
/* Input	This function expects the private WebPublish member	*/
/*		variables to contain the relevant information.		*/
/*									*/
/* Output	If this function is able to publish all or part of an	*/
/*		account's website to the account's remote server then	*/
/*		this function will return OK. If this function is not	*/
/*		able to publish the account's website then this		*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status WebPublish::publish(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		account;
      String		server;
      String		user;
      String		password;
      String		systype;

      itsdatabase.Format_Record(WebPublishAccounts, record);
      record.Set_Data(0, itsaccount);

      /* Make sure that the required information was specified.		*/

      if (itsaccount.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to publish an account";
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Get the account's database record.				*/

      else if (itsdatabase.Get_Record(WebPublishAccounts, record)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to publish an account";
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Make sure that the record was in the database.		*/

	    record.Get_Key(account);
	    if (account.Length() == 0)
	       {
	          /* Account does not exist.				*/

		  itserrorinfo	= "Attempted to publish an account";
		  itserror	= WebPublishBadAccount;
		  result	= ERROR;
		  Report_Error();
	       }
	    else
	       {
	          /* Establish the FTP connection.			*/

		  record.Get_Data(2, server);
		  record.Get_Data(3, user);
		  record.Get_Data(4, password);
		  if (itsverboseflag == true)
		     {
		        std::cout << "Establishing a connection to "
			   << server.Data() << std::endl;
		        fflush(stdout);
		     }
		  if (itsftp.Connect_To_Host(server) == ERROR)
		     {
		        /* Could not establish FTP connection.		*/

			itserrorinfo	= "Attempted to publish an "
					  "account";
			itserror	= WebPublishBadHost;
			result		= ERROR;
			Report_Error();
		     }
		  else if (itsftp.Logon_To_Host(user, password) == ERROR)
		     {
		        /* Could not logon to host.			*/

			itserrorinfo	= "Attempted to publish an "
					  "account";
			itserror	= WebPublishBadLogon;
			result		= ERROR;
			Report_Error();
		     }
		  else
		     {
		        if (itsverboseflag == true)
		           {
			      std::cout << "Successfully logged into " <<
			         server.Data() << ".\n";
			      itsftp.Get_System_Type(systype);
			      if (systype.Length() != 0)
			         {
				    std::cout << "System type = " <<
				       systype.Data() << std::endl;
				 }
			      fflush(stdout);
		           }

			/* If individual files were specified on the	*/
			/* command line...				*/

			if (itssinglecount > 0)
			   {
			      if (itsverboseflag == true)
			         {
				    std::cout << "Publishing command line "
				       "files using account " <<
				       account.Data() << ".\n";
				    fflush(stdout);
				 }
			      result	= publish_files(record);
			   }

			/* Else, publish using qualifiers.		*/

			else
			   {
			      if (itsverboseflag == true)
			         {
				    std::cout << "Publishing the account "
				       << account.Data() << ".\n";
				    fflush(stdout);
				 }
			      result	= publish_account(record);
			   }
			if (itsftp.Close_Connection_To_Host() == ERROR)
			   {
			      /* Failed to close FTP connection.	*/

			      itserrorinfo	= "Attempted to publish "
						  "an account";
			      itserror		= WebPublishFTPClose;
			      result		= ERROR;
			      Report_Error();
			   }
			else if (itsverboseflag == true)
			   {
			      std::cout << "Connection to " << server.Data()
			         << " has been terminated.\n\n";
			      fflush(stdout);
			   }
		     }

	       }
	 }
      return(result);
   }
