/************************************************************************/
/* File		webpublishactions.cpp					*/
/*									*/
/* Purpose	This C++ program file contains the private WebPublish	*/
/*		member functions that process the command line action	*/
/*		commands. The command line action commands are the	*/
/*		options that operate on the WebPublish database.	*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, July 7, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <string.h>			// C string library.
#include "webpublish.h"			// WebPublish class.

/************************************************************************/
/* Function	status remove_shells(void)				*/
/*									*/
/* Purpose	This function will remove all of the shell programs in	*/
/*		the database that are associated with a given account.	*/
/*									*/
/* Input	This function expects the private WebPublish member	*/
/*		variables to contain the name of the account that is to	*/
/*		have its shell programs removed from the database.	*/
/*									*/
/* Output	If this function is able to remove the shell programs	*/
/*		for a given account then this function will return OK.	*/
/*		If this function is not able to remove the shell	*/
/*		programs for a given account then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status WebPublish::remove_shells(void)
   {
      status		result		= OK;	// Return value
      String		key;			// Database record key.
      String		account;		// Account name.
      DataRecord	record;			// Database record.

      /* Begin by formatting a record and getting the first record in	*/
      /* the database file that contains the shell program records.	*/

      itsdatabase.Format_Record(WebPublishShell, record);
      if (itsdatabase.Get_First_Record(WebPublishShell, record) == ERROR)
         {
	    /* Could not read database file.				*/

	    itserrorinfo	= "Attempted to remove the shell "
				  "program(s) for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      record.Get_Key(key);

      /* While the length of the key for the returned record is not 0,	*/
      /* continue to loop as each shell record is read in and compared.	*/

      while (key.Length() != 0 && result == OK)
         {
	    /* Get the record's account name to look for a match.	*/

	    record.Get_Data(1, account);

	    /* If the record's account name matches the account name	*/
	    /* that was specified on the command line then...		*/

	    if (account.Compare(itsaccount) == EQUAL)
	       {
	          /* Remove the current shell record from the database.	*/

	          if (itsdatabase.Remove_Record(WebPublishShell, record)
		     == ERROR)
		     {
		        /* Could not remove record.			*/

			itserrorinfo	= "Attempted to remove the shell "
					  "program(s) for account ";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadGet;
			result		= ERROR;
			Report_Error();
			break;
		     }
	       }

	    /* Get the next shell program record from the database.	*/

	    if (itsdatabase.Get_Next_Record(WebPublishShell, record)
	       == ERROR)
	       {
		  /* Could not read database file.			*/

		  itserrorinfo	= "Attempted to remove the shell "
				  "program(s) for account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Prepare to enter the loop again.				*/

	    record.Get_Key(key);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status remove_synchronize(void)				*/
/*									*/
/* Purpose	This function will remove all of the paths in the	*/
/*		database that are excluded by the synchronize functions	*/
/*		for a given account.					*/
/*									*/
/* Input	This function expects the private WebPublish member	*/
/*		variables to contain the name of the account that is to	*/
/*		have its excluded synchronize paths removed from the	*/
/*		database.						*/
/*									*/
/* Output	If this function is able to remove the synch paths for	*/
/*		a given account then this function will return OK. If	*/
/*		this function is not able to remove the synch paths for	*/
/*		a given account then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status WebPublish::remove_synchronize(void)
   {
      status		result		= OK;
      String		key;
      String		account;
      DataRecord	record;

      /* Begin by formatting a record and getting the first record in	*/
      /* the database file that contains the synch path records.	*/

      itsdatabase.Format_Record(WebPublishSynch, record);
      if (itsdatabase.Get_First_Record(WebPublishSynch, record) == ERROR)
         {
	    /* Could not read database file.				*/

	    itserrorinfo	= "Attempted to remove the excluded "
				  "synchronized path(s) for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      record.Get_Key(key);

      /* While the length of the key for the returned record is not 0,	*/
      /* continue to loop as each synch path is read in and compared.	*/

      while (key.Length() != 0 && result == OK)
         {
	    /* Get the account name to look for a match.		*/

	    record.Get_Data(1, account);

	    /* If the record's account name matches the account name	*/
	    /* that was specified on the command line then...		*/

	    if (account.Compare(itsaccount) == EQUAL)
	       {
	          /* Remove the current synch path record.		*/

	          if (itsdatabase.Remove_Record(WebPublishSynch, record)
		     == ERROR)
		     {
		        /* Could not remove record.			*/

			itserrorinfo	= "Attempted to remove excluded "
					  "synchronized path(s) for "
					  "account ";
			itserrorinfo	+= itsaccount;
			itserror		= WebPublishBadGet;
			result		= ERROR;
			Report_Error();
			break;
		     }
	       }

	    /* Get the next synch path record from the database.	*/

	    if (itsdatabase.Get_Next_Record(WebPublishSynch, record)
	       == ERROR)
	       {
		  /* Could not read database file.			*/

		  itserrorinfo	= "Attempted to remove excluded "
				  "synchronized path(s) for account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Prepare to enter the loop again.				*/

	    record.Get_Key(key);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status remove_publish(void)				*/
/*									*/
/* Purpose	This function will remove all of the paths in the	*/
/*		database that are excluded by the publish functions	*/
/*		for a given account.					*/
/*									*/
/* Input	This function expects the private WebPublish member	*/
/*		variables to contain the name of the account that is to	*/
/*		have its excluded publish paths removed from the	*/
/*		database.						*/
/*									*/
/* Output	If this function is able to remove the publish paths	*/
/*		for a given account then this function will return OK.	*/
/*		If this function is not able to remove the publish	*/
/*		paths for a given account then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status WebPublish::remove_publish(void)
   {
      status		result		= OK;
      String		key;
      String		account;
      DataRecord	record;

      /* Begin by formatting a record and getting the first record in	*/
      /* the database file that contains the publish path records.	*/

      itsdatabase.Format_Record(WebPublishPublish, record);
      if (itsdatabase.Get_First_Record(WebPublishPublish, record) == ERROR)
         {
	    /* Could not read database file.				*/

	    itserrorinfo	= "Attempted to remove the excluded "
				  "publish path(s) for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      record.Get_Key(key);

      /* While the length of the key for the returned record is not 0,	*/
      /* continue to loop as each publish path is read in and compared.	*/

      while (key.Length() != 0 && result == OK)
         {
	    /* Get the account name to look for a match.		*/

	    record.Get_Data(1, account);

	    /* If the record's account name matches the account name	*/
	    /* that was specified on the command line then...		*/

	    if (account.Compare(itsaccount) == EQUAL)
	       {
	          /* Remove the current publish path record.		*/

	          if (itsdatabase.Remove_Record(WebPublishPublish, record)
		     == ERROR)
		     {
		        /* Could not remove record.			*/

			itserrorinfo	= "Attempted to remove excluded "
					  "publish path(s) for "
					  "account ";
			itserrorinfo	+= itsaccount;
			itserror		= WebPublishBadGet;
			result		= ERROR;
			Report_Error();
			break;
		     }
	       }

	    /* Get the next publish path record from the database.	*/

	    if (itsdatabase.Get_Next_Record(WebPublishPublish, record)
	       == ERROR)
	       {
		  /* Could not read database file.			*/

		  itserrorinfo	= "Attempted to remove excluded "
				  "publish path(s) for account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Prepare to enter the loop again.				*/

	    record.Get_Key(key);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status add_account(void)				*/
/*									*/
/* Purpose	This function will add a new account to the WebPublish	*/
/*		database.						*/
/*									*/
/* Input	This function expects the private WebPublish class	*/
/*		member variables to contain the information that is	*/
/*		required to add a new account to the database. The	*/
/*		private member variables that must contain data are:	*/
/*									*/
/*		Variable	Description				*/
/*									*/
/*		itsaccount	The name of the new account to add.	*/
/*		itswebsite	The path to the local copy of the	*/
/*				website that is associated with the	*/
/*				new account.				*/
/*		itsserver	The name of the FTP server to publish	*/
/*				the new account to.			*/
/*		itsuser		The user name to use when logging the	*/
/*				new account into the FTP server.	*/
/*		itspassword	The password to use when logging the	*/
/*				new account into the FTP server.	*/
/*		itsdirectory	This optional variable can contain a	*/
/*				subdirectory on the server that the	*/
/*				new account's website will be published	*/
/*				to.					*/
/*									*/
/* Output	If this function was able to add the new account to the	*/
/*		database then this function will return OK. If this	*/
/*		function was not able to add the new account to the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::add_account(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure the required information is available.		*/

      if (itsaccount.Length() == 0 || itswebsite.Length() == 0 ||
         itsserver.Length() == 0 || itsuser.Length() == 0 ||
	 itspassword.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to add the account ";
	    itserrorinfo	+= itsaccount;
	    itserrorinfo	+= " to the WebPublish database";
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that this account does not already exist.		*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add the account ";
	    itserrorinfo	+= itsaccount;
	    itserrorinfo	+= " to the WebPublish database";
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does account exist?						*/

      else if (value == true)
         {
	    /* Account already exists.					*/

	    itserrorinfo	= "Attempted to add the account ";
	    itserrorinfo	+= itsaccount;
	    itserrorinfo	+= " to the WebPublish database";
	    itserror		= WebPublishExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the new account to the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Adding " << itsaccount.Data() << " to the "
		     "WebPublish database.\n";
		  fflush(stdout);
	       }

	    /* Create a new database record containing the required	*/
	    /* information.						*/

	    itsdatabase.Format_Record(WebPublishAccounts, record);
	    record.Set_Data(0, itsaccount);
	    record.Set_Data(1, itswebsite);
	    record.Set_Data(2, itsserver);
	    record.Set_Data(3, itsuser);
	    record.Set_Data(4, itspassword);
	    record.Set_Data(5, itsdirectory);

	    /* Add the new record to the WebPublish database.		*/

	    if (itsdatabase.Store_Record(WebPublishAccounts, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to add the account ";
		  itserrorinfo	+= itsaccount;
		  itserrorinfo	+= " to the WebPublish database";
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status change_account(void)				*/
/*									*/
/* Purpose	This function can be used to change any of the data	*/
/*		elements that are part of an account's record in the	*/
/*		WebPublish database. If a relevant variable is included	*/
/*		on the command line, its new value will overwrite the	*/
/*		current value in the account record.			*/
/*									*/
/* Input	This function expects the private WebPublish class	*/
/*		member variables to contain the information that will	*/
/*		overwrite the origianal account record in the		*/
/*		WebPublish database. Any private member variable that	*/
/*		contains data will cause WebPublish to overwrite the	*/
/*		respective variable's data in the account's record. If	*/
/*		the private member variable does not contain data then	*/
/*		the original contents of the respective account record	*/
/*		will remain unchanged. The only exception is the	*/
/*		variable 'itsdirectory'. If 'itsdirectory' is an empty	*/
/*		string then the current value in the account's record	*/
/*		will remain unchanged. Therefore, in order to clear the	*/
/*		directory entry in the account record, the variable	*/
/*		'itsdirectory' must be given the value "/". This will	*/
/*		cause WebPublish to clear the record's account entry.	*/
/*		The private member variables that can contain data are:	*/
/*									*/
/*		Variable	Description				*/
/*									*/
/*		itsaccount	The name of the account to change.	*/
/*		itswebsite	The path to the local copy of the	*/
/*				website that is associated with the	*/
/*				account.				*/
/*		itsserver	The name of the FTP server to publish	*/
/*				the account to.				*/
/*		itsuser		The user name to use when logging the	*/
/*				account into the FTP server.		*/
/*		itspassword	The password to use when logging the	*/
/*				account into the FTP server.		*/
/*		itsdirectory	This optional variable can contain a	*/
/*				subdirectory on the server that the	*/
/*				new account's website will be published	*/
/*				to. To get WebPublish to clear this	*/
/*				variables respective record entry, this	*/
/*				variable must contain the value "/".	*/
/*									*/
/* Output	If this function was able to change the account in the	*/
/*		database then this function will return OK. If this	*/
/*		function was not able to change the account in the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::change_account(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure that an account name has been specified.		*/

      if (itsaccount.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to change account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that the account exists in the database.		*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to change account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist in the database?			*/

      else if (value == false)
         {
	    /* Account does not exist.					*/

	    itserrorinfo	= "Attempted to change account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, change the record's values.				*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Changing " << itsaccount.Data() << std::endl;
		  fflush(stdout);
	       }

	    /* Get the account's record from the database.		*/

	    itsdatabase.Format_Record(WebPublishAccounts, record);
	    record.Set_Data(0, itsaccount);
	    if (itsdatabase.Get_Record(WebPublishAccounts, record)
	       == ERROR)
	       {
	          /* Could not get database record.			*/

		  itserrorinfo	= "Attempted to change account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* If no errors getting account's record then change the	*/
	    /* values that were specified on the command line.		*/

	    else
	       {
	          if (itswebsite.Length() != 0)
		     {
		        record.Set_Data(1, itswebsite);
		     }
	          if (itsserver.Length() != 0)
		     {
		        record.Set_Data(2, itsserver);
		     }
	          if (itsuser.Length() != 0)
		     {
		        record.Set_Data(3, itsuser);
		     }
	          if (itspassword.Length() != 0)
		     {
		        record.Set_Data(4, itspassword);
		     }
		  if (itsdirectory.Compare("/") == EQUAL)
		     {
		        record.Set_Data(5, "");
		     }
	          else if (itsdirectory.Length() != 0)
		     {
		        record.Set_Data(5, itsdirectory);
		     }

		  /* Write the changed record back to the database.	*/

		  if (itsdatabase.Store_Record(WebPublishAccounts, record)
		     == ERROR)
		     {
		        /* Could not store database record.		*/

			itserrorinfo	= "Attempted to change account ";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadStore;
			result		= ERROR;
			Report_Error();
		     }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_account(void)				*/
/*									*/
/* Purpose	This function can be used to remove an account from the	*/
/*		WebPublish database. The private member variable	*/
/*		'itsaccount' must specify the account to remove.	*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that	*/
/*		is to be removed from the database.			*/
/*									*/
/* Output	If this function was able to remove the account from	*/
/*		the database then this function will return OK. If this	*/
/*		function was not able to remove the account from the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::delete_account(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure that the required option was included.		*/

      if (itsaccount.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to remove account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that the record exists in the database.		*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Did the record exist?						*/

      else if (value == false)
         {
	    /* Account does not exist.					*/

	    itserrorinfo	= "Attempted to remove account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, remove the record from the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Removing " << itsaccount.Data() << " from "
		     "the WebPublish database.\n";
		  fflush(stdout);
	       }

	    /* If there are any shell paths associated with this	*/
	    /* account then remove them first.				*/

	    if (remove_shells() == ERROR)
	       {
		  itserrorinfo	= "Attempted to remove account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* If there are any excluded synchronize paths associated	*/
	    /* with this account then remove them first.		*/

	    else if (remove_synchronize() == ERROR)
	       {
		  itserrorinfo	= "Attempted to remove account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* If there are any excluded publish paths associated	*/
	    /* with this account then remove them first.		*/

	    else if (remove_publish() == ERROR)
	       {
		  itserrorinfo	= "Attempted to remove account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* Else, remove the account from the database.		*/

	    else
	       {
	          /* Format a record and set the account's key.		*/

		  itsdatabase.Format_Record(WebPublishAccounts, record);
		  record.Set_Data(0, itsaccount);

		  /* Remove the record from the database.		*/

		  if (itsdatabase.Remove_Record(WebPublishAccounts,
		     record) == ERROR)
		     {
		        /* Could not remove record.			*/

			itserrorinfo	= "Attempted to remove account ";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadDel;
			result	= ERROR;
			Report_Error();
		     }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status list_account(void)				*/
/*									*/
/* Purpose	This function can be used to list a specific account	*/
/*		to stdout, or this function can be used to list all of	*/
/*		the accounts to stdout. If the private member variable	*/
/*		'itsaccount' contains an account name then only the	*/
/*		specified account will be listed to stdout. If		*/
/*		'itsaccount' is an empty string then this function will	*/
/*		list all of the accounts to stdout.			*/
/*									*/
/* Input	If the private WebPublish member variable 'itsaccount'	*/
/*		contains an account name then this function will only	*/
/*		list the specified account to stdout. If 'itsaccount'	*/
/*		is an empty string then all of the accounts will be	*/
/*		listed to stdout.					*/
/*									*/
/* Output	If this function was able to list a specific account,	*/
/*		or all of the accounts in the database then this	*/
/*		function will return OK. If this function is not able	*/
/*		to list a specific account, or all of the accounts that	*/
/*		are in the database then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status WebPublish::list_account(void)
   {
      status		result		= OK;
      String		key;
      String		website;
      String		server;
      String		user;
      String		password;
      String		directory;
      DataRecord	record;
      condition		value;

      /* Format a record in case we need it.				*/

      itsdatabase.Format_Record(WebPublishAccounts, record);

      /* If 'itsaccount' is not an empty string then only list the	*/
      /* specified account to stdout.					*/

      if (itsaccount.Length() != 0)
         {
	    /* Set the account's key and find out if the record exists.	*/

	    record.Set_Data(0, itsaccount);
            if (does_record_exist(WebPublishAccounts, itsaccount, value)
               == ERROR)
               {
	          /* Error searching for record.			*/

	          itserrorinfo	= "Attempted to list the accounts in "
				  "the WebPublish database";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Does the record exist?					*/

            else if (value == false)
               {
	          /* Account does not exist.				*/

	          itserrorinfo	= "Attempted to list the accounts in "
				  "the WebPublish database";
	          itserror	= WebPublishNoExist;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Get the account's record from the database.		*/

	    if (itsdatabase.Get_Record(WebPublishAccounts, record)
	       == ERROR)
	       {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list the accounts in "
				  "the WebPublish database";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
	    else
	       {
	          record.Get_Key(key);
		  if (key.Length() == 0)
		     {
			/* Account does not exist.			*/

			itserrorinfo	= "Attempted to list an account ";
			itserror	= WebPublishNoExist;
			result		= ERROR;
			Report_Error();

		     }
		  else
		     {
		        /* Write the account's data to stdout.		*/

		        record.Get_Data(1, website);
		        record.Get_Data(2, server);
		        record.Get_Data(3, user);
		        record.Get_Data(4, password);
		        record.Get_Data(5, directory);
			std::cout << "Listing WebPublish account:\n";
		        std::cout << "Account  : " << itsaccount.Data()
			   << std::endl;
		        std::cout << "Website  : " << website.Data() << std::endl;
		        std::cout << "Server   : " << server.Data() << std::endl;
		        std::cout << "User     : " << user.Data() << std::endl;
		        std::cout << "Password : " << password.Data() << std::endl;
		        std::cout << "Directory: " << directory.Data() << std::endl;
		     }
	       }
	 }

      /* Else, list all of the accounts in the database.		*/

      else
         {
	    /* Get the first account in the database.			*/

            if (itsdatabase.Get_First_Record(WebPublishAccounts, record)
               == ERROR)
               {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list the accounts in "
				  "the WebPublish database";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the returned record's key is an empty string...	*/

            record.Get_Key(key);
            if (key.Length() == 0)
               {
	          std::cout << "There are no accounts in the database.\n";
	       }
            else
               {
	          std::cout << "Listing accounts:\n";
	       }

	    /* This loop will continue as long as records are returned	*/
	    /* by the database functions.				*/

            while (key.Length() != 0 && result == OK)
               {
	          /* Write the account's data to stdout.		*/

	          record.Get_Data(1, website);
	          record.Get_Data(2, server);
	          record.Get_Data(3, user);
	          record.Get_Data(4, password);
	          record.Get_Data(5, directory);
	          std::cout << "\nAccount  : " << key.Data() << std::endl;
	          std::cout << "Website  : " << website.Data() << std::endl;
	          std::cout << "Server   : " << server.Data() << std::endl;
	          std::cout << "User     : " << user.Data() << std::endl;
	          std::cout << "Password : " << password.Data() << std::endl;
	          std::cout << "Directory: " << directory.Data() << std::endl;

		  /* Get the next account from the database.		*/

	          if (itsdatabase.Get_Next_Record(WebPublishAccounts,
		     record) == ERROR)
	             {
	                /* Could not get record.			*/

		        itserrorinfo	= "Attempted to list the "
					  "accounts in the WebPublish "
					  "database";
		        itserror	= WebPublishBadGet;
		        result	= ERROR;
		        Report_Error();
	             }

		  /* Prepare to enter the loop again.			*/
		  
	          record.Get_Key(key);
	       }
         }
      std::cout << std::endl;
      fflush(stdout);
      return(result);
   }

/************************************************************************/
/* Function	status add_shell(void)					*/
/*									*/
/* Purpose	This function will add a shell program to the database.	*/
/*		The shell program must include an account name and a	*/
/*		file extension to associate the shell program with.	*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that	*/
/*		is to have a shell program added. The private member	*/
/*		variable 'itsextension' must contain the file extension	*/
/*		that the shell program is to be associated with.	*/
/*		Finally, the private member variable 'itspath' must	*/
/*		contain the path to use when executing the shell	*/
/*		program.						*/
/*									*/
/* Output	If this function was able to add the shell program to	*/
/*		the database then this function will return OK. If this	*/
/*		function was not able to add the shell program to the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::add_shell(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the shell program key (account name + file extension).	*/

      key		= itsaccount;
      key		+= itsextension;

      /* Make sure that all of the required information was included.	*/

      if (itsaccount.Length() == 0 || itsextension.Length() == 0 ||
	 itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the account exist in the database.			*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The account does not exist.				*/

	    itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the shell program already exists in the database.	*/

      else if (does_record_exist(WebPublishShell, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the shell program already exist?				*/

      else if (value == true)
         {
	    /* Account already exists.					*/

	    itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the shell program to the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Adding " << itsextension.Data() << " to the "
		     "WebPublish database.\n";
	       }
	    itsdatabase.Format_Record(WebPublishShell, record);
	    record.Set_Data(0, key);
	    record.Set_Data(1, itsaccount);
	    record.Set_Data(2, itsextension);
	    record.Set_Data(3, itspath);
	    if (itsdatabase.Store_Record(WebPublishShell, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to add a shell program to "
				  "account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status change_shell(void)				*/
/*									*/
/* Purpose	This function can be used to change the shell program	*/
/*		that is associated with a file extension for a given	*/
/*		account.						*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that is	*/
/*		to have a shell program changed. The private member	*/
/*		variable 'itsextension' must contain the extension that	*/
/*		the shell program is associated with. Finally, the	*/
/*		private member variable 'itspath' must contain the	*/
/*		path to use when executing the shell program.		*/
/*									*/
/* Output	If this function is able to change the shell program	*/
/*		that is associated with a file extension for a given	*/
/*		account then this function will return OK. If this	*/
/*		function was not able to change the shell program for	*/
/*		an account then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::change_shell(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the shell program key (account name + file extension).	*/

      key		= itsaccount;
      key		+= itsextension;

      /* Make sure that all of the required information was included.	*/

      if (itsaccount.Length() == 0 || itsextension.Length() == 0 ||
	 itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the account exist in the database.			*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The account does not exist.				*/

	    itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the shell program already exists in the database.	*/

      else if (does_record_exist(WebPublishShell, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the shell program already exist?				*/

      else if (value == false)
         {
	    /* Account does not exist.					*/

	    itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the shell program to the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Changing shell program associated with "
		     << itsextension.Data() << std::endl;
	       }

	    /* Get the shell program's record from the database.		*/

	    itsdatabase.Format_Record(WebPublishShell, record);
	    record.Set_Data(0, key);
	    if (itsdatabase.Get_Record(WebPublishShell, record)
	       == ERROR)
	       {
	          /* Could not get database record.			*/

		  itserrorinfo	= "Attempted to change a shell program "
				  "for account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* If no errors getting shell program's record then change	*/
	    /* the shell program to the value that was specified on the	*/
	    /* command line.						*/

	    else
	       {
		  record.Set_Data(3, itspath);

		  /* Write the record back to the database.		*/

		  if (itsdatabase.Store_Record(WebPublishShell, record)
		     == ERROR)
		     {
		        /* Could not store database record.		*/

			itserrorinfo	= "Attempted to change a shell "
					  "program for account ";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadStore;
			result		= ERROR;
			Report_Error();
		     }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_shell(void)				*/
/*									*/
/* Purpose	This function can be used to remove a shell program	*/
/*		that is associated with a file extension for a given	*/
/*		account from the WebPublish database.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that is	*/
/*		to have a shell program removed. The private member	*/
/*		variable 'itsextension' must contain the extension that	*/
/*		the shell program is associated with.			*/
/*									*/
/* Output	If this function was able to remove the shell program	*/
/*		from the database then this function will return OK. If	*/
/*		this function was not able to remove the shell program	*/
/*		from the database then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status WebPublish::delete_shell(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the shell program key (account name + file extension).	*/

      key		= itsaccount;
      key		+= itsextension;

      /* Make sure that all of the required information was included.	*/

      if (itsaccount.Length() == 0 || itsextension.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that the account exists.				*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The account does not exist.				*/

	    itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* Make sure that the shell program exists in the database.	*/

      else if (does_record_exist(WebPublishShell, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the shell program exist in the database?			*/

      else if (value == false)
         {
	    /* Shell program does not exist.				*/

	    itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, remove the shell program from the database.		*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Removing " << itsextension.Data() << " from "
		     "the WebPublish database.\n";
	       }

	    /* Format a record and set the shell program's key.		*/

	    itsdatabase.Format_Record(WebPublishShell, record);
	    record.Set_Data(0, key);

	    /* Remove the shell program from the database.		*/

	    if (itsdatabase.Remove_Record(WebPublishShell, record)
	       == ERROR)
	       {
	          /* Could not remove record.				*/

		  itserrorinfo	= "Attempted to remove a shell program "
				  "from the account";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status list_shell(void)					*/
/*									*/
/* Purpose	This function can be used to list all of the shell	*/
/*		programs for a specific account to stdout, or this	*/
/*		function can be used to list all of the shell programs	*/
/*		that are in the WebPublish database to stdout. If the	*/
/*		private member variable 'itsaccount' contains an	*/
/*		account name then only shell programs for the specified	*/
/*		account will be	* listed to stdout. If 'itsaccount' is	*/
/*		an empty string then this function will list all of the	*/
/*		shell programs in the WebPublish database to stdout.	*/
/*									*/
/* Input	If the private WebPublish member variable 'itsaccount'	*/
/*		contains an account name then this function will list	*/
/*		all of the shell programs for the specified account to	*/
/*		stdout. If 'itsaccount' is an empty string then all of	*/
/*		the shell programs in the WebPublish database will be	*/
/*		listed to stdout.					*/
/*									*/
/* Output	If this function was able to list the shell programs	*/
/*		for a specific account, or all of the shell programs in	*/
/*		the database then this function will return OK. If this	*/
/*		function is not able to list all of the shell programs	*/
/*		for a specific account, or all of the shell programs in	*/
/*		the WebPublish database then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status WebPublish::list_shell(void)
   {
      status		result		= OK;
      String		key;
      String		account;
      String		extension;
      String		shell;
      DataRecord	record;
      condition		value;

      /* Format a record in case we need it.				*/

      itsdatabase.Format_Record(WebPublishShell, record);

      /* See if an account name was specified.				*/

      if (itsaccount.Length() != 0)
         {
            if (does_record_exist(WebPublishAccounts, itsaccount, value)
               == ERROR)
	       {
	          /* Error searching for record.			*/

	          itserrorinfo	= "Attempted to list the shell programs "
				  " in the WebPublish database";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else if (value == false)
               {
	          /* The account does not exist.			*/

	          itserrorinfo	= "Attempted to list the shell programs "
				  " in the WebPublish database";
	          itserror	= WebPublishBadAccount;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      if (result == OK)
         {
            if (itsdatabase.Get_First_Record(WebPublishShell, record)
               == ERROR)
               {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list the shell programs "
				  " in the WebPublish database";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else
               {
                  record.Get_Key(key);
		  if (itsaccount.Length() == 0)
		     {
		        std::cout << "Listing all shell programs:\n";
		     }
		  else
		     {
                        std::cout << "Listing shell programs for " <<
	                   itsaccount.Data() << ":\n";
		     }
	       }
	 }
      while (key.Length() != 0 && result == OK)
         {
	    if (itsaccount.Length() != 0)
	       {
	          record.Get_Data(1, account);
	          if (itsaccount.Compare(account) == EQUAL)
	             {
	                record.Get_Data(2, extension);
		        record.Get_Data(3, shell);
		        std::cout << "\nExtension: " << extension.Data()
			   << std::endl;
		        std::cout << "Program  : " << shell.Data() << std::endl;
	             }
	       }
	    else
	       {
	          record.Get_Data(1, account);
	          record.Get_Data(2, extension);
		  record.Get_Data(3, shell);
		  std::cout << "\nAccount  : " << account.Data() << std::endl;
		  std::cout << "Extension: " << extension.Data() << std::endl;
		  std::cout << "Program  : " << shell.Data() << std::endl;
	       }
	    if (itsdatabase.Get_Next_Record(WebPublishShell, record)
	       == ERROR)
	       {
	          /* Could not get record.				*/

		  itserrorinfo	= "Attempted to list the shell programs "
				  " in the WebPublish database";
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }
	    record.Get_Key(key);
	 }
      std::cout << std::endl;
      fflush(stdout);
      return(result);
   }

/************************************************************************/
/* Function	status add_shell(void)					*/
/*									*/
/* Purpose	This function will add a server file or directory path	*/
/*		to the database. The specified path will be ignored by	*/
/*		WebPublish when WebPublish is searching for files and	*/
/*		directories to remove from the server.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that	*/
/*		is to have a server path added. The private member	*/
/*		variable 'itspath' must contain the server path that	*/
/*		is to be added to the database.				*/
/*									*/
/* Output	If this function was able to add the server path to the	*/
/*		database then this function will return OK. If this	*/
/*		function was not able to add the server path to the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::add_synch(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the key (account name + server path).			*/

      key		= itsaccount;
      key		+= itspath;

      /* Make sure the required information was included.		*/

      if (itsaccount.Length() == 0 || itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the specified account exists in the database.		*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The account does not exist.				*/

	    itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the specified server path already exists.		*/

      else if (does_record_exist(WebPublishSynch, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the server path exist?					*/

      else if (value == true)
         {
	    /* Record already exists.					*/

	    itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the server path to the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Adding " << itspath.Data() << " to the "
		     "WebPublish database.\n";
	       }

	    /* Format a record and set the record's data.		*/

	    itsdatabase.Format_Record(WebPublishSynch, record);
	    record.Set_Data(0, key);
	    record.Set_Data(1, itsaccount);
	    record.Set_Data(2, itspath);

	    /* Store the record in the database.			*/

	    if (itsdatabase.Store_Record(WebPublishSynch, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to add a server path to "
				  "the account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_synch(void)				*/
/*									*/
/* Purpose	This function can be used to remove a server path that	*/
/*		is ignored by WebPublish when WebPublish is searching	*/
/*		for files and directories to remove on the server.	*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that is	*/
/*		to have a server path removed. The private member	*/
/*		variable 'itspath' must contain the server path that	*/
/*		is to be removed from the database.			*/
/*									*/
/* Output	If this function was able to remove the server path	*/
/*		from the database then this function will return OK. If	*/
/*		this function was not able to remove the server path	*/
/*		from the database then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status WebPublish::delete_synch(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the key (account name + local path).			*/

      key		= itsaccount;
      key		+= itspath;

      /* Make sure that the required information was included.		*/

      if (itsaccount.Length() == 0 || itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the account exists in the database.			*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The record does not exist.				*/

	    itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the specified path exists in the database.		*/

      else if (does_record_exist(WebPublishSynch, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the path exist?						*/

      else if (value == false)
         {
	    /* Account does not exist.					*/

	    itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, remove the specified path from the database.		*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Removing " << itspath.Data() << " from "
		     "the WebPublish database.\n";
	       }

	    /* Format a record and set the record's key.		*/

	    itsdatabase.Format_Record(WebPublishSynch, record);
	    record.Set_Data(0, key);

	    /* Remove the record from the database.			*/

	    if (itsdatabase.Remove_Record(WebPublishSynch, record)
	       == ERROR)
	       {
	          /* Could not remove record.				*/

		  itserrorinfo	= "Attempted to remove a server path "
				  "from account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status list_synch(void)					*/
/*									*/
/* Purpose	This function can be used to list all of the server	*/
/*		paths for a given account to stdout, or this function	*/
/*		can be used to list all of the server paths in the	*/
/*		WebPublish database. The server paths are the paths	*/
/*		that are ignored by WebPublish when WebPublish is	*/
/*		searching for files and directories to be removed from	*/
/*		the server. If the private member variable 'itsaccount'	*/
/*		contains an account name then only server paths for the	*/
/*		specified account will be listed to stdout. If		*/
/*		'itsaccount' is an empty string then this function will	*/
/*		list all of the server paths in the WebPublish database	*/
/*		to stdout.						*/
/*									*/
/* Input	If the private WebPublish member variable 'itsaccount'	*/
/*		contains an account name then this function will list	*/
/*		all of the server paths for the specified account to	*/
/*		stdout. If 'itsaccount' is an empty string then all of	*/
/*		the server paths in the WebPublish database will be	*/
/*		listed to stdout.					*/
/*									*/
/* Output	If this function was able to list the server paths	*/
/*		for a specific account, or all of the server paths in	*/
/*		the database then this function will return OK. If this	*/
/*		function is not able to list all of the server paths	*/
/*		for a specific account, or all of the server paths in	*/
/*		the WebPublish database then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status WebPublish::list_synch(void)
   {
      status		result		= OK;
      String		key;
      String		account;
      String		path;
      DataRecord	record;
      condition		value;

      /* Format a record in case we need it.				*/

      itsdatabase.Format_Record(WebPublishSynch, record);

      /* See if an account name was specified.				*/

      if (itsaccount.Length() != 0)
         {
            if (does_record_exist(WebPublishAccounts, itsaccount, value)
               == ERROR)
	       {
	          /* Error searching for record.			*/

	          itserrorinfo	= "Attempted to list the server paths "
				  "in account ";
		  itserrorinfo	+= itsaccount;
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else if (value == false)
               {
	          /* The account does not exist.			*/

	          itserrorinfo	= "Attempted to list the server paths "
				  "in account ";
		  itserrorinfo	+= itsaccount;
	          itserror	= WebPublishBadAccount;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      if (result == OK)
         {
            if (itsdatabase.Get_First_Record(WebPublishSynch, record)
               == ERROR)
               {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list server paths";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else
               {
                  record.Get_Key(key);
		  if (itsaccount.Length() == 0)
		     {
		        std::cout << "Listing all server paths:\n";
		     }
		  else
		     {
                        std::cout << "Listing server paths for " <<
	                   itsaccount.Data() << ":\n\n";
		     }
	       }
	 }
      while (key.Length() != 0 && result == OK)
         {
	    if (itsaccount.Length() != 0)
	       {
	          record.Get_Data(1, account);
	          if (itsaccount.Compare(account) == EQUAL)
	             {
		        record.Get_Data(2, path);
			std::cout << "Path: " << path.Data() << std::endl;
	             }
	       }
	    else
	       {
	          record.Get_Data(1, account);
	          record.Get_Data(2, path);
		  std::cout << "\nAccount  : " << account.Data() << std::endl;
		  std::cout << "Path     : " << path.Data() << std::endl;
	       }
	    if (itsdatabase.Get_Next_Record(WebPublishSynch, record)
	       == ERROR)
	       {
	          /* Could not get record.				*/

		  itserrorinfo	= "Attempted to list server paths";
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }
	    record.Get_Key(key);
	 }
      std::cout << std::endl;
      fflush(stdout);
      return(result);
   }

/************************************************************************/
/* Function	status add_publish(void)				*/
/*									*/
/* Purpose	This function will add a local file or directory path	*/
/*		to the database. The specified path will be ignored by	*/
/*		WebPublish when WebPublish is searching for files and	*/
/*		directories to transfer to the server.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that	*/
/*		is to have a local path path added. The private member	*/
/*		variable 'itspath' must contain the local path that	*/
/*		is to be added to the database.				*/
/*									*/
/* Output	If this function was able to add the local path to the	*/
/*		database then this function will return OK. If this	*/
/*		function was not able to add the local path to the	*/
/*		database then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status WebPublish::add_publish(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the key (account name + local path).			*/

      key		= itsaccount;
      key		+= itspath;

      /* Make sure that the required information was included.		*/

      if (itsaccount.Length() == 0 || itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to add a local path to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the account exists in the database.			*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a local path to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The account does not exist.				*/

	    itserrorinfo	= "Attempted to add a local path to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the local path exists in the database.			*/

      else if (does_record_exist(WebPublishPublish, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add a local path to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the local path exist?					*/

      else if (value == true)
         {
	    /* Record already exists.					*/

	    itserrorinfo	= "Attempted to add a local path to "
				  "account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the local path to the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Adding " << itspath.Data() << " to the "
		     "WebPublish database.\n";
	       }

	    /* Format a record and set the record's data.		*/

	    itsdatabase.Format_Record(WebPublishPublish, record);
	    record.Set_Data(0, key);
	    record.Set_Data(1, itsaccount);
	    record.Set_Data(2, itspath);

	    /* Add the record to the database.				*/

	    if (itsdatabase.Store_Record(WebPublishPublish, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to add a local path to "
				  "account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_publish(void)				*/
/*									*/
/* Purpose	This function can be used to remove a local path that	*/
/*		is ignored by WebPublish when WebPublish is searching	*/
/*		for files and directories to transfer to the server.	*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsaccount' to contain the name of the account that is	*/
/*		to have a local path removed. The private member	*/
/*		variable 'itspath' must contain the local path that	*/
/*		is to be removed from the database.			*/
/*									*/
/* Output	If this function was able to remove the local path	*/
/*		from the database then this function will return OK. If	*/
/*		this function was not able to remove the local path	*/
/*		from the database then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status WebPublish::delete_publish(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		key;
      condition		value;

      /* Build the key (account name + local path).			*/

      key		= itsaccount;
      key		+= itspath;

      /* Make sure that the required information was included.		*/

      if (itsaccount.Length() == 0 || itspath.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the account exists in the database.			*/

      else if (does_record_exist(WebPublishAccounts, itsaccount, value)
         == ERROR)
	 {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the account exist?					*/

      else if (value == false)
         {
	    /* The record does not exist.				*/

	    itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadAccount;
	    result		= ERROR;
	    Report_Error();

	 }

      /* See if the specified path exists in the database.		*/

      else if (does_record_exist(WebPublishPublish, key, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the path exist?						*/

      else if (value == false)
         {
	    /* Account does not exist.					*/

	    itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, remove the local path from the database.			*/
      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Removing " << itspath.Data() << " from "
		     "the WebPublish database.\n";
	       }

	    /* Format a record and set the record's key.		*/

	    itsdatabase.Format_Record(WebPublishPublish, record);
	    record.Set_Data(0, key);

	    /* Remove the record from the database.			*/

	    if (itsdatabase.Remove_Record(WebPublishPublish, record)
	       == ERROR)
	       {
	          /* Could not remove record.				*/

		  itserrorinfo	= "Attempted to remove a local path "
				  "from account ";
		  itserrorinfo	+= itsaccount;
		  itserror	= WebPublishBadDel;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status list_publish(void)				*/
/*									*/
/* Purpose	This function can be used to list all of the local	*/
/*		paths for a given account to stdout, or this function	*/
/*		can be used to list all of the local paths in the	*/
/*		WebPublish database. The local paths are the paths	*/
/*		that are ignored by WebPublish when WebPublish is	*/
/*		searching for files and directories to transfer to the	*/
/*		server. If the private member variable 'itsaccount'	*/
/*		contains an account name then only local paths for the	*/
/*		specified account will be listed to stdout. If		*/
/*		'itsaccount' is an empty string then this function will	*/
/*		list all of the local paths in the WebPublish database	*/
/*		to stdout.						*/
/*									*/
/* Input	If the private WebPublish member variable 'itsaccount'	*/
/*		contains an account name then this function will list	*/
/*		all of the local paths for the specified account to	*/
/*		stdout. If 'itsaccount' is an empty string then all of	*/
/*		the local paths in the WebPublish database will be	*/
/*		listed to stdout.					*/
/*									*/
/* Output	If this function was able to list the local paths	*/
/*		for a specific account, or all of the local paths in	*/
/*		the database then this function will return OK. If this	*/
/*		function is not able to list all of the local paths	*/
/*		for a specific account, or all of the local paths in	*/
/*		the WebPublish database then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status WebPublish::list_publish(void)
   {
      status		result		= OK;
      String		key;
      String		account;
      String		path;
      DataRecord	record;
      condition		value;

      /* Format a record in case we need it.				*/

      itsdatabase.Format_Record(WebPublishPublish, record);

      /* See if an account name was specified.				*/

      if (itsaccount.Length() != 0)
         {
            if (does_record_exist(WebPublishAccounts, itsaccount, value)
               == ERROR)
	       {
	          /* Error searching for record.			*/

	          itserrorinfo	= "Attempted to list the local paths "
				  "in account ";
		  itserrorinfo	+= itsaccount;
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else if (value == false)
               {
	          /* The account does not exist.			*/

	          itserrorinfo	= "Attempted to list the local paths "
				  "in account ";
		  itserrorinfo	+= itsaccount;
	          itserror	= WebPublishBadAccount;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      if (result == OK)
         {
            if (itsdatabase.Get_First_Record(WebPublishPublish, record)
               == ERROR)
               {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list local paths";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else
               {
                  record.Get_Key(key);
		  if (itsaccount.Length() == 0)
		     {
		        std::cout << "Listing all local paths:\n";
		     }
		  else
		     {
                        std::cout << "Listing local paths for " <<
	                   itsaccount.Data() << ":\n\n";
		     }
	       }
	 }
      while (key.Length() != 0 && result == OK)
         {
	    if (itsaccount.Length() != 0)
	       {
	          record.Get_Data(1, account);
	          if (itsaccount.Compare(account) == EQUAL)
	             {
		        record.Get_Data(2, path);
			std::cout << "Path: " << path.Data() << std::endl;
	             }
	       }
	    else
	       {
	          record.Get_Data(1, account);
	          record.Get_Data(2, path);
		  std::cout << "\nAccount  : " << account.Data() << std::endl;
		  std::cout << "Path     : " << path.Data() << std::endl;
	       }
	    if (itsdatabase.Get_Next_Record(WebPublishPublish, record)
	       == ERROR)
	       {
	          /* Could not get record.				*/

		  itserrorinfo	= "Attempted to list local paths";
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }
	    record.Get_Key(key);
	 }
      std::cout << std::endl;
      fflush(stdout);
      return(result);
   }

/************************************************************************/
/* Function	status add_mode(void)					*/
/*									*/
/* Purpose	This function can be used to associate a file transfer	*/
/*		mode (text or binary) with a file extension. The	*/
/*		transfer mode will override the default WebPublish	*/
/*		file transfer mode for all accounts.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsextension' to contain the extension that is to have	*/
/*		a file transfer mode associated with it. The private	*/
/*		member variable 'itsmode' must contain either FTPText	*/
/*		or FTPBinary.						*/
/*									*/
/* Output	If this function is able to add a record that		*/
/*		associates a file transfer mode with a file extension	*/
/*		then this function will return OK. If this function	*/
/*		was not able to associate a file transfer mode with a	*/
/*		file extension then this function will return ERROR.	*/
/*		All errors by this function are reported to stdout.	*/
/************************************************************************/

status WebPublish::add_mode(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure that the required information was included.		*/

      if (itsextension.Length() == 0 || itsmodetext.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to add file transfer mode";
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the extension exists in the database.			*/

      else if (does_record_exist(WebPublishModes, itsextension, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to add ";
	    itserrorinfo	+= (itsmode == FTPText ? "TEXT" :
				   "BINARY");
	    itserrorinfo	+= " mode with ";
	    itserrorinfo	+= itsextension;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the extension exist?					*/

      else if (value == true)
         {
	    /* Record already exists.					*/

	    itserrorinfo	= "Attempted to add ";
	    itserrorinfo	+= (itsmode == FTPText ? "TEXT" :
				   "BINARY");
	    itserrorinfo	+= " mode with ";
	    itserrorinfo	+= itsextension;
	    itserror		= WebPublishExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, add the file transfer mode to the database.		*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Adding " <<
		     (itsmode == FTPText ? "TEXT" : "BINARY") <<
		     " mode with extension " << itsextension.Data()
		     << std::endl;
	       }

	    /* Format a record and set the record's data.		*/

	    itsdatabase.Format_Record(WebPublishModes, record);
	    record.Set_Data(0, itsextension);
	    record.Set_Data(1, itsmode);

	    /* Add the record to the database.				*/

	    if (itsdatabase.Store_Record(WebPublishModes, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to add ";
		  itserrorinfo	+= (itsmode == FTPText ?
				   "TEXT" : "BINARY");
		  itserrorinfo	+= " mode with ";
		  itserrorinfo	+= itsextension;
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status change_mode(void)				*/
/*									*/
/* Purpose	This function can be used to change a file transfer	*/
/*		mode (text or binary) that is associated with a file	*/
/*		extension. The transfer mode will override the default	*/
/*		WebPublish file transfer mode for all accounts.		*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsextension' to contain the extension that is to have	*/
/*		a file transfer mode associated with it. The private	*/
/*		member variable 'itsmode' must contain either FTPText	*/
/*		or FTPBinary.						*/
/*									*/
/* Output	If this function is able to change a record that	*/
/*		associates a file transfer mode with a file extension	*/
/*		then this function will return OK. If this function	*/
/*		was not able to change a file transfer mode then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stdout.					*/
/************************************************************************/

status WebPublish::change_mode(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure that the required information was included.		*/

      if (itsextension.Length() == 0 || itsmodetext.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to change file "
				  "transfer mode";
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the extension exists in the database.			*/

      else if (does_record_exist(WebPublishModes, itsextension, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to change ";
	    itserrorinfo	+= itsextension;
	    itserrorinfo	+= " to ";
	    itserrorinfo	+= (itsmode == FTPText ? "TEXT" :
				   "BINARY");
	    itserrorinfo	+= " mode";
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the extension exist?					*/

      else if (value == false)
         {
	    /* Record does not exist.					*/

	    itserrorinfo	= "Attempted to change ";
	    itserrorinfo	+= itsextension;
	    itserrorinfo	+= " to ";
	    itserrorinfo	+= (itsmode == FTPText ? "TEXT" :
				   "BINARY");
	    itserrorinfo	+= " mode";
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, change the transfer mode in the database.		*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Changing " << itsextension.Data() <<
		     " to " << (itsmode == FTPText ? "TEXT" : "BINARY")
		     << " mode\n";
	       }

	    /* Format a record and set the record's data.		*/

	    itsdatabase.Format_Record(WebPublishModes, record);
	    record.Set_Data(0, itsextension);
	    record.Set_Data(1, itsmode);

	    /* Change the record in the database.			*/

	    if (itsdatabase.Store_Record(WebPublishModes, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to change ";
		  itserrorinfo	+= itsextension;
		  itserrorinfo	+= " to ";
		  itserrorinfo	+= (itsmode == FTPText ? "TEXT" :
				   "BINARY");
		  itserrorinfo	+= " mode";
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_mode(void)				*/
/*									*/
/* Purpose	This function can be used to remove a file transfer	*/
/*		mode (text or binary) that is associated with a file	*/
/*		extension.						*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsextension' to contain the extension that is to be	*/
/*		removed from the database.				*/
/*									*/
/* Output	If this function is able to remove a record that	*/
/*		associates a file transfer mode with a file extension	*/
/*		then this function will return OK. If this function	*/
/*		was not able to remove a file transfer mode then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stdout.					*/
/************************************************************************/

status WebPublish::delete_mode(void)
   {
      status		result		= OK;
      DataRecord	record;
      condition		value;

      /* Make sure that the required information was included.		*/

      if (itsextension.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to remove a file "
				  "transfer mode";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the extension exists in the database.			*/

      else if (does_record_exist(WebPublishModes, itsextension, value)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to remove file transfer "
				  "mode for ";
	    itserrorinfo	+= itsextension;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Does the extension exist?					*/

      else if (value == false)
         {
	    /* Record does not exist.					*/

	    itserrorinfo	= "Attempted to remove file transfer "
				  "mode for ";
	    itserrorinfo	+= itsextension;
	    itserror		= WebPublishNoExist;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Else, remove the file transfer mode from the database.			*/

      else
         {
	    /* If -V or --verbose was included on the command line...	*/

	    if (itsverboseflag == true)
	       {
	          std::cout << "Removing file transfer mode for "
		     << itsextension.Data() << std::endl;
	       }

	    /* Format a record and set the record's data.		*/

	    itsdatabase.Format_Record(WebPublishModes, record);
	    record.Set_Data(0, itsextension);

	    /* Change the record in the database.			*/

	    if (itsdatabase.Remove_Record(WebPublishModes, record)
	       == ERROR)
	       {
	          /* Could not add record.				*/

		  itserrorinfo	= "Attempted to remove file transfer "
				  "mode for ";
		  itserrorinfo	+= itsextension;
		  itserror	= WebPublishBadStore;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status list_mode(void)					*/
/*									*/
/* Purpose	This function can be used to list the file transfer	*/
/*		mode for a specific file extension, or this function	*/
/*		can list all of the file transfer modes in the		*/
/*		database. If an extension is specified then the file	*/
/*		transfer mode for the given file extension will be	*/
/*		listed. If no extension is specified then all of the	*/
/*		file transfer modes will be listed.			*/
/*									*/
/* Input	This function expects the private member variable	*/
/*		'itsextension' to contain the extension that is to be	*/
/*		listed. If 'itsextension' is an empty string then all	*/
/*		of the file transfer modes will be listed.		*/
/*									*/
/* Output	If this function is able to list a specific file	*/
/*		transfer mode, or if this function is able to list all	*/
/*		of the file transfer modes then this function will	*/
/*		return OK. If this function was not able to list a	*/
/*		specific file transfer mode, or all of the file		*/
/*		transfer modes then this function will return ERROR.	*/
/*		All errors by this function are reported to stdout.	*/
/************************************************************************/

status WebPublish::list_mode(void)
   {
      status		result		= OK;
      int		tempmode;
      String		extension;
      FTPMode		mode;
      DataRecord	record;
      condition		value;

      /* Format a record in case we need it.				*/

      itsdatabase.Format_Record(WebPublishModes, record);

      /* See if the extension was specified.				*/

      if (itsextension.Length() != 0)
         {
            if (does_record_exist(WebPublishModes, itsextension, value)
               == ERROR)
	       {
	          /* Error searching for record.			*/

	          itserrorinfo	= "Attempted to list the file transfer "
				  "mode for ";
		  itserrorinfo	+= itsextension;
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else if (value == false)
               {
	          /* The account does not exist.			*/


	          itserrorinfo	= "Attempted to list the file transfer "
				  "mode for ";
		  itserrorinfo	+= itsextension;
	          itserror	= WebPublishBadAccount;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      if (result == OK)
         {
            if (itsdatabase.Get_First_Record(WebPublishModes, record)
               == ERROR)
               {
	          /* Could not get record.				*/

	          itserrorinfo	= "Attempted to list file transfer modes";
	          itserror	= WebPublishBadGet;
	          result	= ERROR;
	          Report_Error();
	       }
            else
               {
                  record.Get_Key(extension);
		  if (itsextension.Length() == 0)
		     {
		        std::cout << "Listing all file transfer modes:\n";
		     }
		  else
		     {
                        std::cout << "Listing file transfer mode for " <<
	                   extension.Data() << ":\n\n";
		     }
	       }
	 }
      while (extension.Length() != 0 && result == OK)
         {
	    if (itsextension.Length() != 0)
	       {
	          if (itsextension.Compare(extension) == EQUAL)
	             {
		        record.Get_Data(1, tempmode);
		        mode		= (FTPMode)tempmode;
			std::cout << "Extension: " << extension.Data() << std::endl;
			std::cout << "Mode     : " <<
			   (mode == FTPText ? "TEXT" : "BINARY") << std::endl;
			break;
	             }
	       }
	    else
	       {
	          record.Get_Data(1, tempmode);
	          mode		= (FTPMode)tempmode;
		  std::cout << "\nExtension: " << extension.Data() << std::endl;
		  std::cout << "Mode     : " <<
		     (mode == FTPText ? "TEXT" : "BINARY") << std::endl;
	       }
	    if (itsdatabase.Get_Next_Record(WebPublishModes, record)
	       == ERROR)
	       {
	          /* Could not get record.				*/

		  itserrorinfo	= "Attempted to list file transfer modes";
		  itserror	= WebPublishBadGet;
		  result	= ERROR;
		  Report_Error();
	       }
	    record.Get_Key(extension);
	 }
      std::cout << std::endl;
      fflush(stdout);
      return(result);
   }
