/************************************************************************/
/* File		utilitymodule.cpp					*/
/*									*/
/* Purpose	This C++ program file implements the UtilityModule	*/
/*		class. The UtilityModule class provides a standard set	*/
/*		of routines that can be inherited by any library	*/
/*		module. The UtilityModule class is defined in		*/
/*		utilitymodule.h.					*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	0.0.0 (Prototype)					*/
/*									*/
/* Date		Friday, May 3, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <errno.h>			// OS errno.
#include <iostream>			// Standard stream class.
#include <string.h>			// Standard C string functions.
#include "utilitymodule.h"		// UtilityModule class.

/* Static	The following variable is the default error message	*/
/*		array that is used to initialize a UtilityModule	*/
/*		object. The default value should be overwritten by	*/
/*		modules that inherit the UtilityModule class.		*/

static char* errormessages[] =
   {
      "No utility module error",
   };

/************************************************************************/
/* Function	UtilityModule()						*/
/*									*/
/* Purpose	This is the default constructor for a UtilityModule	*/
/*		object. This constructor initializes the module's error	*/
/*		variables to their default values. The values that	*/
/*		should be overwritten by objects that inherit from the	*/
/*		UtilityModule class are 'itsmodulename', 'itsmaxerror',	*/
/*		and 'itserrormessages'.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This constructor will initialize a UtilityModule	*/
/*		object.							*/
/************************************************************************/

UtilityModule::UtilityModule()
   {
      itsmodulename	= "UtilityModule";
      itsoserror	= 0;
      itserror		= 0;
      itsmaxerror	= 1;
      itserrormessages	= errormessages;
   }

/************************************************************************/
/* Function	~UtilityModule()					*/
/*									*/
/* Purpose	This is the default destructor for a UtilityModule	*/
/*		object. This destructor does not do anything.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

UtilityModule::~UtilityModule()
   {
   }

/************************************************************************/
/* Function	void Report_Error(void)					*/
/*									*/
/* Purpose	This function will report a UtilityModule error to	*/
/*		stderr. When an object inherits UtilityModule, the	*/
/*		object should reimplement the error variables that are	*/
/*		specific to the new object. The variables that it	*/
/*		should reimplement are 'itsmodulename', 'itsmaxerror',	*/
/*		and 'itserrormessages'. The error string will be	*/
/*		written to stderr in the following format:		*/
/*									*/
/*		[UtilityModule name] ermsg: info message string.	*/
/*		[UtilityModule name] oserr: os error string.		*/
/*		[UtilityModule name] error: error string.		*/
/*									*/
/*		There are three parts to the error message. If an error	*/
/*		message has been specified through the			*/
/*		Set_Error_Info_String function then the error info	*/
/*		message will be written first. If there was an		*/
/*		Operating System (OS) error then the OS error is	*/
/*		written next. Finally, the error string that is		*/
/*		specific to the object that has inherited UtilityModule	*/
/*		will be written.					*/
/*									*/
/*		NOTE: 'itserror', 'itsoserror', and 'itserrorinfo' are	*/
/*		all reset to their default values by this function.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will write an error message to stderr.	*/
/*		The error message that is written depends on several	*/
/*		private member variables. If 'itserrorinfo' contains a	*/
/*		string, it is written to stderr. 'itserrorinfo' is	*/
/*		reset after it is written. Next, if 'itsoserror'	*/
/*		contains a value then an OS error message is written to	*/
/*		stderr. The value of 'itsoserror' is reset after the	*/
/*		message is written. Finally, the UtilityModule error	*/
/*		message is written. The value of 'itserror' is reset	*/
/*		after the module error message is written.		*/
/************************************************************************/

void UtilityModule::Report_Error(void)
   {

      /* If there is an informative message then write it.		*/

      if (itserrorinfo.Length() > 0)
         {
	    std::cerr << itsmodulename.Data() << " ermsg: \"" <<
	       itserrorinfo.Data() << "\".\n";
	 }
      /* If there is an OS error then report it first.			*/

      if (itsoserror > 0)
         {
	    std::cerr << itsmodulename.Data() << " oserr: \"" <<
	       strerror(itsoserror) << "\".\n";
	 }

      /* Finally, Report the module error.				*/

      std::cerr << itsmodulename.Data() << " error: \"";
      if (itserror < itsmaxerror)
         {
            std::cerr << itserrormessages[itserror];
	 }
      else
         {
	    std::cerr << "Error number is out of range";
	 }
      std::cerr << "\".\n";
      Reset_Error();
   }

/************************************************************************/
/* Function	void Reset_Error(void)					*/
/*									*/
/* Purpose	This function will reset a UtilityModule object's error	*/
/*		variables to their default values (no error). The error	*/
/*		variables and their default values are:			*/
/*									*/
/*		Variable	Default Value				*/
/*									*/
/*		itserrorinfo	"" (zero length string)			*/
/*		itsoserror	0 (Success)				*/
/*		itserror	0 (No error)				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The object's error variables will be set to their	*/
/*		default values.						*/
/************************************************************************/

void UtilityModule::Reset_Error(void)
   {
      itserrorinfo	= "";
      itsoserror	= 0;
      itserror		= 0;
   }

/************************************************************************/
/* Function	int Get_Error_Number(void) const			*/
/*									*/
/* Purpose	This function can be used to get the current value of	*/
/*		a UtilityModule object's error variable, 'itserror'.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The value of 'itserror' is returned.			*/
/************************************************************************/

int UtilityModule::Get_Error_Number(void) const
   {
      return(itserror);
   }

/************************************************************************/
/* Function	int Get_OS_Error_Number(void) const			*/
/*									*/
/* Purpose	This function can be used to get the current value of a	*/
/*		UtilityModule object's OS error variable, 'itsoserror'.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The value of 'itsoserror' is returned.			*/
/************************************************************************/

int UtilityModule::Get_OS_Error_Number(void) const
   {
      return(itsoserror);
   }

/************************************************************************/
/* Function	void Get_Error_String(String& error)			*/
/*									*/
/* Purpose	This function can be used to get a copy of the current	*/
/*		UtilityModule error string. The error string is in the	*/
/*		format of:						*/
/*									*/
/*		[UtilityModule name] error: error string.		*/
/*									*/
/* Input	This function expects a reference to a String object in	*/
/*		the variable 'error'.					*/
/*									*/
/* Output	This function will write a formatted error string to	*/
/*		the variable 'error'.					*/
/************************************************************************/

void UtilityModule::Get_Error_String(String& error)
   {
      error		= "";

      /* Report the module error.					*/

      error		+= itsmodulename;
      error		+= " error: \"";
      if (itserror < itsmaxerror)
         {
            error	+= itserrormessages[itserror];
	 }
      else
         {
	    error	+= "Error number is out of range";
	 }
      error		+= "\".\n";
   }

/************************************************************************/
/* Function	void Get_OS_Error_String(String& oserror)		*/
/*									*/
/* Purpose	This function can be used to get a copy of the current	*/
/*		UtilityModule OS error string. The OS error string is	*/
/*		in the format of:					*/
/*									*/
/*		[UtilityModule name] oserr: os error string.		*/
/*									*/
/* Input	This function expects a reference to a String object in	*/
/*		the variable 'oserror'.					*/
/*									*/
/* Output	This function will write a formatted error string to	*/
/*		the variable 'oserror'.					*/
/************************************************************************/

void UtilityModule::Get_OS_Error_String(String& oserror)
   {
      oserror		= "";

      /* If there is an OS error then report it.			*/

      if (itsoserror > 0)
         {
	    oserror	= itsmodulename;
	    oserror	+= " oserr: \"";
	    oserror	+= strerror(itsoserror);
	    oserror	+= "\".\n";
	 }
   }

/************************************************************************/
/* Function	void Get_Error_Info_String(String& message)		*/
/*									*/
/* Purpose	This function can be used to get a copy of the current	*/
/*		UtilityModule error info message. The error info	*/
/*		message is in the format of:				*/
/*									*/
/*		[UtilityModule name] ermsg: info message string.	*/
/*									*/
/* Input	This function expects a reference to a String object in	*/
/*		the variable 'message'.					*/
/*									*/
/* Output	This function will write a formatted error info message	*/
/*		to the variable 'message'.				*/
/************************************************************************/

void UtilityModule::Get_Error_Info_String(String& message)
   {
      message		= "";

      /* Finally, if there is any information then write it.		*/

      if (itserrorinfo.Length() > 0)
         {
	    message	+= itsmodulename;
	    message	+= " ermsg: \"";
	    message	+= itserrorinfo;
	    message	+= "\".\n";
	 }
   }

/************************************************************************/
/* Function	void Get_Complete_Error_String(String& error)		*/
/*									*/
/* Purpose	This function will write a UtilityModule error string	*/
/*		to the variable 'error'. When an object inherits	*/
/*		UtilityModule, the object should reimplement the error	*/
/*		variables that are specific to the new object. The	*/
/*		variables that it should reimplement are		*/
/*		'itsmodulename', 'itsmaxerror', and 'itserrormessages'.	*/
/*		The error string will be written to the variable	*/
/*		'error' in the format:					*/
/*									*/
/*		[UtilityModule name] ermsg: info message string.	*/
/*		[UtilityModule name] oserr: os error string.		*/
/*		[UtilityModule name] error: error string.		*/
/*									*/
/*		There are three parts to the error message. If an error	*/
/*		message has been specified through the			*/
/*		Set_Error_Info_String function then the error info	*/
/*		message will be written first. If there was an		*/
/*		Operating System (OS) error then the OS error is	*/
/*		written next. Finally, the error string that is		*/
/*		specific to the object that has inherited UtilityModule	*/
/*		will be written.					*/
/*									*/
/*		NOTE: 'itserror', 'itsoserror', and 'itserrorinfo' are	*/
/*		NOT reset to their default values by this function.	*/
/*									*/
/* Input	This function expects a reference to a String object in	*/
/*		the variable 'error'.					*/
/*									*/
/* Output	This function will write a formatted error string to	*/
/*		the variable 'error'.					*/
/************************************************************************/

void UtilityModule::Get_Complete_Error_String(String& error)
   {
      error		= "";

      /* If there is an informative message then write it.		*/

      if (itserrorinfo.Length() > 0)
         {
	    error	+= itsmodulename;
	    error	+= " ermsg: \"";
	    error	+= itserrorinfo;
	    error	+= "\".\n";
	 }

      /* If there is an OS error then report it first.			*/

      if (itsoserror > 0)
         {
	    error	+= itsmodulename;
	    error	+= " oserr: \"";
	    error	+= strerror(itsoserror);
	    error	+= "\".\n";
	 }

      /* Finally, Report the module error.				*/

      error		+= itsmodulename;
      error		+= " error: \"";
      if (itserror < itsmaxerror)
         {
            error	+= itserrormessages[itserror];
	 }
      else
         {
	    error	+= "Error number is out of range";
	 }
      error		+= "\".\n";
   }

/************************************************************************/
/* Function	void Set_Error_Number(const int error)			*/
/*									*/
/* Purpose	This function can be used to set the current value of a	*/
/*		UtilityModule object's error value, 'itserror'.		*/
/*									*/
/* Input	This function accepts an integer that represents a	*/
/*		UtilityModule object error in the variable 'error'.	*/
/*									*/
/* Output	The private member variable 'itserror' will be set to	*/
/*		the value passed to this function in the variable	*/
/*		'error'.						*/
/************************************************************************/

void UtilityModule::Set_Error_Number(const int error)
   {
      itserror		= error;
   }

/************************************************************************/
/* Function	void Set_OS_Error_Number(const int oserror)		*/
/*									*/
/* Purpose	This function can be used to set the current value of a	*/
/*		UtilityModule object's OS error value, 'itsoserror'.	*/
/*									*/
/* Input	This function accepts an integer that represents an OS	*/
/*		error in the variable 'oserror'.			*/
/*									*/
/* Output	The private member variable 'itsoserror' will be set to	*/
/*		the value passed to this function in the variable	*/
/*		'oserror'.						*/
/************************************************************************/

void UtilityModule::Set_OS_Error_Number(const int oserror)
   {
      itsoserror	= oserror;
   }

/************************************************************************/
/* Function	void Set_Error_Info_String(const String& info)		*/
/*									*/
/* Purpose	This function can be used to set the current value of a	*/
/*		UtilityModule object's error information string,	*/
/*		'itserrorinfo'.						*/
/*									*/
/* Input	This function accepts a reference to a String object	*/
/*		that contains an information message in the variable	*/
/*		'info'.							*/
/*									*/
/* Output	The private member variable 'itserrorinfo' will be set	*/
/*		to the value passed to this function in the variable	*/
/*		'info'.							*/
/************************************************************************/

void UtilityModule::Set_Error_Info_String(const String& info)
   {
      itserrorinfo	= info;
   }
