/************************************************************************/
/* File		physicalos.cpp						*/
/*									*/
/* Purpose	This C++ program file implements the PhysicalOS class.	*/
/*		The PhysicalOS class provides a C++ interface to the	*/
/*		physical operating system. The PhysicalOS class is	*/
/*		defined in the file physicalos.h.			*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 18, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <sys/types.h>			// PhysicalOS types.
#include <sys/stat.h>			// umask() function.
#include "os.h"				// Physical operating system.

/************************************************************************/
/* Function	int Get_Linux_Umask(void) const				*/
/*									*/
/* Purpose	This function will return the linux umask value. The	*/
/*		linux umask is maintained by the linux kernel at the	*/
/*		process level. Whenever a file or a directory is	*/
/*		created or opened, the permission bits for the created	*/
/*		or opened file or directory are AND'd with the value of	*/
/*		the linux kernel umask. If your file or directory	*/
/*		permission is 0666, for example, and the linux umask	*/
/*		value is 0770, then the file or directory will be	*/
/*		created or opened with the mode value of 0660.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the current umask value that	*/
/*		is maintained by the linux kernel at the process level.	*/
/************************************************************************/

int PhysicalOS::Get_Linux_Umask(void) const
   {
      int		mask;

      mask		= umask(0);
      (void)umask(mask);
      return(mask);
   }

/************************************************************************/
/* Function	void Set_Linux_Umask(const int mask)			*/
/*									*/
/* Purpose	This function will set the linux umask value. The linux	*/
/*		umask is maintained by the linux kernel at the process	*/
/*		level. Whenever a file or a directory is created or	*/
/*		opened, the permission bits for the created or opened	*/
/*		file or directory are AND'd with the value of the linux	*/
/*		kernel umask. If your file or directory permission is	*/
/*		0666, for example, and the linux umask value is 0770,	*/
/*		then the file or directory will be created or opened	*/
/*		with the mode value of 0660.				*/
/*									*/
/*		The following constants are defined in physical.h for	*/
/*		your use:						*/
/*									*/
/*		Constant		Value	Description		*/
/*									*/
/*		PhysicalOSOwnRWX	0700	Owner read/write/exec.	*/
/*		PhysicalOSOwnRead	0400	Owner can read.		*/
/*		PhysicalOSOwnWrite	0200	Owner can write.	*/
/*		PhysicalOSOwnExec	0100	Owner can execute.	*/
/*		PhysicalOSGrpRWX	0070	Group read/write/exec.	*/
/*		PhysicalOSGrpRead	0040	Group can read.		*/
/*		PhysicalOSGrpWrite	0020	Group can write.	*/
/*		PhysicalOSGrpExec	0010	Group can execute.	*/
/*		PhysicalOSAllRWX	0007	All read/write/exec.	*/
/*		PhysicalOSAllRead	0004	All others can read.	*/
/*		PhysicalOSAllWrite	0002	All others can write.	*/
/*		PhysicalOSAllExec	0001	All others can execute.	*/
/*									*/
/*		Create the umask value by combining the values that you	*/
/*		want. (e.g. PhysicalOSOwnRead | PhysicalOSOwnWrite).	*/
/*									*/
/* Input	This function expects the variable 'mask' to contain	*/
/*		the new value for the umask that is maintained by the	*/
/*		linux kernel at the process level.			*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void PhysicalOS::Set_Linux_Umask(const int mask)
   {
      (void)umask(mask);
   }
