/************************************************************************/
/* File		dataindex.cpp						*/
/*									*/
/* Purpose	This C++ program file contains the DataIndex class	*/
/*		member functions that remove a key from a DataIndex	*/
/*		object. All of the public and private member functions	*/
/*		that are required to remove a key from a DataIndex	*/
/*		object are located in this file.			*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 26, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include "dataindex.h"			// DataIndex class.

/************************************************************************/
/* Function	status Remove_Key(const char* key)			*/
/*									*/
/* Purpose	This function is responsible for removing a key from	*/
/*		the DataIndex red/black tree index file. After the key	*/
/*		has been removed the red/black tree will be rebalanced.	*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is to be removed from the DataIndex index	*/
/*		file.							*/
/*									*/
/* Output	If this function is able to remove the key from the	*/
/*		DataIndex index file then this function will return OK.	*/
/*		If this function was not able to remove the key from	*/
/*		the index file then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/*		NOTE: Some errors by this function will lock the index.	*/
/************************************************************************/

status DataIndex::Remove_Key(const char* key)
   {
      status		result		= OK;

      /* Make sure that the index is not already locked.		*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to remove ";
	    itserrorinfo	+= key;
	    itserrorinfo	+= " from file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Locate the key's node in the red/black tree.			*/

      else if (locate_deletion_point(key) == ERROR)
         {
	    /* Could not remove key from index.				*/

            itsheader.Set_Locked_Index_State(true);
	 }

      /* If the key does not exist, report the error and return.	*/

      else if (itsdeletedpointer == 0)
         {
	    /* Key does not exist in index.				*/

	    itserrorinfo	= "Attempted to remove ";
	    itserrorinfo	+= key;
	    itserrorinfo	+= " from file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexKeyNot;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Delete the key from the red/black tree and from the file.	*/

      if (delete_key() == ERROR)
         {
	    /* Could not remove key from index.				*/

            itsheader.Set_Locked_Index_State(true);
	 }

      /* Rebalance the red/black tree if it is needed.			*/

      else if (rebalance_after_delete() == ERROR)
         {
	    /* Could not remove key from index.				*/

            itsheader.Set_Locked_Index_State(true);
	 }

      /* Else, the key was removed successfully and we need to		*/
      /* decrement the node count.					*/

      else
         {
	    /* Decrement the node count and write the file's header to	*/
	    /* record the change.					*/

	    itsheader.Decrement_Node_Count();
            if (itsindexfile.Write_ObjectFile_Header() == ERROR)
               {
	          /* Could not add key to index file.			*/

	          itserrorinfo	= "Could not write header to index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexBadDel;
	          result	= ERROR;
	          Report_Error();
	          itsheader.Set_Locked_Index_State(true);
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_Key(const String& key)			*/
/*									*/
/* Purpose	This function is responsible for removing a key from	*/
/*		the DataIndex red/black tree index file. After the key	*/
/*		has been removed the red/black tree will be rebalanced.	*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is to be removed from the DataIndex index	*/
/*		file.							*/
/*									*/
/* Output	If this function is able to remove the key from the	*/
/*		DataIndex index file then this function will return OK.	*/
/*		If this function was not able to remove the key from	*/
/*		the index file then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/*		NOTE: Some errors by this function will lock the index.	*/
/************************************************************************/

status DataIndex::Remove_Key(const String& key)
   {
      return(Remove_Key(key.Data()));
   }

/************************************************************************/
/* Function	status locate_deletion_point(const char* key)		*/
/*									*/
/* Purpose	This function is responsible for locating the position	*/
/*		in the tree to delete a key from. The stack is used to	*/
/*		keep track of the branch of nodes to the location.	*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is being deleted from the tree.		*/
/*									*/
/* Output	If this function is able to locate the key to delete in	*/
/*		the tree then this function will return OK and the	*/
/*		stack will contain the branch of nodes that lead up to	*/
/*		the key. Also, the private member variables		*/
/*		'itsdeletedpointer' and 'itsdeletedobject' will contain	*/
/*		the record pointer and record for the key that is being	*/
/*		deleted. If the private member variable 'itsindex'	*/
/*		equals -1 then the root key is being deleted from the	*/
/*		tree. If this function is not able to locate the key in	*/
/*		the tree then it the private member variable		*/
/*		'itsdeletedpointer' will be set to 0. If there was an	*/
/*		error searching for the key then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::locate_deletion_point(const char* key)
   {
      status		result		= OK;
      relation		difference;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Prepare to enter the loop that will search for the key.	*/

      itsindex		= -1;
      itsdeletedpointer	= 0;
      workpointer	= itsheader.Get_Root_Node();

      /* The following loop will search the tree, looking for the key.	*/

      while (workpointer != 0)
         {
	    /* Read in the record for the current pointer.		*/

	    if (read_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not locate deletion point.			*/

		  itserrorinfo	= "Attempted to locate deletion point "
				  "for \"";
		  itserrorinfo	+= key;
		  itserrorinfo	+= "\" in index file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= DataIndexLocateDelete;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Determine the difference between the current record and	*/
	    /* the given key.						*/

	    difference	= workobject.Get_Record_Key().Compare(key);

	    /* If they are the same then this is the key that is to be	*/
	    /* deleted from the red/black tree.				*/

	    if (difference == EQUAL)
	       {
	          /* Save the record for the key that is being deleted.	*/

	          itsdeletedobject	= workobject;
		  itsdeletedpointer	= workpointer;

		  /* Break out of the loop with the key located.	*/

		  break;
	       }

	    /* If we made it here then we need to keep looking so save	*/
	    /* the current record on the stack and get the next key.	*/

	    push_record(workobject, workpointer, difference);
	    if (difference == LESS)
	       {
	          workpointer	= workobject.Get_Right_Node();
	       }
	    else
	       {
	          workpointer	= workobject.Get_Left_Node();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status case_one_delete(void)				*/
/*									*/
/* Purpose	This function is responsible for replacing the deleted	*/
/*		key in the tree with the deleted key's left subtree.	*/
/*									*/
/* Input	This function assumes that the private member variables	*/
/*		'itsdeletedpointer' and 'itsdeletedobject' contain the	*/
/*		record number and record for the key that is being	*/
/*		removed from the tree. Also, the stack is assumed to	*/
/*		contain the branch of nodes that lead up to the key	*/
/*		that is being deleted.					*/
/*									*/
/* Output	If this function is able to replace the deleted key	*/
/*		with its left subtree then this function will return	*/
/*		OK. If this function is not able to replace the deleted	*/
/*		key with its left subtree then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::case_one_delete(void)
   {
      status		result		= OK;

      /* If the private member variable 'itsindex' equals -1 then we	*/
      /* need to change the root node.					*/

      if (itsindex == -1)
	 {
	    /* Set the new root node. The change is recorded when the	*/
	    /* Remove_Key function decrements the node count.		*/

	    itsheader.Set_Root_Node(itsdeletedobject.Get_Left_Node());
	 }
      else
	 {
	    /* Set the parent's left of right node appropriately.	*/

	    if (itsdifferences[itsindex] == LESS)
	       {
	          itsobjects[itsindex].Set_Right_Node(
		     itsdeletedobject.Get_Left_Node());
	       }
	    else
	       {
	          itsobjects[itsindex].Set_Left_Node(
		     itsdeletedobject.Get_Left_Node());
	       }

	    /* Write the record to the index file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result		= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status case_two_delete(DataIndexRec& object,		*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function is responsible for replacing the deleted	*/
/*		key with the deleted key's right child.			*/
/*									*/
/* Input	This function assumes that the private member variables	*/
/*		'itsdeletedpointer' and 'itsdeletedobject' contain the	*/
/*		record number and record for the key that is being	*/
/*		removed from the tree. Also, the stack is assumed to	*/
/*		contain the branch of nodes that lead up to the key	*/
/*		that is being deleted.					*/
/*									*/
/* Output	If this function was able to replace the deleted key	*/
/*		with its right child then this function will return OK.	*/
/*		If this function was not able to replace the deleted	*/
/*		key with its right child then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::case_two_delete(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;
      RBColor		color;

      /* Set the right child's left node to the left deleted node.	*/

      object.Set_Left_Node(itsdeletedobject.Get_Left_Node());

      /* Save the color of the right child.				*/

      color		= object.Get_Color();

      /* Set the right child color to the color of the deleted key.	*/

      object.Set_Color(itsdeletedobject.Get_Color());

      /* Save the right child's color as the deleted color.		*/

      itsdeletedobject.Set_Color(color);

      /* Push this right child on the stack for rebalancing later.	*/

      push_record(object, pointer, LESS);

      /* Write the right child to the file to record the change.	*/

      if (write_index_record(object, pointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the private member variable 'itsindex' equals 0 then we	*/
      /* need to change the root node.					*/

      else if (itsindex == 0)
	 {
	    /* Make the right child the new root node. The change is	*/
	    /* recorded when the Remove_Key function decrements the	*/
	    /* node count.						*/

	    itsheader.Set_Root_Node(pointer);
	 }

      /* Else, we need to make the parent of the deleted key point to	*/
      /* the right child that is replacing the deleted key.		*/

      else
	 {
	    /* Determine which node will point to the right child.	*/

	    if (itsdifferences[itsindex - 1] == LESS)
	       {
	          itsobjects[itsindex - 1].Set_Right_Node(pointer);
	       }
	    else
	       {
	          itsobjects[itsindex - 1].Set_Left_Node(pointer);
	       }

	    /* Write the record to the file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex - 1],
	       itspointers[itsindex - 1]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status case_three_delete(DataIndexRec& object,		*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function is responsible for replacing the deleted	*/
/*		key with the deleted key's inorder successor.		*/
/*									*/
/* Input	This function assumes that the private member variables	*/
/*		'itsdeletedpointer' and 'itsdeletedobject' contain the	*/
/*		record number and record for the key that is being	*/
/*		removed from the tree. Also, the stack is assumed to	*/
/*		contain the branch of nodes that lead up to the key	*/
/*		that is being deleted.					*/
/*									*/
/* Output	If this function was able to replace the deleted key	*/
/*		with its inorder successor then this function will	*/
/*		return OK. If this function was not able to replace the	*/
/*		deleted key with its inorder successor then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::case_three_delete(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;
      int		index;
      RBColor		color;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Push the object onto the stack to reserve a space on the	*/
      /* stack. The record at this stack position will be replaced	*/
      /* after the following loop locates the inorder successor.	*/

      push_record(object, pointer, LESS);

      /* Save the current stack position so that the record at this	*/
      /* position can be replaced later.				*/

      index		= itsindex;

      /* This is the loop that will locate the key's inorder successor.	*/

      while (true)
         {
	    /* Save the current object on the stack in case we need to	*/
	    /* rebalance these nodes later on.				*/

	    push_record(object, pointer, MORE);

	    /* Get the left node of the current object.			*/

	    workpointer	= object.Get_Left_Node();

	    /* Read in the record for this node.			*/

	    if (read_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
		  break;
	       }

	    /* Break out of the loop there are no more nodes.		*/

	    if (workobject.Get_Left_Node() == 0)
	       {
	          break;
	       }

	    /* Prepare to enter the loop again.				*/

	    object	= workobject;
	    pointer	= workpointer;
	 }

      /* If there were no errors locating the inorder node then go	*/
      /* ahead and replace the deleted key with its inorder successor.	*/

      if (result == OK)
         {
	    /* The record that we reserved on the stack is replaced	*/
	    /* with the inorder successor.				*/

	    itsobjects[index]	= workobject;
	    itspointers[index]	= workpointer;

	    /* The left node of the inorder successor is set to the	*/
	    /* left node of the deleted key.				*/

	    itsobjects[index].Set_Left_Node(
	       itsdeletedobject.Get_Left_Node());

	    /* The left node of the record on the top of the stack is	*/
	    /* set to the right node of the inorder successor.		*/

	    itsobjects[itsindex].Set_Left_Node(
	       itsobjects[index].Get_Right_Node());

	    /* The right node of the inorder successor is set to the	*/
	    /* right node of the deleted object.			*/

	    itsobjects[index].Set_Right_Node(
	       itsdeletedobject.Get_Right_Node());

	    /* Save the color of the inorder successor.			*/

	    color		= itsobjects[index].Get_Color();

	    /* Set the color of the inorder successor to the color of	*/
	    /* the deleted key.						*/

	    itsobjects[index].Set_Color(itsdeletedobject.Get_Color());

	    /* Save the color of the inorder successor as the color of	*/
	    /* the deleted node. This helps in rebalancing.		*/

	    itsdeletedobject.Set_Color(color);

	    /* Write the records back to the file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(itsobjects[index],
	       itspointers[index]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the reserved stack record is the first record then	*/
	    /* we need to change the root node.				*/

	    else if (index == 0)
	       {
	          /* Change the root node of the tree to the inorder	*/
		  /* successor. This change is recorded when the	*/
		  /* Remove_Key function decrements the node count.	*/

	          itsheader.Set_Root_Node(workpointer);
	       }
	    else
	       {
	          /* Make the parent node of the deleted key point to	*/
		  /* the inorder successor.				*/

	          if (itsdifferences[index - 1] == LESS)
		     {
		        itsobjects[index - 1].Set_Right_Node(
			   workpointer);
		     }
		  else
		     {
		        itsobjects[index - 1].Set_Left_Node(
			   workpointer);
		     }

		  /* Write the record to the file to record the change.	*/

	          if (write_index_record(itsobjects[index - 1],
		     itspointers[index - 1]) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result		= ERROR;
	                Report_Error();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_key(void)					*/
/*									*/
/* Purpose	This function is responsible for removing a key from	*/
/*		the red/black tree. The tree will still need to be	*/
/*		rebalanced after this function is called.		*/
/*									*/
/* Input	This function assumes that the private member variables	*/
/*		'itsdeletedpointer' and 'itsdeletedobject' contain the	*/
/*		record number and record for the key that is being	*/
/*		removed from the tree. Also, the stack is assumed to	*/
/*		contain the branch of nodes that lead up to the key	*/
/*		that is being deleted.					*/
/*									*/
/* Output	If this function was able to remove the key from the	*/
/*		red/black tree then it will return OK. If this function	*/
/*		was not able to remove the key from the red/black tree	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status DataIndex::delete_key(void)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Begin by removing the key's record from the physical file.	*/

      if (itsindexfile.Remove_ObjectFile_Record(itsdeletedpointer)
         == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the deleted key has no right child then this is case one.	*/

      else if (itsdeletedobject.Get_Right_Node() == 0)
         {
	    /* Case 1 of delete.					*/

	    result	= case_one_delete();
	 }
      else
         {
	    /* Get the deleted key's right node.			*/

	    workpointer		= itsdeletedobject.Get_Right_Node();

	    /* Read in the record for the right node.			*/

	    if (read_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the right child has no left node then this is case 2.	*/

	    else if (workobject.Get_Left_Node() == 0)
	       {
	          /* Case 2 of delete.					*/

		  result	= case_two_delete(workobject,
				  workpointer);
	       }

	    /* Else, this is case 3.					*/

	    else
	       {
	          /* Case 3 of delete.					*/

		  result	= case_three_delete(workobject,
				  workpointer);
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status ensure_right_side_black(DataIndexRec& object,	*/
/*		   FilePointer& pointer)				*/
/*									*/
/* Purpose	This function is responsible for reducing the number of	*/
/*		cases that we need to consider in the main loop. This	*/
/*		function does this by performing a right rotation	*/
/*		around the parent of the object and recolor if the	*/
/*		object is red.						*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the record for the node that is being rebalanced. The	*/
/*		variable 'pointer' must contain the record pointer for	*/
/*		the record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to ensure that the right side	*/
/*		is black then this function will return OK. If this	*/
/*		function is not able to ensure that the right side is	*/
/*		black then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::ensure_right_side_black(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;

      /* The object's color is set to black.				*/

      object.Set_Color(RBColorBlack);

      /* Set the color of the record on the top of the stack to red.	*/

      itsobjects[itsindex].Set_Color(RBColorRed);

      /* Set the left node of the record on the top of the stack to the	*/
      /* right node of the object.					*/

      itsobjects[itsindex].Set_Left_Node(object.Get_Right_Node());

      /* Set the right node of the object to the record that is on the	*/
      /* top of the stack.						*/

      object.Set_Right_Node(itspointers[itsindex]);

      /* Write the records back to the file to record the change.	*/

      if (write_index_record(object, pointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (write_index_record(itsobjects[itsindex],
         itspointers[itsindex]) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the stack index equals 0 then we need to change the root.	*/

      if (itsindex == 0)
         {
	    /* Make the object the new root node. The change is		*/
	    /* recorded when the Remove_Key function decrements the	*/
	    /* node count.						*/

	    itsheader.Set_Root_Node(pointer);
	 }

      /* Else, make the parent on the stack point to the object.	*/

      else
         {
	    /* Determine which side should point to the record.		*/

	    if (itsdifferences[itsindex - 1] == LESS)
	       {
	          itsobjects[itsindex - 1].Set_Right_Node(pointer);
	       }
	    else
	       {
	          itsobjects[itsindex - 1].Set_Left_Node(pointer);
	       }

	    /* Write the record to the file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex - 1],
	       itspointers[itsindex - 1]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }

      /* If there have not been any errors to this point then...	*/

      if (result == OK)
         {
	    /* Push the record on the top of the stack onto the stack	*/
	    /* a second time.						*/

	    push_record(itsobjects[itsindex], itspointers[itsindex],
	       LESS);

	    /* Now replace the original record on the top of the stack	*/
	    /* with the object.						*/

	    itsobjects[itsindex - 1]	= object;
	    itspointers[itsindex - 1]	= pointer;

	    /* Make the left node of the record on the top of the stack	*/
	    /* the new object to consider in the rebalancing loop.	*/

	    pointer			= itsobjects[itsindex].
					  Get_Left_Node();
	    if (read_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_case_one_both_sides(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function is called when the node that is being	*/
/*		rebalanced does not have any red children. This		*/
/*		function recolors the node red. The loop will then	*/
/*		continue the rebalancing with the next node on the	*/
/*		stack.							*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the current node that is to be recolored. The variable	*/
/*		'pointer' must contain the record pointer for the index	*/
/*		record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to recolor the record red then	*/
/*		this function will return OK. If this function is not	*/
/*		able to recolor the record red then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::delete_case_one_both_sides(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;

      /* Recolor the record and write it to the index file.		*/

      object.Set_Color(RBColorRed);
      if (write_index_record(object, pointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_case_two_right_side(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function is called when the left node of the	*/
/*		record that is being rebalanced is red. This function	*/
/*		will perform a right rotation around the record on the	*/
/*		top of the stack and recolor some nodes.		*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the current node that is to be rebalanced. The variable	*/
/*		'pointer' must contain the record pointer for the index	*/
/*		record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to perform a right rotation	*/
/*		around the record that is on the top of the stack then	*/
/*		this function will return OK. If this function was not	*/
/*		able to perform a right rotation around the record that	*/
/*		is on the top of the stack then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::delete_case_two_right_side(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Set the color of the record that is being rebalanced to the	*/
      /* color of the record that is on the top of the stack.		*/

      object.Set_Color(itsobjects[itsindex].Get_Color());

      /* Set the color of the record that is on the top of the stack	*/
      /* to black.							*/

      itsobjects[itsindex].Set_Color(RBColorBlack);

      /* Get the left node of the record that is being rebalanced.	*/

      workpointer	= object.Get_Left_Node();

      /* Get the left node from the index file.				*/

      if (read_index_record(workobject, workpointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Set the color of this node to black.			*/

	    workobject.Set_Color(RBColorBlack);

	    /* Set the left node of the record on the top of the stack	*/
	    /* to the right node of the record we just read in.		*/

	    itsobjects[itsindex].Set_Left_Node(object.Get_Right_Node());

	    /* Set the right node of the record that we just read in to	*/
	    /* the record that is on the top of the stack.		*/

	    object.Set_Right_Node(itspointers[itsindex]);

	    /* Write the records to the index file to record the change.*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the stack pointer equals 0 then we need to change the	*/
	    /* root node to the record that was read in.		*/

	    else if (itsindex == 0)
	       {
	          /* Change the root node to the record that was read	*/
		  /* in from the index file. The change is recorded	*/
		  /* when the Remove_Key function decrements the node	*/
		  /* count.						*/

	          itsheader.Set_Root_Node(pointer);
	       }

	    /* Else, we need to make the parent of the the record that	*/
	    /* is on the top of the stack point to the record that was	*/
	    /* read in from the index file.				*/

	    else
	       {
	          /* Determine which node to change in the parent of	*/
		  /* the record that is on the top of the stack.	*/

	          if (itsdifferences[itsindex - 1] == LESS)
		     {
		        itsobjects[itsindex - 1].Set_Right_Node(pointer);
		     }
		  else
		     {
		        itsobjects[itsindex - 1].Set_Left_Node(pointer);
		     }

		  /* Write the record to the file to record the change.	*/

	          if (write_index_record(itsobjects[itsindex - 1],
		     itspointers[itsindex - 1]) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result		= ERROR;
	                Report_Error();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_case_three_right_side(			*/
/*		   DataIndexRec& object, FilePointer& pointer)		*/
/*									*/
/* Purpose	This function is called when the right child of the	*/
/*		record that is being rebalanced is red. This function	*/
/*		will rotate the tree at the record that is being	*/
/*		rebalanced to transform the case 3 violation to a case	*/
/*		2 violation.						*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the current node that is to be rebalanced. The variable	*/
/*		'pointer' must contain the record pointer for the index	*/
/*		record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to transform case 3 to case 2	*/
/*		then this function will return OK. If this function is	*/
/*		not able to transform case 3 to case 2 then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::delete_case_three_right_side(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Get the right node of the record that is being rebalanced and	*/
      /* read the record in from the index file.			*/

      workpointer	= object.Get_Right_Node();
      if (read_index_record(workobject, workpointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Set the color of the record that was read in to black.	*/

	    workobject.Set_Color(RBColorBlack);

	    /* Set the record that is being rebalanced to red.		*/

	    object.Set_Color(RBColorRed);

	    /* Set the left node of the record that is being rebalanced	*/
	    /* to the right node of the record that was read in.	*/

	    object.Set_Right_Node(workobject.Get_Left_Node());

	    /* Set the left node of the record that was read in to the	*/
	    /* object that is being rebalanced.				*/

	    workobject.Set_Left_Node(pointer);

	    /* Write the records back to the file to record the change.	*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else
	       {
	          /* Replace the record that is being rebalanced with	*/
		  /* the record that was read in.			*/

	          object	= workobject;
	          pointer	= workpointer;

		  /* Set the right node of the record that is on the	*/
		  /* top of the stack to the record that was read in.	*/

	          itsobjects[itsindex].Set_Left_Node(workpointer);

		  /* Write the record to the file to record the change.	*/

	          if (write_index_record(itsobjects[itsindex],
		     itspointers[itsindex]) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result		= ERROR;
	                Report_Error();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status ensure_left_side_black(DataIndexRec& object,	*/
/*		   FilePointer& pointer)				*/
/*									*/
/* Purpose	This function is responsible for reducing the number of	*/
/*		cases that we need to consider in the main loop. This	*/
/*		function does this by performing a left rotation around	*/
/*		the parent of the object and recolor if the object is	*/
/*		red.							*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the record for the node that is being rebalanced. The	*/
/*		variable 'pointer' must contain the record pointer for	*/
/*		the record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to ensure that the left side	*/
/*		is black then this function will return OK. If this	*/
/*		function is not able to ensure that the left side is	*/
/*		black then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::ensure_left_side_black(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;

      /* The object's color is set to black.				*/

      object.Set_Color(RBColorBlack);

      /* Set the color of the record on the top of the stack to red.	*/

      itsobjects[itsindex].Set_Color(RBColorRed);

      /* Set the right node of the record on the top of the stack to	*/
      /* the left node of the object.					*/

      itsobjects[itsindex].Set_Right_Node(object.Get_Left_Node());

      /* Set the left node of the object to the record that is on the	*/
      /* top of the stack.						*/

      object.Set_Left_Node(itspointers[itsindex]);

      /* Write the records back to the file to record the change.	*/

      if (write_index_record(object, pointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (write_index_record(itsobjects[itsindex],
         itspointers[itsindex]) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the stack index equals 0 then we need to change the root.	*/

      if (itsindex == 0)
         {
	    itsheader.Set_Root_Node(pointer);
	 }

      /* Else, make the parent on the stack point to the object.	*/

      else
         {
	    /* Determine which side should point to the record.		*/

	    if (itsdifferences[itsindex - 1] == LESS)
	       {
	          itsobjects[itsindex - 1].Set_Right_Node(pointer);
	       }
	    else
	       {
	          itsobjects[itsindex - 1].Set_Left_Node(pointer);
	       }

	    /* Write the record to the file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex - 1],
	       itspointers[itsindex - 1]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }

      /* If there have not been any errors to this point then...	*/

      if (result == OK)
         {
	    /* Push the record on the top of the stack onto the stack	*/
	    /* a second time.						*/

	    push_record(itsobjects[itsindex], itspointers[itsindex],
	       MORE);

	    /* Now replace the original record on the top of the stack	*/
	    /* with the object.						*/

	    itsobjects[itsindex - 1]	= object;
	    itspointers[itsindex - 1]	= pointer;

	    /* Make the right node of the record on the top of the	*/
	    /* stack the new object to consider in the rebalancing loop.*/

	    pointer			= itsobjects[itsindex].
					  Get_Right_Node();
	    if (read_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_case_two_left_side(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function is called when the right node of the	*/
/*		record that is being rebalanced is red. This function	*/
/*		will perform a left rotation around the record on the	*/
/*		top of the stack and recolor some nodes.		*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the current node that is to be rebalanced. The variable	*/
/*		'pointer' must contain the record pointer for the index	*/
/*		record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to perform a left rotation	*/
/*		around the record that is on the top of the stack then	*/
/*		this function will return OK. If this function was not	*/
/*		able to perform a left rotation around the record that	*/
/*		is on the top of the stack then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::delete_case_two_left_side(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Set the color of the record that is being rebalanced to the	*/
      /* color of the record that is on the top of the stack.		*/

      object.Set_Color(itsobjects[itsindex].Get_Color());

      /* Set the color of the record that is on the top of the stack	*/
      /* to black.							*/

      itsobjects[itsindex].Set_Color(RBColorBlack);

      /* Get the right node of the record that is being rebalanced.	*/

      workpointer	= object.Get_Right_Node();

      /* Get the right node from the index file.			*/

      if (read_index_record(workobject, workpointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Set the color of this node to black.			*/

	    workobject.Set_Color(RBColorBlack);

	    /* Set the right node of the record on the top of the stack	*/
	    /* to the left node of the record we just read in.		*/

	    itsobjects[itsindex].Set_Right_Node(object.Get_Left_Node());

	    /* Set the left node of the record that we just read in to	*/
	    /* the record that is on the top of the stack.		*/

	    object.Set_Left_Node(itspointers[itsindex]);

	    /* Write the records to the index file to record the change.*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* If the stack pointer equals 0 then we need to change the	*/
	    /* root node to the record that was read in.		*/

	    else if (itsindex == 0)
	       {
	          /* Change the root node to the record that was read	*/
		  /* in from the index file. The change is recorded	*/
		  /* when the Remove_Key function decrements the node	*/
		  /* count.						*/

	          itsheader.Set_Root_Node(pointer);
	       }

	    /* Else, we need to make the parent of the the record that	*/
	    /* is on the top of the stack point to the record that was	*/
	    /* read in from the index file.				*/

	    else
	       {
	          /* Determine which node to change in the parent of	*/
		  /* the record that is on the top of the stack.	*/

	          if (itsdifferences[itsindex - 1] == LESS)
		     {
		        itsobjects[itsindex - 1].Set_Right_Node(pointer);
		     }
		  else
		     {
		        itsobjects[itsindex - 1].Set_Left_Node(pointer);
		     }

		  /* Write the record to the file to record the change.	*/

	          if (write_index_record(itsobjects[itsindex - 1],
		     itspointers[itsindex - 1]) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result		= ERROR;
	                Report_Error();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status delete_case_three_left_side(DataIndexRec& object,*/
/*		   FilePointer& pointer)				*/
/*									*/
/* Purpose	This function is called when the left child of the	*/
/*		record that is being rebalanced is red. This function	*/
/*		will rotate the tree at the record that is being	*/
/*		rebalanced to transform the case 3 violation to a case	*/
/*		2 violation.						*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the current node that is to be rebalanced. The variable	*/
/*		'pointer' must contain the record pointer for the index	*/
/*		record that is referenced by the variable 'object'.	*/
/*									*/
/* Output	If this function is able to transform case 3 to case 2	*/
/*		then this function will return OK. If this function is	*/
/*		not able to transform case 3 to case 2 then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::delete_case_three_left_side(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Get the left node of the record that is being rebalanced and	*/
      /* read the record in from the index file.			*/

      workpointer	= object.Get_Left_Node();
      if (read_index_record(workobject, workpointer) == ERROR)
         {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Set the color of the record that was read in to black.	*/

	    workobject.Set_Color(RBColorBlack);

	    /* Set the record that is being rebalanced to red.		*/

	    object.Set_Color(RBColorRed);

	    /* Set the left node of the record that is being rebalanced	*/
	    /* to the right node of the record that was read in.	*/

	    object.Set_Left_Node(workobject.Get_Right_Node());

	    /* Set the left node of the record that was read in to the	*/
	    /* object that is being rebalanced.				*/

	    workobject.Set_Right_Node(pointer);

	    /* Write the records back to the file to record the change.	*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else if (write_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	    else
	       {
	          /* Replace the record that is being rebalanced with	*/
		  /* the record that was read in.			*/

	          object	= workobject;
	          pointer	= workpointer;

		  /* Set the right node of the record that is on the	*/
		  /* top of the stack to the record that was read in.	*/

	          itsobjects[itsindex].Set_Right_Node(workpointer);

		  /* Write the record to the file to record the change.	*/

	          if (write_index_record(itsobjects[itsindex],
		     itspointers[itsindex]) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result		= ERROR;
	                Report_Error();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status rebalance_after_delete(void)			*/
/*									*/
/* Purpose	This function is responsible for rebalancing the	*/
/*		red/black tree after a key has been removed from the	*/
/*		tree. This function assumes that the stack contains the	*/
/*		list of nodes that may need to be replaced.		*/
/*									*/
/* Input	This function assumes that a key has just been removed	*/
/*		from the tree and that the stack contains a list of the	*/
/*		nodes that may need to be rebalanced.			*/
/*									*/
/* Output	If this function is able to rebalance the red/black	*/
/*		tree then it will return OK. If this function is not	*/
/*		able to rebalance the red/black tree then this function	*/
/*		will return ERROR. All errors by this function are	*/
/*		reported to stderr.					*/
/************************************************************************/

status DataIndex::rebalance_after_delete(void)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;
      DataIndexRec	rightobject;
      DataIndexRec	leftobject;

      /* The tree is balanced if the deleted key's color was red.	*/

      if (itsdeletedobject.Get_Color() == RBColorBlack)
         {
	    /* This is the loop that will rebalance the nodes that are	*/
	    /* on the stack.						*/

	    while (itsindex > -1 && result == OK)
	       {
	          /* Determine which side of the record on the top of	*/
		  /* the stack the key was deleted from.		*/

		  if (itsdifferences[itsindex] == LESS)
		     {
			/* Get the right node of the record on the	*/
			/* top of the stack.				*/

			workpointer	= itsobjects[itsindex].
					  Get_Right_Node();
		     }
		  else
		     {
		        /* Get the left node of the record on the	*/
			/* top of the stack.				*/

		        workpointer	= itsobjects[itsindex].
					  Get_Left_Node();
		     }

		  /* If the pointer is not equal to zero then it points	*/
		  /* to a record in the index file.			*/

		  if (workpointer != 0)
		     {
		        /* Read in the record from the file.		*/

		        if (read_index_record(workobject, workpointer)
			   == ERROR)
		           {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
			      break;
			   }
		     }

		  /* If the pointer does not equal 0 AND the color of	*/
		  /* the record is red then...				*/

		  if (workpointer != 0 && workobject.Get_Color()
		     == RBColorRed)
		     {
		        /* Set the color of the record to red.		*/

		        workobject.Set_Color(RBColorBlack);

			/* Write the record back to record the change.	*/

		        if (write_index_record(workobject, workpointer)
			   == ERROR)
		           {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
			   }

			/* Break out of the loop, we are done.		*/

			break;
		     }

		  /* If the stack index is out of range then break out	*/
		  /* of the loop, we are done.				*/

	          if (itsindex < 0)
		     {
		        break;
		     }

		  /* Determine which side of the tree needs rebalancing.*/

		  if (itsdifferences[itsindex] == LESS)
		     {
		        /* Right side rebalancing.			*/

			workpointer	= itsobjects[itsindex].
					  Get_Left_Node();

			/* Read in the left node of the record on the	*/
			/* top of the stack.				*/

			if (read_index_record(workobject, workpointer)
			   == ERROR)
			   {
			      /* Could not delete key.			*/

			      itserrorinfo	= "Attempted to delete "
						  "key \"";
			      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
			      itserrorinfo	+= "\" from index file ";
			      itserrorinfo	+= itsfilename;
			      itserror		= DataIndexNoDelete;
			      result		= ERROR;
			      Report_Error();
			      break;
			   }

			/* If the record is red then make it black.	*/

			if (workobject.Get_Color() == RBColorRed)
			   {
			      /* Ensure workobject is black.		*/

			      result	= ensure_right_side_black(
					  workobject, workpointer);
			   }

			/* If the record has a right node then...	*/

			if (workobject.Get_Right_Node() != 0)
			   {
			      /* Read in the right node.		*/

			      if (read_index_record(rightobject,
			         workobject.Get_Right_Node()) == ERROR)
			         {
			            /* Could not delete key.		*/

			            itserrorinfo  = "Attempted to "
						    "delete key \"";
			            itserrorinfo  += itsdeletedobject.
						     Get_Record_Key();
			            itserrorinfo  += "\" from index "
						     "file ";
			            itserrorinfo  += itsfilename;
			            itserror	  = DataIndexNoDelete;
			            result	  = ERROR;
			            Report_Error();
			            break;
			         }
			   }

			/* If the record has a left node then...	*/

			if (workobject.Get_Left_Node() != 0)
			   {
			      /* read in the left node.			*/

			      if (read_index_record(leftobject,
			         workobject.Get_Left_Node()) == ERROR)
			         {
			            /* Could not delete key.		*/

			            itserrorinfo  = "Attempted to "
						    "delete key \"";
			            itserrorinfo  += itsdeletedobject.
						     Get_Record_Key();
			            itserrorinfo  += "\" from index "
						     "file ";
			            itserrorinfo  += itsfilename;
			            itserror	  = DataIndexNoDelete;
			            result	  = ERROR;
			            Report_Error();
			            break;
			         }
			   }

			/* If the record does not have a right node OR	*/
			/* the record's right node is black AND the	*/
			/* record does not have a left node OR the left	*/
			/* node is black then...			*/

			if ((workobject.Get_Right_Node() == 0 ||
			   rightobject.Get_Color() == RBColorBlack)
			   && (workobject.Get_Left_Node() == 0 ||
			   leftobject.Get_Color() == RBColorBlack))
			   {
			      /* Case 1.				*/

			      result	= delete_case_one_both_sides(
					  workobject, workpointer);
			   }
			else
			   {
			      /* If the record does not have a left	*/
			      /* node OR the record's left node is	*/
			      /* black then...				*/

			      if (workobject.Get_Left_Node() == 0 ||
			         leftobject.Get_Color() == RBColorBlack)
			         {
				    /* Convert case 3 to case 2.	*/

				    result = delete_case_three_right_side(
					     workobject, workpointer);
				 }
			      /* Case 2.				*/

			      if (result == OK)
			         {
				    result = delete_case_two_right_side(
				 	     workobject, workpointer);
				 }
			      break;
			   }
		     }
		  else
		     {
		        /* Left side rebalancing.			*/

			workpointer	= itsobjects[itsindex].
					  Get_Right_Node();

			/* Read in the right node of the record on the	*/
			/* top of the stack.				*/

			if (read_index_record(workobject, workpointer)
			   == ERROR)
			   {
			      /* Could not delete key.			*/

			            itserrorinfo  = "Attempted to "
						    "delete key \"";
			            itserrorinfo  += itsdeletedobject.
						     Get_Record_Key();
			            itserrorinfo  += "\" from index "
						     "file ";
			            itserrorinfo  += itsfilename;
			            itserror	  = DataIndexNoDelete;
			            result	  = ERROR;
			            Report_Error();
			            break;
			   }

			/* If the record is red then make it black.	*/

			if (workobject.Get_Color() == RBColorRed)
			   {
			      /* Ensure workobject is black.		*/

			      result	= ensure_left_side_black(
					  workobject, workpointer);
			   }

			/* If the record has a right node then...	*/

			if (workobject.Get_Right_Node() != 0)
			   {
			      /* Read in the record.			*/

			      if (read_index_record(rightobject,
			         workobject.Get_Right_Node()) == ERROR)
			         {
			            /* Could not delete key.		*/

			            itserrorinfo  = "Attempted to "
						    "delete key \"";
			            itserrorinfo  += itsdeletedobject.
						     Get_Record_Key();
			            itserrorinfo  += "\" from index "
						     "file ";
			            itserrorinfo  += itsfilename;
			            itserror	  = DataIndexNoDelete;
			            result	  = ERROR;
			            Report_Error();
			            break;
			         }
			   }

			/* If the record has a left node then...	*/

			if (workobject.Get_Left_Node() != 0)
			   {
			      /* Read in the record.			*/

			      if (read_index_record(leftobject,
			         workobject.Get_Left_Node()) == ERROR)
			         {
			            /* Could not delete key.		*/

			            itserrorinfo  = "Attempted to "
						    "delete key \"";
			            itserrorinfo  += itsdeletedobject.
						     Get_Record_Key();
			            itserrorinfo  += "\" from index "
						     "file ";
			            itserrorinfo  += itsfilename;
			            itserror	  = DataIndexNoDelete;
			            result	  = ERROR;
			            Report_Error();
			            break;
			         }
			   }

			/* If the record does not have a right node OR	*/
			/* the record's right node is black AND the	*/
			/* record does not have a left node OR the left	*/
			/* node is black then...			*/

			if ((workobject.Get_Right_Node() == 0 ||
			   rightobject.Get_Color() == RBColorBlack)
			   && (workobject.Get_Left_Node() == 0 ||
			   leftobject.Get_Color() == RBColorBlack))
			   {
			      /* Case 1.				*/

			      result	= delete_case_one_both_sides(
					  workobject, workpointer);
			   }
			else
			   {
			      /* If the record does not have a left	*/
			      /* node OR the record's left node is	*/
			      /* black then...				*/

			      if (workobject.Get_Right_Node() == 0 ||
			         rightobject.Get_Color() == RBColorBlack)
			         {
				    /* Convert case 3 to case 2.	*/

				    result = delete_case_three_left_side(
					     workobject, workpointer);
				 }
			      /* Case 2.				*/

			      if (result == OK)
			         {
				    result = delete_case_two_left_side(
				 	     workobject, workpointer);
				 }

			      /* Break out of the loop, we are done.	*/

			      break;
			   }
		     }

		  /* Decrement the stack pointer and go through the	*/
		  /* loop again.					*/

		  itsindex--;
	       }
	 }
      return(result);
   }
