/*
 *   Copyright (C) 1997, 1998, 1999 Loic Dachary
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>
#endif /* HAVE_DMALLOC_H */

#include <hash.h>
#include <mime.h>
#include <split.h>
#include <strsubs.h>
#include <mysql.h>
#include <salloc.h>
#include <strlower.h>

#include <sqlutil.h>

static int verbose = 0;

static int mime_accept_1(webbase_t* base, char* spec, char* content_type, char* path);
static void mime_expand(webbase_t* base, char* spec, int spec_length,
			char** mime_listp, int* mime_list_lengthp,
			char** ext_listp, int* ext_list_lengthp);

char* mime_ext2mime(webbase_t* base, char* path, char* default_type)
{
  static char mime_object[128];
  char query[256];
  char* ext = strrchr(path, '.');
  char* mime = default_type;

  if(ext && strlen(ext) > 1) {
    MYSQL_RES *res;
    MYSQL_ROW row;
    ext++;
    sprintf(query, "select mime from mime2ext where ext like '%%%s,%%'", ext);
    smysql_query(&base->mysql, query);
    res = smysql_store_result(&base->mysql);
    if(mysql_num_rows(res) > 0) {
      row = mysql_fetch_row(res);
      strcpy(mime_object, row[0]);
      mime = mime_object;
    }
    mysql_free_result(res);
  }

  return mime;
}

int mime_accept_ext(webbase_t* base, char* spec, char* path)
{
  return mime_accept_1(base, spec, 0, path);
}

int mime_accept_type(webbase_t* base, char* spec, char* content_type)
{
  return mime_accept_1(base, spec, content_type, 0);
}

/*
 * Return true if <content_type> mime type is accepted according to <spec> 
 */
static int mime_accept_1(webbase_t* base, char* spec, char* content_type, char* path)
{
  static char* spec_current = 0;
  static int spec_current_size = 0;
  static char* mime_list = 0;
  static int mime_list_size = 0;
  static char* ext_list = 0;
  static int ext_list_size = 0;
  
  int spec_length = strlen(spec);
  int unallocated = spec_current_size;

  static_alloc(&spec_current, &spec_current_size, spec_length + 1);
  if(unallocated)
    spec_current[0] = '\0';

  /*
   * Load spec in memory from database if not current
   */
  if(strcmp(spec_current, spec)) {
    static char* query = 0;
    static int query_size = 0;
    MYSQL_RES *res;
    MYSQL_ROW row;
    char* mime_tmp;
    char* ext_tmp;
    int mime_tmp_length;
    int ext_tmp_length;

    static_alloc(&query, &query_size, spec_length + 128);
    
    sprintf(query, "select mime,ext from mime_restrict where spec = '%s'", sql_quote_char_simple(spec, strlen(spec)));
    smysql_query(&base->mysql, query);
    res = smysql_store_result(&base->mysql);
    if(mysql_num_rows(res) > 0) {
      row = mysql_fetch_row(res);

      mime_tmp = row[0];
      mime_tmp_length = strlen(mime_tmp);
      ext_tmp = row[1];
      ext_tmp_length = strlen(ext_tmp);
    } else {
      mime_expand(base, spec, spec_length, &mime_tmp, &mime_tmp_length, &ext_tmp, &ext_tmp_length);
      static_alloc(&query, &query_size, spec_length + mime_tmp_length + ext_tmp_length + 128);
      sprintf(query, "replace into mime_restrict (spec, mime, ext) values ('%s', '%s', '%s')",
	      spec,
	      sql_quote_char(&mime_list, &mime_list_size, mime_tmp, mime_tmp_length),
	      sql_quote_char(&ext_list, &ext_list_size, ext_tmp, ext_tmp_length)
	      );
      smysql_query(&base->mysql, query);
    }

    static_alloc(&mime_list, &mime_list_size, mime_tmp_length + 1);
    strcpy(mime_list, mime_tmp);

    static_alloc(&ext_list, &ext_list_size, ext_tmp_length + 1);
    strcpy(ext_list, ext_tmp);

    mysql_free_result(res);
  }

  {
#define TMP_SIZE 64
    char tmp[TMP_SIZE];
    int accepted = 1;

    /*
     * If no extentions or extension not in the list of refused 
     * extensions, accept.
     */
    if(path) {
      char* ext = strrchr(path, '.');
      if(ext) {
	ext++;

	strncpy(tmp, ext, TMP_SIZE - 2);
	tmp[TMP_SIZE - 2] = '\0';
	strcat(tmp, ",");
	strlower(tmp, strlen(tmp));

	if(verbose > 1) fprintf(stderr, "looking for %s in %s\n", tmp, ext_list);
	if(strstr(ext_list, tmp))
	  accepted = 0;
      }
    }
    /*
     * If content_type is in the the list of accepted content types
     * accept.
     */
    if(content_type) {
      tmp[0] = ',';
      strncpy(tmp + 1, content_type, TMP_SIZE - 2);
      tmp[TMP_SIZE - 2] = '\0';

      if(!strstr(mime_list, tmp))
	accepted = 0;
    }

    return accepted;
  }
}

/*
 * Transform an accept (HTTP style) <spec> into a list of
 * accepted mime types <mime_listp> and a list of accepted extensions
 * <ext_listp>.
 */
static void mime_expand(webbase_t* base, char* spec, int spec_length,
			char** mime_listp, int* mime_list_lengthp,
			char** ext_listp, int* ext_list_lengthp)
{
  static char* mime_list = 0;
  static int mime_list_size = 0;
  int mime_list_length = 0;
  static char* ext_list = 0;
  static int ext_list_size = 0;
  int ext_list_length = 0;
  char** splitted;
  int count;
  static char* query = 0;
  static int query_size = 0;

  split(spec, spec_length, &splitted, &count, ',', SPLIT_TRIM);

  static_alloc(&mime_list, &mime_list_size, 256);
  static_alloc(&ext_list, &ext_list_size, 256);
  static_alloc(&query, &query_size, 256);

  mime_list[0] = '\0';
  ext_list[0] = '\0';

  {
    MYSQL_RES *res;
    MYSQL_ROW row;
    int query_length;
    char* tmp_query;
    int i;

    strcpy(query, "select mime,ext from mime2ext where 1 = 1");
    query_length = strlen(query);

    for(i = 0; i < count; i++) {
      char* spec = splitted[i];
      int spec_length = strlen(spec);
      if(spec[spec_length - 1] == '*')
	spec[spec_length - 1] = '%';
      static_alloc(&query, &query_size, query_length + spec_length);
      sprintf(query + query_length, " and mime not like '%s'", sql_quote_char_simple(spec, strlen(spec)));
      query_length += strlen(query + query_length);
    }

    /*
     * Search for non valid extensions.
     */
    smysql_query(&base->mysql, query);
    res = smysql_store_result(&base->mysql);
    if(mysql_num_rows(res) > 0) {
      while((row = mysql_fetch_row(res))) {
	char* ext;
	int ext_length;
	ext = row[1];
	ext_length = strlen(ext);
	/*
	 * + 1 for the null at end
	 */
	static_alloc(&ext_list, &ext_list_size, ext_list_length + ext_length + 1);
	strcpy(ext_list + ext_list_length, ext);
	ext_list_length += ext_length;
      }
    }
    mysql_free_result(res);

    /*
     * Search for content type list
     */
    {
      char* from[] = {
	"1 = 1",
	" and",
	"not like",
	0
      };
      char* to[] = {
	"1 = 0",
	"  or",
	"    like",
	0
      };
      int size;
      tmp_query = strsubs(query, query_length, &size, from, to);
      if(verbose > 1) fprintf(stderr, "%s -> %s\n", query, tmp_query);
    }
    smysql_query(&base->mysql, tmp_query);
    res = smysql_store_result(&base->mysql);
    if(mysql_num_rows(res) > 0) {
      while((row = mysql_fetch_row(res))) {
	char* mime;
	int mime_length;
	mime = row[0];
	mime_length = strlen(mime);
	/*
	 * + 1 for the comma
	 * + 1 for the null at end
	 */
	static_alloc(&mime_list, &mime_list_size, mime_list_length + mime_length + 1 + 1);
	mime_list[mime_list_length] = ',';
	mime_list_length++;
	strcpy(mime_list + mime_list_length, mime);
	mime_list_length += mime_length;
      }
    }
    mysql_free_result(res);
  }

  if(verbose) fprintf(stderr, "mime_expand: mime_list = %s, ext_list = %s\n", mime_list, ext_list);
    
  *mime_listp = mime_list;
  *mime_list_lengthp = mime_list_length;
  *ext_listp = ext_list;
  *ext_list_lengthp = ext_list_length;
}
