/* Source Installer, Copyright (c) 2005-2006 Claudio Fontana

 action.c - main actions

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "global.h"
#include "gui.h"
#include "menu.h"
#include "dialog.h"
#include "console.h"
#include "greeting.h"
#include "text.h"
#include "package_list.h"
#include "status.h"

#include "action.h"

static int action_install_aux(SRCINST_PHANDLE h, char* filename, int reconf);
static int action_find_configure(SRCINST_PHANDLE h, char* filename);
static int action_configure(SRCINST_PHANDLE h);
static void action_start(void);
static void action_end(void);
static void action_error(SRCINST_ERR rv, char* title, char* text, char* when);

static void action_error(SRCINST_ERR rv, char* title, char* text, char* when) {
  if (rv == SRCINST_ERR_INTR) {
    dialog_warning(title, "Action interrupted.", "An interrupt signal has been received while", when);
  } else {
    dialog_warning(title, text, "This error condition has been detected while", when);
  }
}

void action_add(char* filename, char* packagename) {
  SRCINST_PHANDLE h; int result;
  char* title = "Adding new package";
  SRCINST_TYPE ft;

  result = 0; h = 0;
  action_start();

  ft = srcinst_file_type(filename);
  if (ft != SRCINST_TYPE_DIR && ft != SRCINST_TYPE_FILE) {
    dialog_warning(title, "Invalid package",
		   "Not a regular file or directory:", filename);
    goto end_proc;
  }
  
  if (!(h = srcinst_new_package(packagename))) {
    dialog_warning(title, "Package already exists",
		   "Could not create already existing package", packagename);
    goto end_proc;
  }

  if (!action_install_aux(h, filename, 1))
    goto end_proc;

  result = 1;
  
 end_proc:
  if (h && !(srcinst_close_package(h))) {
    dialog_warning(title, "Error updating package information", "An error condition has been detected trying to update and close the package information.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Package added successfully.", "Your new package has been successfully added.", 0);
  }
}

static int action_install_aux(SRCINST_PHANDLE h, char* filename, int reconf) {
  float v;			/* autoconf version */
  SRCINST_ERR rv;
  char* title = "Installation";

  status_done(0.0, 10.0);

  if ((rv = srcinst_init_build(h, filename, 0)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Could not initialize build", "initializing the build prior to installation.");
    return 0;
  }

  status_done(10.0, 12.5);

  if (reconf) {
    if ((rv = srcinst_chdir_build(h, 0)) != SRCINST_ERR_OK) {
      action_error(rv, title, "Could not change build directory", "preparing to reconfigure package.");
      return 0;
    }

    if ((rv = srcinst_detect_autoconf(h, &v)) == SRCINST_ERR_MISSING) {
      status_done(12.5, 15.0);

      if (!action_find_configure(h, filename))
	return 0;
      
    } else if (rv != SRCINST_ERR_OK) {
      action_error(rv, title, "Error detecting configuration script", "reconfiguring package prior to installation.");
      return 0;
    }

    status_done(15.0, 20.0);
    
    if (!action_configure(h)) {
      return 0;
    }
  }
  
  status_done(20.0, 98.0);

  if ((rv = srcinst_install(h, state.o.install, state.o.source, state.o.strip,
			    0, state.o.src_compress)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Installation failed", "performing actual installation.");
    return 0;
  }

  status_done(98.0, 100.0);

  return 1;
}

static int action_configure(SRCINST_PHANDLE h) {
  struct srcinst_configure_sets sets;
  struct srcinst_string_list configure_strings;
  SRCINST_ERR rv;
  int result = 0;
  char* title = "Configuration";
  
  srcinst_init_string_list(&configure_strings);
  
  if (state.o.manual_conf) {
    if ((rv = srcinst_detect_configure_opt(h, &sets)) != SRCINST_ERR_OK) {
      action_error(rv, title, "Failed to detect configure options", "trying to detect supported configure options.");
      if (rv == SRCINST_ERR_INTR)
	goto end_proc;

    } else {
      int result2;
      result2 = dialog_configure(&sets, &configure_strings);
      srcinst_free_configure_opt(&sets);
      if (!result2)
	goto end_proc;
    }
  }
  
  if ((rv = srcinst_configure(h, state.o.prefix, &configure_strings)) !=
      SRCINST_ERR_OK) {
    action_error(rv, title, "Configuration failed!", "running configuration scripts.");
    goto end_proc;
  }

  result = 1;

 end_proc:
  srcinst_free_string_list(&configure_strings);
  return result;
}

static int action_find_configure(SRCINST_PHANDLE h, char* filename) {
  struct srcinst_string_list scripts;
  SRCINST_ERR rv;
  int result;
  char* subdir = 0;
  char* title = "Find configure script";
      
  srcinst_init_string_list(&scripts);

  if ((rv = srcinst_find_configure(h, &scripts)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Error searching for configuration script", "searching for a configuration script.");
    goto end_proc;
  }

  if (scripts.count == 1) {
    subdir = srcinst_dirname(scripts.strings[0]);

  } else {
    int idx;
    idx = dialog_input_select_string(title, "Multiple build directories have been detected.\nThis is a sometimes a deprecated way to support different systems.\nPlease select a build directory from these choices.", &scripts);
    if (idx == -1)
      goto end_proc;

    subdir = srcinst_dirname(scripts.strings[idx]);
  }

  if ((rv = srcinst_chdir_build(h, subdir)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Could not change build subdirectory", "attempting to move to the chosen build subdirectory.");
    goto end_proc;
  }

  result = 1;

 end_proc:
  if (subdir)
    free(subdir);

  srcinst_free_string_list(&scripts);
  return result;
}

void action_install(int reconf) {
  int result;
  char* title = "Install package";
  char* source;

  action_start();

  result = 0;
  
  if (!(source = srcinst_get_source_location(state.package))) {
    dialog_warning(title, "Failed to find source code", "This package source code is not reachable, even though it should be available. Installation impossible.", 0);
    goto end_proc;
  }
  
  if (!action_install_aux(state.package, source, reconf))
    goto end_proc;

  result = 1;

 end_proc:
  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package", "An unexpected error has occured when finalizing and closing the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Installation complete", "Package has been successfully installed.", 0);
  }
}

void action_remove(void) {
  int result = 0;
  SRCINST_ERR rv;
  char* title = "Remove package";
  
  action_start();

  status_done(0.0, 10.0);
  
  if (srcinst_is_installed(state.package)) {
    if (srcinst_is_source_available(state.package)) {
      rv = srcinst_init_build(state.package,
			      srcinst_get_source_location(state.package), 0);

      if (rv != SRCINST_ERR_OK) {
	action_error(rv, title, "Could not initialize build", "initializing source build for optional source-assisted uninstallation.");
	
	if (rv == SRCINST_ERR_INTR)
	  goto end_proc;
	/* ignore other errors of init_build, because we can uninstall
	   internally if source is not available/does not work */
      }
    }

    status_done(10.0, 95.0);

    if ((rv = srcinst_uninstall_binary(state.package)) != SRCINST_ERR_OK) {
      action_error(rv, title, "Could not uninstall binaries", "deleting the installed files.");
      goto end_proc;
    }
  }

  status_done(95.0, 98.0);
    
  if (srcinst_is_source_available(state.package)) {
    if ((rv = srcinst_uninstall_source(state.package)) != SRCINST_ERR_OK) {
      action_error(rv, title, "Could not remove sources", "trying to delete the stored source code");
      goto end_proc;
    }
  }

  status_done(98.0, 100.0);
  result = 1;

 end_proc:
  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package", "An unexpected error has occured when finalizing and closing the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Removal complete", "Package has been successfully removed.", 0);
  }
}

void action_uninstall(void) {
  SRCINST_ERR rv;
  int result = 0;
  char* title = "Uninstall package";

  action_start();

  status_done(0.0, 10.0);

  if (srcinst_is_source_available(state.package)) {
    rv = srcinst_init_build(state.package, 
			    srcinst_get_source_location(state.package), 0);

    if (rv != SRCINST_ERR_OK) {
      action_error(rv, title, "Could not initialize build", "initializing source build for optional source-assisted uninstallation.");
	
      if (rv == SRCINST_ERR_INTR)
	goto end_proc;
      /* ignore other errors of init_build, because we can uninstall
	 internally if source is not available/does not work */
    }
  }

  status_done(10.0, 98.0);

  if ((rv = srcinst_uninstall_binary(state.package)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Could not uninstall binaries", "deleting the installed files.");
    goto end_proc;
  }

  status_done(98.0, 100.0);
  result = 1;

 end_proc:

  if (!srcinst_close_package(state.package)) {
    dialog_warning(title, "Error closing package", "An unexpected error has occured when finalizing and closing the package.", 0);
    result = 0;
  }

  status_done(100.0, 100.0);
  action_end();

  if (result) {
    dialog_message(title, "Package uninstalled.", "The package has been uninstalled successfully.", 0);
  }
}

void action_export_info(SRCINST_PHANDLE h, char* fname, SRCINST_EXPORT fmt) {
  SRCINST_ERR rv;
  char* title = "Export information";

  if ((rv = srcinst_export_info(h, fname, fmt)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Failed to export information", "trying to export package information to the chosen file.");

  } else {
    dialog_message(title, "Information successfully exported", "Package information has been successfully written to", fname);
  }
}

void action_export_bin(char* fname, SRCINST_COMP fmt) {
  SRCINST_ERR rv; SRCINST_PHANDLE h;
  char* title = "Export binary package";
  h = state.package;

  if ((rv = srcinst_export_binary(h, fname, fmt)) != SRCINST_ERR_OK) {
    action_error(rv, title, "Failed to export binary package", "trying to write the chosen file.");

  } else {
    dialog_message(title, "Binary package successfully exported", "Binary package has been successfully written as", fname);
  }
}

static void action_start(void) {
  console_clear();
  console_show();
  status_reset();
  status_show();

  state.actions_locked = 1;
  srcinst_register_stdout_callback(console_print_stdout);
  srcinst_register_stderr_callback(console_print_stderr);
  srcinst_register_spawn_callback(console_print_spawn);
  srcinst_register_beat_callback(gui_doevents);
  gui_update();
}

static void action_end(void) {
  status_hide();

  state.package = 0;
  state.actions_locked = 0;
  package_list_fill();
  srcinst_unregister_stdout_callback(console_print_stdout);
  srcinst_unregister_stderr_callback(console_print_stderr);
  srcinst_unregister_spawn_callback(console_print_spawn);
  srcinst_unregister_beat_callback(gui_doevents);
  gui_update();
}
