/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : pips/src/pipsAddPnm.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsAddPnm (int ncId, char *varName, char *imgName, nc_type dataType)
 *
 *   int ncId         IS   the identifier returned by ncopen() or nccreate()
 *   char *varName    IS   name of variable to create, default is imgName
 *   char *imgName    IS   name of file to open, NULL means read from stdin
 *   nc_type dataType IS   type of variable to create
 * 
 * Description   : 
 *
 *   Reads the image file header to determine type and size of image,
 *   then creates a variable 'varName' or 'imgName', if varName is empty
 *   and the variable doesn't exist.
 *   The image data is read and converted to NC_BYTE or NC_LONG (only ppm)
 *   and written to the end of the variable.
 */

#include "pips.h"   /* PIPS system header file  */

#ifdef __STDC__

int pipsAddPnm (int ncId, char *varName, char *imgName,nc_type dataType )

#else

pipsAddPnm (ncId, varName, imgName, dataType)
    int ncId;
    char *varName;
    char *imgName;
    nc_type dataType;

#endif
{
/*
 * Indicates, if a variable for the current image(sequence) has been created
 */
  static int variableCreated = 0;
/*
 * Variables for the image file
 */
  FILE *file = NULL;      /* File handle of image file                       */
  long width=0;           /* Width and height of image in file 'name'        */
  long height=0;
  int fileType;           /* Type of presented image file                    */
/*
 * Variables for the netcdf file
 */
  static int dataId;      /* Data variable Id                                */
  static long image = 0;  /* current entries in record dimension             */
  int typeLength;         /* Length of 'dataType' in byte                    */
/*
 * Variables for the transfer of the image body
 */
  long edge  [3];         /* Number of values to write                       */
  long start [3];         /* Coordinates of pixel                            */
  long x, y;	    	  /* indizes for width and height and pixel value    */
  char *pixelRow;	      /* pointer to 1 row of pixels                      */
  char *pixelCol;         /* pointer to current column in pixelrow           */

/*
 * Open image file and give error message on failure
 * empty name means: read stdin
 */
 if (! imgName)
    file = stdin;
  else
    file = fopen (imgName, "rb");
  if (!file)
    pipsPrintError (ncId, PipsErrorFileRead, imgName, PipsExit);

/******************************************************************************
 *    R E A D   I M A G E   F I L E   H E A D E R 
 ******************************************************************************
 */
  switch (fileType = pipsReadPnmHeader (file, &width, &height)) {
    case EOF : return (PipsNoImages);
/*
 * Give an error message, if file is not in known format
 */
    case PipsUnknown :
      fclose (file);
      pipsPrintError (ncId, PipsImageType, imgName, PipsExit);
  } /* end switch */

/*
 * Set the data type that is to be used in the netcdf file
 * if it's not specified by the user
 */
  if (dataType == NC_UNSPECIFIED)
    if ((fileType == PipsPpmAscii) || (fileType == PipsPpmRaw))
      dataType = NC_LONG;
    else
      dataType = NC_BYTE;


/******************************************************************************
 *  D E F I N E   N E W   V A R.   F O R   I M A G E ( S E Q U E N C E )
 ******************************************************************************
 */
  if (! variableCreated) {               /* Variable hasn't been created yet */
    variableCreated = 1;

/*
 * Set 'varName' to the default 'imgName', if varName was omitted
 * or to "stdin", if both was NULL
 */
    if (varName[0] == 0)
      strcat (varName, imgName);
    if (varName[0] == 0)
      strcat (varName, "stdin");

/*
 * Create a variable that can hold an image sequence of width x height
 */
    dataId = pipsVarDef (ncId, varName, &dataType, &fileType,
                          &image, &height, &width);

  } /* end if */

/*
 * Set constant values for hyperslab of data to write
 */
  start [0] = image++;             /* recDim: number of image           */
  start [2] = 0;                   /* columnDim: start with left column */
  edge  [0] = 1;                   /* recDim: write in current image    */
  edge  [1] = 1;                   /* rowDim: write in current row      */
  edge  [2] = width;               /* columnDim: write all columns      */

/*
 * Calculate length of 'dataType'
 */
  typeLength = nctypelen (dataType);

/*
 * Allocate memory to hold one row of data
 */
  pixelRow = (char *) calloc (width, typeLength);

/******************************************************************************
 *  R E A D   I M A G E   D A T A
 ******************************************************************************
 * Read pnm data and put into variable line by line
 */
  for (y=0; y < height; y++) {
    start [1] = y;
    pixelCol = pixelRow;
    for (x=0; x < width; x++) {

/*
 * Read one data value from image file
 */
      pipsReadPnmValue (file, fileType, pixelCol);
      pixelCol += typeLength;
    } /* end for (x) */
    ncvarput (ncId, dataId, start, edge, (void *) pixelRow);
  }	/* for (y) */

/*
 * Free 'row' memory
 */
  free (pixelRow);

/*
 * Close file if it is not stdin
 */
  if (imgName)
    fclose (file);
  return (PipsOk);
}  /* end pipsAddPnm */


/* Conversion statements pgm <-> ppm
 *
 * 8->24 Bit: Duplicate colour to red, green, blue
 *
 *       if (fileType > nfileType)
 *        p = p | (p << 8) | (p << 16);
 *  or   p *= 65793;
 *
 * 24->8 Bit: Calculate average of red, green and blue colour
 *
 *      if (fileType < nfileType)
 *        p = ((p >> 16) + ((p >> 8) & 0xFF) + (p & 0xFF)) / 3;
 *  or   p /= 65793;
 */

/* end pipsAddPnm.c */
