/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : pips/src/pipsExtPnm.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsExtPnm (ncId, varName, number, imgName, imgType)
 *
 *   int ncId         IS   the identifier returned by ncopen() or nccreate();
 *   char *varName    IS   name of variable to query, NULL means all variables
 *                         
 *   long number      IS   number of image to extract from variable,
 *                         default is all images.
 *   char *imgName    IS   name of image file to create, default is 'varName'
 *   int imgType      IS   type of image to create, possible values:
 *                         PipsUnknown, PipsPbmAscii, PipsPgmAscii,
 *                         PpmAscii, PgmRaw, PpmRaw.
 *                         PipsUnknown causes to determine the type from the
 *                         variable attribute or data type
 * 
 * Description   : 
 *
 *   Creates an image file named 'imgName' of type 'imgType' and copies the 
 *   data from the variable 'varName' to it. If 'number' > -1, then that image
 *   from the image sequence is extracted, otherwise all images are extracted
 *   and stored in numbered files (foo.pgm.0, foo.pgm.1, ...).
 *   If 'varName' is empty, then all variables are processed.
 */

#include "pips.h"    /* PIPS system header file  */
#include <malloc.h>  /* Memory allocation        */

#ifdef __STDC__

int pipsExtPnm (int ncId,char *varName,long number,char *imgName,int imgType)

#else

pipsExtPnm (ncId, varName, number, imgName, imgType)
    int ncId;
    char *varName;
    long number;
    char *imgName;
    int imgType;

#endif
{
/*
 * The 4th value of edge and start is needed, if the variable dim is 0
 */
  long edge [4];             /* Number of values to read (always 1)          */
  long start [4];            /* Coordinates of pixel                         */
  FILE *file;                /* File handle of image file                    */
  long x, y;                 /* indizes for width and height                 */
  char data [16];            /* buffer for data                              */
  char *name;                /* Text buffer for constructing a filename      */
  int ndims;                 /* Number of dimensions of 'dataId'             */
  nc_type type;              /* NC-type of variable 'dataId'                 */
  int dataId = 0;            /* Current variable id                          */
  long images;               /* Images per variable in the netcdf file       */
  long height;               /* Height of images in 'dataId'                 */
  long width;                /* Width of images in 'dataId'                  */
  int imgTypeSet = 0;        /* True means the user named the image type     */
  int extAllImg = 1;         /* True means extract all images of a variable  */
  char identifier[MAX_NC_NAME];  /* Identifier ('name' of fileType)          */
  char *position;            /* Pointer to a substring                       */

/*
 * Set size of data package (1 value)
 */
  edge [0] = 1;  edge [1] = 1;  edge [2] = 1;  edge [3] = 1;

/*
 * If an image name is given, extract only one variable
 * (as writing all variables in the same image isn't very usefull)
 */
  if (imgName != NULL)
    if (strlen (varName) == 0)
      ncvarinq (ncId, dataId, varName, 0, 0, 0, 0);

/*
 * Set a flag to determine, whether or not the image type should be detected
 * and wheter or not all images should be extracted
 */
  imgTypeSet = (imgType != PipsUnknown);
  extAllImg  = (number < 0);

/*
 * As long as varInq() says there is still a variable, do extract it
 */
  while (pipsVarInq (ncId, &dataId, varName, &type, &images,&height,&width)) {

/*
 * If no image name was given, take the variable name
 */
    if (! imgName)
      imgName = varName;

/*
 * Derive number of dimensions from images, height, width
 */
    ndims = 0;
    ndims += (images > 0);
    ndims += (height > 0);
    ndims += (width  > 0);

/*
 * Start with image number 0, if all images are to be extracted
 */
    if (extAllImg)
      number = 0;

/*
 * Extract the filetype from the variable attribute, if user didn't name it
 * Default is PipsPpmRaw for NC_LONG and PipsPgmRaw for NC_BYTE
 */
    identifier[0] = 0;
    if (! imgTypeSet) {
      imgType = ncattget (ncId, dataId, PipsFileTypeAtt, identifier);
/*
 * Convert the string 'identifier' to integer 'imgType'
 */
      position = strstr (
        "ppm raw  pgm raw  pbm raw  ppm asciipgm asciipbm ascii", identifier);
      if (position)                            /* so substring was found ... */
        imgType = strlen (position) / 9;
      else
        imgType = -1;
      if (imgType == -1) {
        if (nctypelen(type) == nctypelen (NC_LONG))
          imgType = PipsPpmRaw;
        else
          if (nctypelen(type) == nctypelen (NC_BYTE))
            imgType = PipsPgmRaw;
          else {
/*
 * Don't convert incompatible variable
 */
            pipsPrintError (-1, PipsDataIncompat, varName, PipsDontExit);
            number = images;                 /* proceed to next variable */
          } /* end else */
      } /* end if */
    } /* end if ! */

/*
 * Extract images from current variable
 */
    while ((number < images) && (extAllImg)) {

/*
 * Append the image number to the filename
 * imgName == 'foo.pgm', number == 34, -> name == 'foo.pgm.34'
 */
      name = (char *) calloc (strlen (imgName)+15, sizeof(char));
      sprintf (name,"%s.%d", imgName, number);
      file = fopen (name, "wb");
      if (!file)
        pipsPrintError (-1, PipsErrorFileWrite, name, PipsExit);
      free (name);

/*
 * Write file header
 */
      pipsWritePnmHeader (file, imgType, width, height);

/*
 * Query all image data values and write to the image file
 * Handles 0 to 3 dimensional data.
 */
      start [0] = number;
      for (y=0; y < height; y++) {
        start [1] = y;
        for (x=0; x < width; x++) {
          start [2] = x;
          start [3] = x;   /* Relevant only for 0-dimensional variables */
          ncvarget (ncId, dataId, &(start[3-ndims]), &(edge[3-ndims]), data);
          pipsWritePnmValue (file, imgType, data);
        } /* for (x) */
      } /* for (y) */

      number++;

/*
 * Close image file
 */
      fclose (file);
    } /* while (number) */

  } /* while pipsVarInq() */

  return (PipsOk);
}  /* end pipsExtPnm */


/* end pipsExtPnm.c */
