/* -*- C++ -*-
   XGContext - Drawing context using the XGPS Library.

   Copyright (C) 1998,1999 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@boulder.colorado.edu>
   Date: Nov 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include "gnustep/xgps/XGContext.h"
#include "gnustep/xgps/XGDrawObject.h"
#include "gnustep/xgps/XGGState.h"
#include "SharedX/xrtools.h"
#include <AppKit/AppKitExceptions.h>
#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSView.h>
#include <AppKit/NSWindow.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSData.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSString.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSDebug.h>

#include "gnustep/xgps/XGContextPrivate.h"
#include "XGDrawingEngine.h"

#include "math.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>


#define GSI_ARRAY_TYPES       GSUNION_OBJ

#if     GS_WITH_GC == 0
#define GSI_ARRAY_RELEASE(X)    [(X).obj release]
#define GSI_ARRAY_RETAIN(X)     [(X).obj retain]
#else
#define GSI_ARRAY_RELEASE(X)
#define GSI_ARRAY_RETAIN(X)     (X).obj
#endif

#ifdef GSIArray
#undef GSIArray
#endif
#include <base/GSIArray.h>

static NSMutableDictionary	*globalfontdir = nil;
Atom			WM_STATE;

extern int XGErrorHandler(Display *display, XErrorEvent *err);

@interface XGContext (Private)
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode; 
@end

@interface XGContext (PrivateDPSOps)
- (void) _checkFontDir;
- (void) _findXFont: (font_t *)font;
@end

#define XDPY (((RContext *)context)->dpy)
#define XSCR (((RContext *)context)->screen_number)

@implementation XGContext 

+ (Display*) currentXDisplay
{
  return [(XGContext*)[self currentContext] xDisplay];
}

- (RContextAttributes *) _getXDefaults
{
  int dummy;
  RContextAttributes *attribs;

  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
  attribs = (RContextAttributes *)malloc(sizeof(RContextAttributes));

  attribs->flags = 0;
  if ([defaults boolForKey: @"NSDefaultVisual"])
    attribs->flags |= RC_DefaultVisual;
  if ((dummy = [defaults integerForKey: @"NSDefaultVisual"]))
    {
      attribs->flags |= RC_VisualID;
      attribs->visualid = dummy;
    }
  if ((dummy = [defaults integerForKey: @"NSColorsPerChannel"]))
    {
      attribs->flags |= RC_ColorsPerChannel;
      attribs->colors_per_channel = dummy;
    }
  /*
  if ((str = [defaults stringForKey: @"NSColormapStyle"]))
    {
      attribs->flags |= RC_ColormapStyle;
      if ([str isEqual: @"none"])
        attribs->colormap_style = no_colormap;
      else if ([str isEqual: @"four"])
        attribs->colormap_style = four_colormap;
      else if ([str isEqual: @"gray"])
        attribs->colormap_style = gray_colormap;
      else
        attribs->colormap_style = rgb_colormap;
    }
  if ([defaults boolForKey: @"NSPrivateColormap"])
    attribs->flags |= RC_PrivateColormap;
  */

  return attribs;
}

- _initXContext
{
  Display		*dpy;
  int			screen_number;
  NSString		*display_name;
  RContext		*rcontext;
  RContextAttributes	*attribs;
  XColor		testColor;
  unsigned char		r, g, b;
  
  display_name = [context_info objectForKey: @"DisplayName"];
  if (display_name)
    {
      dpy = XOpenDisplay([display_name cString]);
    }
  else
    { 
      dpy = XOpenDisplay(NULL);
    }

  /* Use the fact that the screen number is specified like an extension
     e.g. hostname:0.1 */
  screen_number = [[display_name pathExtension] intValue];

  if (dpy == NULL)
    {
      char *dname = XDisplayName([display_name cString]);
      DPS_FATAL(DPSconfigurationerror, @"Unable to connect to X Server %s", 
		dname);
    }

  /* Get the visual information */
  attribs = NULL;
  //attribs = [self _getXDefaults];
  rcontext = RCreateContext(dpy, screen_number, attribs);
  context  = (void *)rcontext;

  /*
   * If we have shared memory available, only use it when the XGPS-Shm
   * default is set to YES
   */
  if (rcontext->attribs->use_shared_memory == True
    && [[NSUserDefaults standardUserDefaults] boolForKey: @"XGPS-Shm"] != YES)
    rcontext->attribs->use_shared_memory = False;

  /* Create a window for initial drawing */
  /* This window is never mapped */
  appRootWindow = XCreateSimpleWindow(XDPY, 
				      RootWindow(XDPY, XSCR),
				      0, 0, 100, 100, 1, 1,
				      rcontext->black);

  /*
   *	Crude tests to see if we can accelerate creation of pixels from
   *	8-bit red, green and blue color values.
   */
  if (rcontext->depth == 12 || rcontext->depth == 16)
    {
      drawMechanism = XGDM_FAST16;
      r = 8;
      g = 9;
      b = 7;
      testColor.pixel = (((r << 5) + g) << 6) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 11) != r)
	|| ((testColor.green >> 11) != g)
	|| ((testColor.blue >> 11) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 16-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else if (rcontext->depth == 15)
    {
      drawMechanism = XGDM_FAST15;
      r = 8;
      g = 9;
      b = 7;
      testColor.pixel = (((r << 5) + g) << 5) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 11) != r)
	|| ((testColor.green >> 11) != g)
	|| ((testColor.blue >> 11) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 15-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else if (rcontext->depth == 24 || rcontext->depth == 32)
    {
      drawMechanism = XGDM_FAST32;
      r = 32;
      g = 33;
      b = 31;
      testColor.pixel = (((r << 8) + g) << 8) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 8) == r)
        && ((testColor.green >> 8) == g)
        && ((testColor.blue >> 8) == b))
	{
	  drawMechanism = XGDM_FAST32;
	}
      else if (((testColor.red >> 8) == b)
	&& ((testColor.green >> 8) == g)
	&& ((testColor.blue >> 8) == r))
	{
	  drawMechanism = XGDM_FAST32_BGR;
	}
      else
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 32-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else
    {
      NSLog(@"WARNING - XGContext is unable to use a "
	@"fast algorithm for writing to the display on "
	@"this host - perhaps you'd like to adjust the code "
	@"to work ... and submit a patch.");
      drawMechanism = XGDM_PORTABLE;
    }

  XSetErrorHandler(XGErrorHandler);

  return self;
}

- (id) initWithContextInfo: (NSDictionary *)info
{
  NSZone	*z = [self zone];

  alloc_mode = alloc_local;
  /* Initialize lists and stacks */
  windowList = [[NSMutableArray allocWithZone: z] initWithCapacity: 8];
  opstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, z, 2);
  gstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)gstack, z, 2);
  ulist  = [[NSMutableArray allocWithZone: z] initWithCapacity: 32];
  
  [self _initXContext];

  /* Create a default gstate */
  gstate = [CTXT_OBJ_ALLOC(XGGState) initWithDrawContext: self];

  [super initWithContextInfo: info];

  /*
   * Make self the current context 'cos some of our initialization needs it.
   */
  [isa setCurrentContext: self];

  [self setupRunLoopInputSourcesForMode: NSDefaultRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSConnectionReplyMode]; 
  [self setupRunLoopInputSourcesForMode: NSModalPanelRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSEventTrackingRunLoopMode]; 

  WM_STATE = XInternAtom(XDPY, "WM_STATE", False);
  return self;
}

- (void) dealloc
{
  if (windowList)
    {
      /* This could be ugly, but...
           we need to make sure all our X resources are freed before
	   we close our display, thus destroy everything, then add ourselves
	   back to the autorelease list so we get called again a final time
      */
      GSIArrayEmpty((GSIArray)opstack);
      NSZoneFree([self zone], opstack);
      GSIArrayEmpty((GSIArray)gstack);
      NSZoneFree([self zone], gstack);
      DESTROY(windowList);
      DESTROY(ulist);
      [[self retain] autorelease];
    }
  else 
    {
      XDestroyWindow(XDPY, appRootWindow);
      XCloseDisplay(XDPY);
      [super dealloc];
    }
}


- (XGDrawMechanism) drawMechanism
{
  return drawMechanism;
}

- (BOOL)isDrawingToScreen
{
  return YES;
}

- (void *) xrContext
{
  return context;
}

- (XGGState *) xrCurrentGState
{
  return gstate;
}

- (XGGState *) xrGStateWithIdent: (int)gst
{
  XGGState *g;
  [self DPSexecuserobject: gst];
  ctxt_pop(g, opstack, XGGState);
  return g;
}

- (Region)viewclipRegion
{
  return ((RContext *)context)->viewclip;
}

- (void) setViewclipRegion: (Region)region;
{
  if (((RContext *)context)->viewclip)
    XDestroyRegion(((RContext *)context)->viewclip);
  ((RContext *)context)->viewclip = region;
}

- (Display *) xDisplay
{
  return XDPY;
}

- (void) setXDisplay: (Display *)xdisplay
{
  NSLog(@"Warning: resetting X display\n");
  XDPY = xdisplay;
  /* FIXME: Should update visualinfo, colormaps, etc? */
}

- (Window) xDisplayRootWindow
{
  return RootWindow(XDPY, XSCR);
}

- (Window) xAppRootWindow
{
  return appRootWindow;
}

- (XColor)xColorFromColor: (XColor)color
{
  Status ret;
  RColor rcolor;
  Colormap colormap = XDefaultColormap(XDPY, XSCR);
  XAllocColor(XDPY, colormap, &color);
  rcolor.red   = color.red / 256;
  rcolor.green = color.green / 256;
  rcolor.blue  = color.blue / 256;
  ret = RGetClosestXColor((RContext *)context, &rcolor, &color);
  if (ret == False)
    NSLog(@"Failed to alloc color (%d,%d,%d)\n",
          (int)rcolor.red, (int)rcolor.green, (int)rcolor.blue);
  return color;
}

- (void) flush
{
  XFlush(XDPY);
}

@end

@implementation XGContext (Ops)

/* ----------------------------------------------------------------------- */
/* Color operations */
/* ----------------------------------------------------------------------- */
- (void)DPScurrentcmykcolor: (float *)c : (float *)m : (float *)y : (float *)k 
{
  [gstate DPScurrentcmykcolor:c :m :y :k];
}

- (void)DPSsetcmykcolor: (float)c : (float)m : (float)y : (float)k 
{
  [gstate DPSsetcmykcolor:c :m :y :k];
}

/* ----------------------------------------------------------------------- */
/* Data operations */
/* ----------------------------------------------------------------------- */
- (void)DPSclear 
{
  GSIArrayEmpty((GSIArray)opstack);
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, [self zone], 2);
}

- (void)DPScleartomark 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);
  unsigned pos = count;

  while (pos > 0)
    {
      NSObject *obj = (GSIArrayItemAtIndex((GSIArray)opstack, --pos)).obj;

      if ([obj isKindOfClass: [XGDrawObject class]]
	  && [(XGDrawObject *)obj drawType] == ctxt_mark)
	{
	  int i;
	  for (i = count; i >=pos; i--)
	    GSIArrayRemoveItemAtIndex((GSIArray)opstack, i);
	  return;
	}
    }
}

- (void)DPScopy: (int)n 
{
}

- (void)DPScount: (int *)n 
{
  CHECK_NULL_OUTPUT(n);
  *n = GSIArrayCount((GSIArray)opstack);
}

- (void)DPScounttomark: (int *)n 
{
}

- (void)DPSdup 
{
}

- (void)DPSexch 
{
  int count = GSIArrayCount((GSIArray)opstack);
  if (count < 2)
    DPS_ERROR(DPSstackunderflow, @"Attempt to exch in empty stack");
  GSIArrayInsertItem((GSIArray)opstack, 
		 GSIArrayLastItem((GSIArray)opstack), count-2);
  GSIArrayRemoveLastItem((GSIArray)opstack);
}

- (void)DPSexecstack 
{
}

- (void)DPSget 
{
}

- (void)DPSindex: (int)i 
{
}

- (void)DPSmark 
{
}

- (void)DPSmatrix 
{
  ctxt_push([NSAffineTransform transform], opstack);
}

- (void)DPSnull 
{
}

- (void)DPSpop 
{
  id obj;
  ctxt_pop(obj, opstack, NSObject);
}

- (void)DPSput 
{
}

- (void)DPSroll: (int)n : (int)j 
{
}

/* ----------------------------------------------------------------------- */
/* Font operations */
/* ----------------------------------------------------------------------- */

/* Takes a font name stored in font.name and tries to find the X11 font
   information for that font. If the the font name does not begin with a
   '-', then it is assumed to be a standard PostScript name that should
   be converted to the X11 font name format. */
- (void) _findXFont: (font_t *)font
{
  int i, len;
  char   buf[1024];

  /* Try to convert the font name into a X11 font name */
  if (font->name[0] != '-')
    {
      char *s;
      len = strlen(font->name);
      font->internal_name = copy_string_buffer(font->name);
      for (i=0; i < len; i++)
	font->internal_name[i] = tolower(font->internal_name[i]);
      if (font->weight == 0)
	font->weight = "medium";
      if (font->slant == 0)
	font->slant  = 'r';
      if (font->size == 0)
	font->size   = 120;
      s = strchr(font->internal_name, '-');
      if (s)
	{
	  switch(*s) 
	    {
	    case 'b': font->weight = "bold"; break;
	    case 'd': font->weight = "demibold"; break;
	    case 'o': font->slant  = 'o'; break;
	    case 'i': font->slant  = 'i'; break;
	    default:
	    }
	}
      sprintf(buf, "-*-%s-%s-%c-*-%d-*", font->internal_name,
	      font->weight, font->slant, (int)font->size);
      free(font->internal_name);
      font->internal_name = NULL;
    }
  else
    strcpy(buf, font->name);

  font->info = XLoadQueryFont(XDPY, buf);
  if (font->info)
    {
      font->internal_name = copy_string_buffer(buf);
      font->id   = font->info->fid;
    }
}

- (void) _checkFontDir
{
  if (globalfontdir == nil)
    globalfontdir = [CTXT_OBJ_ALLOC(NSMutableDictionary) init];
  if (fontid == nil)
    fontid = [NSMutableArray arrayWithCapacity:32];
}

- (void)DPSdefineresource: (const char *)category 
{
  id obj;
  NSString *key;

  ctxt_pop(obj, opstack, NSNumber);
  ctxt_pop(key, opstack, NSNumber);
  if (strcmp(category, "Font") == 0)
    [self _checkFontDir];
  else
    DPS_ERROR(DPSundefined, [NSString stringWithCString: category]);

  [globalfontdir setObject: obj forKey: key];
  ctxt_push(obj, opstack);
}

- (void)DPSfindresource: (const char *)key : (const char *)category 
{
  id obj;

  if (strcmp(category, "Font") == 0)
    [self _checkFontDir];
  else
    DPS_ERROR(DPSundefined, @"Could not find key %s", key);

  obj = [globalfontdir objectForKey: [NSString stringWithCString:key]];

  if (obj == nil)
    {
      if (strcmp(category, "Font") == 0)
	{
	  int ident;
	  font_t font;
	  font.name = copy_string_buffer(key);
	  font.weight = 0;
	  font.slant = 0;
	  font.size = 0;
	  [self _findXFont: &font];
	  if (font.internal_name != NULL)
	    {
	      ident = [fontid count];
	      obj = [NSValue value: &font withObjCType: @encode(font_t)];
	      [fontid addObject: obj];
	      obj = [NSNumber numberWithLong: ident];
	    }
	}
      if (obj == nil)
	{
	  DPS_ERROR(DPSundefinedresource, @"Could not find resource %s", key);
	}
    }

  ctxt_push(obj, opstack);
}

- (void)DPSFontDirectory 
{
}

- (void)DPSISOLatin1Encoding 
{
}

- (void)DPSSharedFontDirectory 
{
}

- (void)DPSStandardEncoding 
{
}

- (void)DPScurrentcacheparams 
{
}

- (void)DPScurrentfont 
{
  int ident;
  //[self _checkFontDir];
  //ident = [fontid indexOfObject: fontobj];
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSdefinefont 
{
  [self DPSdefineresource: "Font"];
}

- (void)DPSfindfont: (const char *)name 
{
  [self DPSfindresource: name : "Font"];
}

- (void)DPSmakefont 
{
  id obj, fontobj;
  long ident;
  font_t font;

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);
  fontobj = [fontid objectAtIndex: ident];
  [fontobj getValue:&font];
  [gstate setFontStruct: font.info];
}

- (void)DPSscalefont: (float)size 
{
  id obj;
  long ident;
  font_t font;

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);

  obj = [fontid objectAtIndex:ident];
  [obj getValue:&font];
  font.size = size * 10;

  /* Try to find a X font that matches this size */
  [self _findXFont: &font];
  if (font.internal_name != NULL)
    {
      obj = [NSValue value: &font withObjCType: @encode(font_t)];
      ident = [fontid count];
      [fontid addObject: obj];
    }
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSselectfont: (const char *)name : (float)scale 
{
}

- (void)DPSsetfont: (int)f 
{
  id fontobj;
  font_t font;
  CHECK_INVALID_FONT(f);
  fontobj = [fontid objectAtIndex: f];
  [fontobj getValue:&font];
  [gstate setFontStruct: font.info];
}

- (void)DPSundefinefont: (const char *)name 
{
}

/* ----------------------------------------------------------------------- */
/* Gstate operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcat: (const float *)m 
{
  [gstate DPSconcat: m];
}

- (void)DPScurrentdash 
{
}

- (void)DPScurrentflat: (float *)flatness 
{
}

- (void)DPScurrentgray: (float *)gray 
{
  CHECK_NULL_OUTPUT(gray);
  [gstate DPScurrentgray: gray];
}

- (void)DPScurrentgstate: (int)gst 
{
  /* Associate/copy current gstate with gst */
  ctxt_push([NSNumber numberWithInt: gst], opstack);
  ctxt_push(gstate, opstack);
  [self DPSdefineuserobject];
  [self DPSexecuserobject: gst];
}

- (void)DPScurrenthalftone 
{
}

- (void)DPScurrenthalftonephase: (float *)x : (float *)y 
{
}

- (void)DPScurrenthsbcolor: (float *)h : (float *)s : (float *)b 
{
  CHECK_NULL_OUTPUT(h);
  CHECK_NULL_OUTPUT(s);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrenthsbcolor:h :s :b];
}

- (void)DPScurrentlinecap: (int *)linecap 
{
  [gstate DPScurrentlinecap: linecap];
}

- (void)DPScurrentlinejoin: (int *)linejoin 
{
  [gstate DPScurrentlinejoin: linejoin];
}

- (void)DPScurrentlinewidth: (float *)width 
{
  [gstate DPScurrentlinewidth: width];
}

- (void)DPScurrentmatrix 
{
  float m[6];
  id obj;
  ctxt_pop(obj, opstack, NSAffineTransform);
  [gstate DPScurrentmatrix: m];
  [obj setMatrix: m];
  ctxt_push(obj, opstack);
}

- (void)DPScurrentmiterlimit: (float *)limit 
{
}

- (void)DPScurrentpoint: (float *)x : (float *)y 
{
  CHECK_NULL_OUTPUT(x);
  CHECK_NULL_OUTPUT(y);
  [gstate DPScurrentpoint:x :y];
}

- (void)DPScurrentrgbcolor: (float *)r : (float *)g : (float *)b 
{
  CHECK_NULL_OUTPUT(r);
  CHECK_NULL_OUTPUT(g);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrentrgbcolor:r :g :b];
}

- (void)DPScurrentscreen 
{
}

- (void)DPScurrentstrokeadjust: (int *)b 
{
}

- (void)DPScurrenttransfer 
{
}

- (void)DPSdefaultmatrix 
{
}

- (void)DPSgrestore 
{
  RELEASE(gstate);
  ctxt_pop(gstate, gstack, XGGState);
  RETAIN(gstate);
}

- (void)DPSgrestoreall 
{
}

- (void)DPSgsave 
{
  ctxt_push(gstate, gstack);
  AUTORELEASE(gstate);
  gstate = [gstate copy];
}

- (void)DPSgstate 
{
  ctxt_push(AUTORELEASE([gstate copy]), opstack);
}

- (void)DPSinitgraphics 
{
  [gstate DPSinitgraphics];
}

- (void)DPSinitmatrix 
{
  [gstate DPSinitmatrix];
}

- (void)DPSrotate: (float)angle 
{
  [gstate DPSrotate: angle];
}

- (void)DPSscale: (float)x : (float)y 
{
  [gstate DPSscale:x :y];
}

- (void)DPSsetdash: (const float *)pat : (int)size : (float)offset 
{
}

- (void)DPSsetflat: (float)flatness 
{
}

- (void)DPSsetgray: (float)gray 
{
  [gstate DPSsetgray: gray];
}

- (void)DPSsetgstate: (int)gst 
{
  [self DPSexecuserobject: gst];
  RELEASE(gstate);
  ctxt_pop(gstate, opstack, XGGState);
  RETAIN(gstate);
}

- (void)DPSsethalftone 
{
}

- (void)DPSsethalftonephase: (float)x : (float)y 
{
}

- (void)DPSsethsbcolor: (float)h : (float)s : (float)b 
{
  [gstate DPSsethsbcolor:h :s :b];
}

- (void)DPSsetlinecap: (int)linecap 
{
  [gstate DPSsetlinecap: linecap];
}

- (void)DPSsetlinejoin: (int)linejoin 
{
  [gstate DPSsetlinejoin: linejoin];
}

- (void)DPSsetlinewidth: (float)width 
{
  [gstate DPSsetlinewidth: width];
}

- (void)DPSsetmatrix 
{
  float matrix[6];
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform);
  [obj getMatrix: matrix];
  [gstate DPSsetmatrix: matrix];
}

- (void)DPSsetmiterlimit: (float)limit 
{
}

- (void)DPSsetrgbcolor: (float)r : (float)g : (float)b 
{
  [gstate DPSsetrgbcolor:r :g :b];
}

- (void)DPSsetscreen 
{
}

- (void)DPSsetstrokeadjust: (int)b 
{
}

- (void)DPSsettransfer 
{
}

- (void)DPStranslate: (float)x : (float)y 
{
  [gstate DPStranslate:x :y];
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */
- (void)DPSflush
{
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcatmatrix
{
  NSAffineTransform *obj1, *obj2, *obj3;

  ctxt_pop(obj3, opstack, NSAffineTransform);
  ctxt_pop(obj2, opstack, NSAffineTransform);
  ctxt_pop(obj1, opstack, NSAffineTransform);
  [obj2 concatenateWith: obj1];
  obj3 = [obj2 copy];
  ctxt_push(obj3, opstack);
  RELEASE(obj3);
}

- (void)DPSdtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSidentmatrix 
{
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform);
  obj = [NSAffineTransform transform];
  [obj makeIdentityMatrix];
  ctxt_push(obj, opstack);
}

- (void)DPSidtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  [obj inverse];
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSinvertmatrix 
{
  NSAffineTransform *obj, *obj2;

  ctxt_pop(obj, opstack, NSAffineTransform);
  ctxt_pop(obj2, opstack, NSAffineTransform);
  [obj inverse];
  ctxt_push(obj, opstack);
}

- (void)DPSitransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  [obj inverse];
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPStransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, Nil);
    }
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

/* ----------------------------------------------------------------------- */
/* Opstack operations */
/* ----------------------------------------------------------------------- */

- (void)DPSdefineuserobject
{
  int n;
  id obj;
  NSNumber *number;
  ctxt_pop(obj, opstack, NSObject);
  ctxt_pop(number, opstack, NSNumber);
  n = [number intValue];
  if (n < 0)
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  else if (n < [ulist count])
    [ulist replaceObjectAtIndex: n withObject: obj];
  else
    {
      int i = n - [ulist count];
      while (i--)
	[ulist addObject: AUTORELEASE([[NSObject alloc] init])];
      [ulist addObject: obj];
    }
}

- (void)DPSexecuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  ctxt_push([ulist objectAtIndex: index], opstack);
}

- (void)DPSundefineuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  [ulist replaceObjectAtIndex: index
                   withObject: AUTORELEASE([[NSObject alloc] init])];
}

- (void)DPSgetboolean: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number boolValue];
}

- (void)DPSgetchararray: (int)size : (char *)s 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(s, [data bytes], size*sizeof(char));
}

- (void)DPSgetfloat: (float *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number floatValue];
}

- (void)DPSgetfloatarray: (int)size : (float *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(float));
}

- (void)DPSgetint: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number intValue];
}

- (void)DPSgetintarray: (int)size : (int *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(int));
}

- (void)DPSgetstring: (char *)s 
{
  NSString *str;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(str, opstack, NSString);
  strcpy(s, [str cString]);
}

- (void)DPSsendboolean: (int)it 
{
  ctxt_push([NSNumber numberWithBool: it], opstack);
}

- (void)DPSsendchararray: (const char *)s : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: s length: size*sizeof(char)];
  ctxt_push(data, opstack);
}

- (void)DPSsendfloat: (float)it 
{
  ctxt_push([NSNumber numberWithFloat: it], opstack);
}

- (void)DPSsendfloatarray: (const float *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(float)];
  ctxt_push(data, opstack);
}

- (void)DPSsendint: (int)it 
{
  ctxt_push([NSNumber numberWithInt: it], opstack);
}

- (void)DPSsendintarray: (const int *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(int)];
  ctxt_push(data, opstack);
}

- (void)DPSsendstring: (const char *)s 
{
  ctxt_push([NSString stringWithCString: s], opstack);
}

/* ----------------------------------------------------------------------- */
/* Paint operations */
/* ----------------------------------------------------------------------- */

- (void)DPSashow: (float)x : (float)y : (const char *)s 
{
  [gstate DPSashow: x : y : s];
}

- (void)DPSawidthshow: (float)cx : (float)cy : (int)c : (float)ax : (float)ay : (const char *)s 
{
  [gstate DPSawidthshow: cx : cy : c : ax : ay : s];
}

- (void)DPScopypage 
{
}

- (void)DPSeofill 
{
  [gstate DPSeofill];
}

- (void)DPSerasepage 
{
}

- (void)DPSfill 
{
  [gstate DPSfill];
}

- (void)DPSimage 
{
  id obj, key;
  NSMutableDictionary *dict;

  dict = [NSMutableDictionary dictionary];
  ctxt_pop(obj, opstack, NSNumber);
  [dict setObject: obj forKey: @"Width"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"Height"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"BitsPerComponent"];
  ctxt_pop(key, opstack, NSAffineTransform);
  [dict setObject: obj forKey: @"Matrix"];
  /* Fixme - handle multiple data sources */
  ctxt_pop(key, opstack, NSString);
  [dict setObject: obj forKey: @"DataSource"];
  ctxt_pop(key, opstack, NSNumber);

  [gstate DPSimage: dict];
}

- (void)DPSimagemask 
{
}

- (void)DPScolorimage
{
  id obj, key;
  NSMutableDictionary *dict;

  dict = [NSMutableDictionary dictionary];
  ctxt_pop(obj, opstack, NSNumber);
  [dict setObject: obj forKey: @"Width"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"Height"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"BitsPerComponent"];
  ctxt_pop(key, opstack, NSAffineTransform);
  [dict setObject: obj forKey: @"Matrix"];
  /* Fixme - handle multiple data sources */
  ctxt_pop(key, opstack, NSString);
  [dict setObject: obj forKey: @"DataSource"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"MultipleDataSources"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"SamplesPerPixel"];

  [gstate DPSimage: dict];
}

- (void)DPSalphaimage
{
  id obj, key;
  NSMutableDictionary *dict;

  dict = [NSMutableDictionary dictionary];
  ctxt_pop(obj, opstack, NSNumber);
  [dict setObject: obj forKey: @"Width"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"Height"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"BitsPerComponent"];
  ctxt_pop(key, opstack, NSAffineTransform);
  [dict setObject: obj forKey: @"Matrix"];
  /* Fixme - handle multiple data sources */
  ctxt_pop(key, opstack, NSString);
  [dict setObject: obj forKey: @"DataSource"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"MultipleDataSources"];
  ctxt_pop(key, opstack, NSNumber);
  [dict setObject: obj forKey: @"SamplesPerPixel"];

  [dict setObject: [NSNumber numberWithBool: YES] forKey: @"Alpha"];
  [gstate DPSimage: dict];
}

- (void)DPSkshow: (const char *)s 
{
}

- (void)DPSrectfill: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectfill:x :y :w :h];
}

- (void)DPSrectstroke: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectstroke:x :y :w :h];
}

- (void)DPSshow: (const char *)s 
{
  [gstate DPSshow: s];
}

- (void)DPSshowpage 
{
}

- (void)DPSstroke 
{
  [gstate DPSstroke];
}

- (void)DPSstrokepath 
{
  [gstate DPSstrokepath];
}

- (void)DPSueofill: (const char *)nums : (int)n : (const char *)op : (int)l 
{
}

- (void)DPSufill: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustroke: (const char *)nums   : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustrokepath: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSwidthshow: (float)x : (float)y : (int)c : (const char *)s 
{
  [gstate DPSwidthshow: x : y : c : s];
}

- (void)DPSxshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxshow: s : numarray : size];
}

- (void)DPSxyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxyshow: s : numarray : size];
}

- (void)DPSyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSyshow: s : numarray : size];
}

/* ----------------------------------------------------------------------- */
/* Path operations */
/* ----------------------------------------------------------------------- */

- (void)DPSarc: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarc: x : y : r : angle1 : angle2];
}

- (void)DPSarcn: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarc: x : y : r : angle1 : angle2];
}

- (void)DPSarct: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
{
  [gstate DPSarc: x1 : y1 : x2 : y2 : r];
}

- (void)DPSarcto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r : (float *)xt1 : (float *)yt1 : (float *)xt2 : (float *)yt2 
{
  [gstate DPSarcto: x1 : y1 : x2 : y2 : r : xt1 : yt1 : xt2 : yt2];
}

- (void)DPScharpath: (const char *)s : (int)b 
{
}

- (void)DPSclip 
{
  [gstate DPSclip];
}

- (void)DPSclippath 
{
  [gstate DPSclippath];
}

- (void)DPSclosepath 
{
  [gstate DPSclosepath];
}

- (void)DPScurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  [gstate DPScurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSeoclip 
{
  [gstate DPSeoclip];
}

- (void)DPSeoviewclip 
{
  [gstate DPSeoviewclip];
}

- (void)DPSflattenpath 
{
  [gstate DPSflattenpath];
}

- (void)DPSinitclip 
{
  [gstate DPSinitclip];
}

- (void)DPSinitviewclip 
{
  [gstate DPSinitviewclip];
}

- (void)DPSlineto: (float)x : (float)y 
{
  [gstate DPSlineto: x : y];
}

- (void)DPSmoveto: (float)x : (float)y 
{
  [gstate DPSmoveto: x : y];
}

- (void)DPSnewpath 
{
  [gstate DPSnewpath];
}

- (void)DPSpathbbox: (float *)llx : (float *)lly : (float *)urx : (float *)ury 
{
  [gstate DPSpathbbox: llx : lly : urx : ury];
}

- (void)DPSpathforall 
{
  [gstate DPSpathforall];
}

- (void)DPSrcurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  [gstate DPSrcurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSrectclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectclip: x : y : w : h];
}

- (void)DPSrectviewclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectviewclip: x : y : w : h];
}

- (void)DPSreversepath 
{
  [gstate DPSreversepath];
}

- (void)DPSrlineto: (float)x : (float)y 
{
  [gstate DPSrlineto: x : y];
}

- (void)DPSrmoveto: (float)x : (float)y 
{
  [gstate DPSrmoveto: x : y];
}

- (void)DPSsetbbox: (float)llx : (float)lly : (float)urx : (float)ury 
{
  [gstate DPSsetbbox: llx : lly : urx : ury];
}

- (void)DPSviewclip 
{
  [gstate DPSviewclip];
}

- (void)DPSviewclippath 
{
}

/* ----------------------------------------------------------------------- */
/* Window system ops */
/* ----------------------------------------------------------------------- */

- (void) DPScurrentdrawingfunction: (int *)function
{
}

- (void) DPScurrentgcdrawable: (void **)gc : (void **)draw : (int *)x : (int *)y
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPScurrentgcdrawablecolor: (void **)gc : (void **)draw : (int *)x 
				  : (int *)y : (int *)colorInfo
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
  if (colorInfo)
    *colorInfo = 0;
}

- (void) DPScurrentoffset: (int *)x : (int *)y
{
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPSsetdrawingfunction: (int) function
{
  XGCValues values;
  values.function = function;
  [gstate setGCValues: values withMask: (GCFunction | GCForeground)];
}

- (void) DPSsetgcdrawable: (void *)gc : (void *)draw : (int)x : (int)y
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetgcdrawablecolor: (void *)gc : (void *)draw : (int)x : (int)y
				  : (const int *)colorInfo
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetoffset: (short int)x : (short int)y
{
  [gstate setOffset: NSMakePoint(x,y)];
}

- (void) DPSsetrgbactual: (double)r : (double)g : (double)b : (int *)success
{
  unsigned long pixel;
  pixel = xrAllocActualRGB((RContext *)context, r, g, b);
  if (pixel > 0)
    [gstate DPSsetrgbcolor: r : g : b];
  *success = (pixel > 0);
}

/* Add the current gstate to the gstate index (if not there already)
   and return its index */
- (void) DPScapturegstate: (int *)gst
{
  CHECK_NULL_OUTPUT(gst);
  if ([ulist indexOfObject: gstate] == NSNotFound)
    [ulist addObject: gstate];
  *gst = [ulist indexOfObject: gstate];
}

/*-------------------------------------------------------------------------*/
/* Graphics Extension Ops */
/*-------------------------------------------------------------------------*/
- (void) DPScomposite: (float)x : (float)y : (float)w : (float)h : (int)gstateNum : (float)dx : (float)dy : (int)op
{
  NSRect rect;
  NSPoint p;
  XGCValues gcv;
  XGGState *g = gstate;

  if (gstateNum)
    {
      Drawable source;
      [self DPSexecuserobject: gstateNum];
      ctxt_pop(g, opstack, XGGState);
      source = [g drawable];
      if (!source) 
        DPS_ERROR(DPSundefined, @"Composite drawable not realized\n");
    }

  rect = NSMakeRect(x, y, w, h);
  p    = NSMakePoint(dx, dy);

#if 0
  [gstate compositeGState: g fromRect: rect toPoint: p op: op];
#else
  switch (op)
    {
    case   NSCompositeClear:
      gcv.function = GXclear;
      break;
    case   NSCompositeCopy:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeXOR:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusDarker:
      gcv.function = GXcopy;
      break;
    case   NSCompositeHighlight:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusLighter:
      gcv.function = GXcopy;
      break;
    default:
      gcv.function = GXcopy;
      break;
    }
  if (gcv.function != GXcopy)
    [gstate setGCValues: gcv withMask: GCFunction];
  [gstate copyBits: g fromRect: rect toPoint: p];
  if (gcv.function != GXcopy)
    {
      gcv.function = GXcopy;
      [gstate setGCValues: gcv withMask: GCFunction];
    }
#endif
}

- (void) DPScompositerect: (float)x : (float)y : (float)w : (float)h : (int)op
{
  float gray;
  XGCValues gcv;

  [gstate DPScurrentgray: &gray];
  if (fabs(gray - 0.667) < 0.005)
    [gstate DPSsetgray: 0.333];
  else    
    [gstate DPSsetrgbcolor: 0.121 : 0.121 : 0];

  /* FIXME: Really need alpha dithering to do this right - combine with
     XGBitmapImageRep code? */
  switch (op)
    {
    case   NSCompositeClear:
      gcv.function = GXclear;
      break;
    case   NSCompositeCopy:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeXOR:
      gcv.function = GXcopy;
      break;
    case   NSCompositePlusDarker:
      gcv.function = GXcopy;
      break;
    case   NSCompositeHighlight:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusLighter:
      gcv.function = GXcopy;
      break;
    default:
      gcv.function = GXcopy;
      break;
    }
  [gstate setGCValues: gcv withMask: GCFunction];
  [gstate DPSrectfill: x : y : w : h];

  gcv.function = GXcopy;
  [gstate setGCValues: gcv withMask: GCFunction];
  [gstate DPSsetgray: gray];
}

- (void) DPSdissolve: (float)x : (float)y : (float)w : (float)h : (int)gstateNum
 : (float)dx : (float)dy : (float)delta
{
  return [self DPScomposite: x : y : w : h : gstateNum : dx : dy 
	 : NSCompositeSourceOver];
}

- (void) DPSreadimage
{
}

- (void) DPSsetalpha: (float)a
{
  [gstate DPSsetalpha: a];
}

- (void) DPScurrentalpha: (float *)alpha
{
  [gstate DPScurrentalpha: alpha];
}

@end
