/* 
   PXKCursor.m

   NSCursor for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: March 1996
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "config.h"
#include <X11/cursorfont.h>

#include <gnustep/xdps/NSDPSContext.h>
#include <AppKit/NSApplication.h>

//
// Backend structure for PXKCursor
//
typedef struct _PXKCursor_struct
{
  Cursor cursor;
  BOOL is_valid;
} PXKCursor_struct;

#define XCURSOR (((PXKCursor_struct *)be_cursor_reserved)->cursor)
#define XVALID (((PXKCursor_struct *)be_cursor_reserved)->is_valid)

static char xdps_blank_cursor_bits [] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//
// Class variables
//
static NSCursor *xdps_blank_cursor = nil;
static NSCursor *xdps_hidden_cursor = nil;

@implementation PXKCursor

//
// X Window private methods
//
- (void)setXCursor:(Cursor)cursor
{
  XCURSOR = cursor;
  XVALID = YES;
}

- (Cursor)xCursor
{
  // If the xCursor isn't valid
  // then try to create it from the image
  if (!XVALID)
    {
#if 0
      NSApplication *theApp = [NSApplication sharedApplication];
      NSWindow *w = [theApp mainWindow];
      Pixmap shape, mask;
      XColor black, white;

      shape = XCreatePixmapFromBitmapData(xDisplay, [(PXKWindow *)w xWindow],
					  cursor_bits, 16, 16, 1, 0, 1);
      mask = XCreatePixmapFromBitmapData(xDisplay, [(PXKWindow *)w xWindow],
					 cursor_bits, 16, 16, 1, 0, 1);
      black = [(PXKColor *)[NSColor blackColor] xColor];
      white = [(PXKColor *)[NSColor whiteColor] xColor];

      XCURSOR = XCreatePixmapCursor(xDisplay, shape, mask, 
				    &white, &black, 0, 0);
      XVALID = YES;
#endif
    }

  return XCURSOR;
}

- (BOOL)xCursorIsValid
{
  return XVALID;
}

//
// Class methods
//

//
// Setting the Cursor
//
+ (void)hide
{
  // Create the blank cursor if necessary
  if (!xdps_blank_cursor)
    {
      Display *xDisplay = [(NSDPSContext *)[NSDPSContext currentContext]
					    xDisplay];
      NSApplication *theApp = [NSApplication sharedApplication];
      NSWindow *w = [theApp mainWindow];
      Pixmap shape, mask;
      XColor black, white;
      Cursor c;

      Window win;

      if (w)
	win = [(PXKWindow *)w xWindow];
      else
	{
	  NSDPSContext* context = (NSDPSContext *)[NSDPSContext currentContext];
	  win = [context xAppRootWindow];
	}
      
      xdps_blank_cursor = [[NSCursor alloc] initWithImage: nil];
      shape = XCreatePixmapFromBitmapData(xDisplay, win,
					  xdps_blank_cursor_bits, 
					  16, 16, 1, 0, 1);
      mask = XCreatePixmapFromBitmapData(xDisplay, win,
					 xdps_blank_cursor_bits, 
					 16, 16, 1, 0, 1);
      black = [(PXKColor *)[NSColor blackColor] xColor];
      white = [(PXKColor *)[NSColor whiteColor] xColor];
	
      c = XCreatePixmapCursor(xDisplay, shape, mask, &white, &black, 0, 0);
      [(PXKCursor *)xdps_blank_cursor setXCursor: c];
    }

  // If the cursor is already hidden then don't hide again.
  if (xdps_blank_cursor == [self currentCursor])
    return;

  // Save the current cursor and
  // set the blank cursor
  xdps_hidden_cursor = [self currentCursor];
  [xdps_blank_cursor set];
}

+ (void)unhide
{
  // if no blank cursor then we couldn't have hidden
  if (!xdps_blank_cursor)
    return;

  // If cursor isn't hidden then just return
  if (xdps_blank_cursor != [self currentCursor])
    return;

  // Set back to the current cursor
  [xdps_hidden_cursor set];
}

//
// Getting the Cursor
//
+ (NSCursor *)arrowCursor
{
  NSCursor *cur;
  Cursor c;
  Display *xDisplay = [(NSDPSContext *)[NSDPSContext currentContext]
					xDisplay];

  cur = [[[NSCursor alloc] initWithImage: nil] autorelease];

  // Create standard left pointer arrow
  c = XCreateFontCursor(xDisplay, XC_left_ptr);
  [(PXKCursor *)cur setXCursor: c];

  return cur;
}

+ (NSCursor *)IBeamCursor
{
  NSCursor *cur;
  Cursor c;
  Display *xDisplay = [(NSDPSContext *)[NSDPSContext currentContext]
					xDisplay];

  cur = [[[NSCursor alloc] initWithImage: nil] autorelease];

  // Create standard I beam
  c = XCreateFontCursor(xDisplay, XC_xterm);
  [(PXKCursor *)cur setXCursor: c];

  return cur;
}

//
// Instance methods
//

//
// Initializing a New NSCursor Object
//
- (id)initWithImage:(NSImage *)newImage
{
  [super initWithImage: newImage];

  // Allocate backend structure
  be_cursor_reserved = malloc(sizeof(PXKCursor_struct));
  XCURSOR = 0;
  XVALID = NO;

  return self;
}

- (void)dealloc
{
  [super dealloc];

  // Free backend structure
  free(be_cursor_reserved);
}

@end

//
// Backend Methods
//
@implementation PXKCursor (GNUstepBackend)

+ (void)currentCursorHasChanged
{
  NSCursor *c = [self currentCursor];
  BOOL isValidXCursor = [(PXKCursor *)c xCursorIsValid];

  if (isValidXCursor)
    {
      Display *xDisplay = [(NSDPSContext *)[NSDPSContext currentContext]
					    xDisplay];
      Cursor xc = [(PXKCursor *)c xCursor];
      NSApplication *theApp = [NSApplication sharedApplication];
      NSWindow *w = [theApp mainWindow];
      Window win;
      
      if (w)
	win = [(PXKWindow *)w xWindow];
      else
	{
	  NSDPSContext* context =(NSDPSContext *)[NSDPSContext currentContext];
	  win = [context xAppRootWindow];
	}

      XDefineCursor(xDisplay, win, xc);
    }
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [PXKCursor class])
    return [super class];
  return [self class];
}

@end
