/* 
   PXKApplication.m

   NSApplication for GNUstep GUI X/DPS Backend

   The NSApplication class manages the main event loop of
   the program, keeps track of the application's windows
   and which one is the key window.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Ovidiu Predescu <ovidiu@net-community.com>
   Date: June 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: August 1998

   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <gnustep/xdps/NSDPSContext.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSRunLoop.h>
#include <Foundation/NSTimer.h>
#include <Foundation/NSNotification.h>

#include <AppKit/NSWindow.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSCStringText.h>

@interface PXKIconW : PXKWindow
@end

@implementation PXKIconW

- (BOOL) canBecomeMainWindow
{
  return NO;
}

- (BOOL) canBecomeKeyWindow
{
  return NO;
}

- (void) initDefaults
{
  [super initDefaults];
  menu_exclude = YES;           // Don't show in windows menu.
  is_released_when_closed = NO;
}

- (id) initWithContentRect: (NSRect)contentRect
		 styleMask: (unsigned int)aStyle
		   backing: (NSBackingStoreType)bufferingType
		     defer: (BOOL)flag
		    screen: (NSScreen*)aScreen
{
  XSetWindowAttributes winattrs;
  unsigned long valuemask;

  self = [super initWithContentRect: contentRect
			  styleMask: aStyle
			    backing: bufferingType
			      defer: flag
			     screen: aScreen];

  valuemask = (CWSaveUnder|CWOverrideRedirect);
  winattrs.save_under = True;
  winattrs.override_redirect = True;
  XChangeWindowAttributes ([NSDPSContext currentXDisplay], [self xWindow],
                            valuemask, &winattrs);
  [[NSApplication sharedApplication] removeWindowsItem: self];

  return self;
}
@end

@interface PXKAppIconView : NSView
- (void) setImage: (NSImage *)anImage;
@end

@implementation PXKAppIconView

// Class variables
static NSCell* dragCell = nil;
static NSCell* tileCell = nil;

+ (void) initialize
{
  NSImage	*defImage = [NSImage imageNamed: @"GNUstep"];
  NSImage	*tileImage = [NSImage imageNamed: @"common_Tile"];

  dragCell = [[NSCell alloc] initImageCell: defImage];
  [dragCell setBordered: NO];
  tileCell = [[NSCell alloc] initImageCell: tileImage];
  [tileCell setBordered: NO];
}

- (BOOL) acceptsFirstMouse: (NSEvent*)theEvent
{
  return YES;
}

- (void) drawRect: (NSRect)rect
{                                                
  [tileCell drawWithFrame: rect inView: self];
  [dragCell drawWithFrame: rect inView: self];
}

- (void) mouseDown: (NSEvent*)theEvent
{
  if ([theEvent clickCount] >= 2)
    {
      PXKWindow *kw = (PXKWindow *)[NSApp keyWindow];

      if (!kw)
	kw = (PXKWindow *)[self window];

      [NSApp unhide: self];
      if (kw)
	{
	  Display	*xDisplay = [NSDPSContext currentXDisplay];

	  [kw orderFrontRegardless];
	  XFlush(xDisplay);
	  XSetInputFocus(xDisplay, [kw xWindow], RevertToParent, CurrentTime);
	}
    }
  else
    {
      NSPoint	lastLocation;
      NSPoint	location;
      unsigned	eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
				| NSPeriodicMask | NSRightMouseUpMask;
      NSDate	*theDistantFuture = [NSDate distantFuture];
      NSApplication *theApp = [NSApplication sharedApplication];
      BOOL	done = NO;

      lastLocation = [theEvent locationInWindow];
      [NSEvent startPeriodicEventsAfterDelay: 0.02 withPeriod: 0.02];

      while (!done)
	{
	  theEvent = [theApp nextEventMatchingMask: eventMask
					 untilDate: theDistantFuture
					    inMode: NSEventTrackingRunLoopMode
					   dequeue: YES];
	
	  switch ([theEvent type])
	    {
	      case NSRightMouseUp:
	      case NSLeftMouseUp:
	      /* right mouse up or left mouse up means we're done */
		done = YES;
		break;
	      case NSPeriodic:
		location = [window mouseLocationOutsideOfEventStream];
		if (NSEqualPoints(location, lastLocation) == NO)
		  {
		    NSPoint	origin = [window frame].origin;

		    origin.x += (location.x - lastLocation.x);
		    origin.y += (location.y - lastLocation.y);
		    [window setFrameOrigin: origin];
		  }
		break;

	      default:
		break;
	    }
	}
      [NSEvent stopPeriodicEvents];
    }
}                                                        

- (void) setImage: (NSImage *)anImage
{
  [self lockFocus];
  [tileCell drawWithFrame: NSMakeRect(0,0,64,64) inView: self];
  [dragCell setImage: anImage];
  [dragCell drawWithFrame: NSMakeRect(8,8,48,48) inView: self];
  [window flushWindow];
  [self unlockFocus];
}

@end

@interface PXKApplication : NSApplication
- (void) setGrabXWindow: (Window)w;
- (Window) grabXWindow;
@end

//
// Backend structure for PXKApplication
//
typedef struct _PXKApp_struct
{
  Window grab_window;
  Display *xDisplay;
  PXKWindow *iconWindow;

} PXKApp_struct;

#define PXKRAB (((PXKApp_struct *)be_app_reserved)->grab_window)
#define PXKDISPLAY (((PXKApp_struct *)be_app_reserved)->xDisplay)
#define PXKICONW (((PXKApp_struct *)be_app_reserved)->iconWindow)

//
// Class variables
//
static Atom WM_STATE;


@implementation PXKApplication

//
// Class methods
//
+ (void)initialize
{
	if (self == [PXKApplication class])
		[self setVersion:1];							// Initial version
}

//
// Instance methods
//
- init
{
  PXKAppIconView	*iv;
  NSDPSContext	*context;
  Display	*xDisplay;

  be_app_reserved = malloc(sizeof(PXKApp_struct));	// Allocate backend
  PXKRAB = 0;

  [super init];

  PXKDISPLAY = [NSDPSContext currentXDisplay];
  WM_STATE = XInternAtom(PXKDISPLAY, "WM_STATE", False);

  context = (NSDPSContext*)[NSDPSContext currentContext];
  xDisplay = [context xDisplay];

  if (app_icon == nil)
    app_icon = [[NSImage imageNamed: @"GNUstep"] retain];

  PXKICONW = [[PXKIconW alloc] initWithContentRect: NSMakeRect(0,0,64,64)
				styleMask: NSBorderlessWindowMask
				  backing: NSBackingStoreRetained
				    defer: NO
				   screen: nil];

  iv = [[PXKAppIconView alloc] initWithFrame: NSMakeRect(0,0,64,64)];
  [PXKICONW setContentView: iv];
  [iv setImage: app_icon];
  [PXKICONW orderFrontRegardless];
  XSetInputFocus(xDisplay, [PXKICONW xWindow], RevertToParent, CurrentTime);

  return self;
}

- (void)dealloc
{
	free(be_app_reserved);								// Deallocate backend
														// structure
	[super dealloc];
}

- (void) setGrabXWindow: (Window)w
{
  PXKRAB = w;
}

- (Window) grabXWindow
{
  return PXKRAB;
}

- (NSArray*) windows
{
  NSDPSContext	*ctxt = (NSDPSContext*)[NSDPSContext currentContext];

  return ctxt->windowList;
}

- (Class) classForCoder: (NSCoder*)aCoder
{
  if ([self class] == [PXKApplication class])
    return [super class];
  return [self class];
}

- (void) setApplicationIconImage: (NSImage*)anImage
{
  [super setApplicationIconImage: anImage];
  [[PXKICONW contentView] setImage: anImage];
  [[PXKICONW contentView] setNeedsDisplay: YES];
}

- (void) activateIgnoringOtherApps: (BOOL)flag
{
  if (app_is_active == NO)
    {
      [super activateIgnoringOtherApps: flag];
      if (app_is_active == YES)
	{
	  PXKWindow	*kw = (PXKWindow *)[self keyWindow];

	  if (kw == nil || [kw isVisible] == NO)
	    {
	      kw = (PXKWindow *)PXKICONW;
	    }
	  XFlush(PXKDISPLAY);
	  XSetInputFocus(PXKDISPLAY, [kw xWindow], RevertToParent, CurrentTime);
	}
    }
}


@end
