/* 
   PXKWindow.m

   NSWindow for GNUstep GUI X/DPS Backend.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: February 1996
   Author:  Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: May 1997
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <stdlib.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSMapTable.h>
#include <Foundation/NSNotification.h>

#include <AppKit/NSCursor.h>

#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKView.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKScreen.h>
#include <gnustep/xdps/PXKColor.h>
#include <gnustep/xdps/PXKApplication.h>
#include <gnustep/xdps/PXKMenu.h>
#include <gnustep/xdps/PXKWindowView.h>

#include "general.h"


#ifndef LIB_FOUNDATION_LIBRARY

static void __NSRetainNothing(void *table, const void *anObject)
{
}

static void __NSReleaseNothing(void *table, void *anObject)
{
}

static NSString* __NSDescribeObjects(void *table, const void *anObject)
{
    return [(NSObject*)anObject description];
}

static const NSMapTableValueCallBacks NSNonRetainedObjectMapValueCallBacks = {
    (void (*)(NSMapTable *, const void *))__NSRetainNothing,
    (void (*)(NSMapTable *, void *))__NSReleaseNothing,
    (NSString *(*)(NSMapTable *, const void *))__NSDescribeObjects
}; 

#endif /* LIB_FOUNDATION_LIBRARY */


#define XWINDOW (((PXKWindow_struct *)be_wind_reserved)->xWindow)
#define XGC (((PXKWindow_struct *)be_wind_reserved)->context)
#define XEXPOSE (((PXKWindow_struct *)be_wind_reserved)->is_exposed)
#define exposedRectangles \
  (((PXKWindow_struct *)be_wind_reserved)->exposedRectangles)
#define xPixmap \
  (((PXKWindow_struct *)be_wind_reserved)->xPixmap)
#define X (((PXKWindow_struct *)be_wind_reserved)->x)
#define Y (((PXKWindow_struct *)be_wind_reserved)->y)
#define WIDTH (((PXKWindow_struct *)be_wind_reserved)->width)
#define HEIGHT (((PXKWindow_struct *)be_wind_reserved)->height)
#define DEPTH (((PXKWindow_struct *)be_wind_reserved)->depth)
#define region (((PXKWindow_struct *)be_wind_reserved)->region)

@implementation PXKWindow

static NSMapTable* windowsForXWindows = NULL;
static NSMapTable* windowsForNumbers = NULL;
static BOOL _needsFlushWindows = YES;

+ (void)initialize
{
  windowsForXWindows = NSCreateMapTable (NSNonOwnedPointerMapKeyCallBacks,
					 NSNonRetainedObjectMapValueCallBacks,
					 20);
  windowsForNumbers = NSCreateMapTable (NSIntMapKeyCallBacks,
					NSNonRetainedObjectMapValueCallBacks,
					20);
}

+ (PXKWindow*)windowForXWindow:(Window)xWindow
{
  return NSMapGet (windowsForXWindows, (void*)xWindow);
}

+ (int)numberForXWindow:(Window)xWindow
{
  return [[self windowForXWindow:xWindow] windowNumber];
}

+ (NSWindow*)windowWithNumber:(int)windowNumber
{
  return NSMapGet (windowsForNumbers, (void*)windowNumber);
}

+ (void)_setNeedsFlushWindows:(BOOL)flag
{
	_needsFlushWindows = flag;
}

+ (BOOL)_needsFlushWindows		{ return _needsFlushWindows; }

+ (NSView *)_windowViewWithFrame:(NSRect)frameRect			// create the top
{															// most view in the
	return [[PXKWindowView alloc] initWithFrame:frameRect];	// view's heirarchy
}

- (id)initWithContentRect:(NSRect)contentRect
		styleMask:(unsigned int)aStyle
		  backing:(NSBackingStoreType)bufferingType
		    defer:(BOOL)flag
		   screen:aScreen;
{
  // Allocate the back-end structure
  be_wind_reserved = calloc(1, sizeof(PXKWindow_struct));
  exposedRectangles = [NSMutableArray new];
  region = XCreateRegion();
  xPixmap = None;

  /* Provide some reasonable values for the contentRect size so the XWindow
     server doesn't complain. */
  if (!contentRect.size.width)
    contentRect.size.width = 200;
  if (!contentRect.size.height)
    contentRect.size.height = 100;

  [super initWithContentRect:contentRect styleMask:aStyle
	 backing:bufferingType defer:flag screen:aScreen];

  NSDebugLog(@"PXKWindow default initializer\n");

  [self createXWindow: contentRect];

  /* Insert the window into the mapping from X windows to NS windows */
  NSMapInsert (windowsForXWindows, (void*)XWINDOW, self);
  NSMapInsert (windowsForNumbers, (void*)[self windowNumber], self);

#ifdef TRANSPARENCY
  [content_view setOpaque:YES];
#endif

  NSDebugLog(@"PXKWindow end of default initializer\n");
  return self;
}

- (void)dealloc
{
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];

  /* Remove from X to NS windows mapping */
  NSMapRemove (windowsForXWindows, (void*)XWINDOW);
  NSMapRemove (windowsForNumbers, (void*)[self windowNumber]);

  // Destroy the X Window
  if (XWINDOW)
    {
      XDestroyWindow(xDisplay, XWINDOW);
      XWINDOW = 0;
    }

  [exposedRectangles release];
  if (xPixmap) {
    DPSWaitContext ([(PXKDPSContext *)[NSDPSContext currentContext]
					  xDPSContext]);
    XFreePixmap (xDisplay, xPixmap);
    xPixmap = None;
  }

  XDestroyRegion (region);

  // Release the back-end structure
  free(be_wind_reserved);

  [super dealloc];
}

- (void)setTitle:(NSString*)aString
{
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];
  XTextProperty windowName;

  [super setTitle:aString];

  if (window_title && XWINDOW)
    {
      const char* newTitle = [window_title cString];
      XStringListToTextProperty((char**)&newTitle, 1, &windowName);
      XSetWMName(xDisplay, XWINDOW, &windowName);
      XSetWMIconName(xDisplay, XWINDOW, &windowName);
    }
}

- (void)orderOut:sender
{
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];

  [super orderOut:sender];

  if (XEXPOSE && XWINDOW)
    {
      XUnmapWindow(xDisplay, XWINDOW);
      /* xxx Should we also set the PS X drawable to null? */
    }
  XEXPOSE = NO;
}

- (void)orderFront:sender
{
  PXKDPSContext *cur = [NSDPSContext currentContext];
  Display *xDisplay = [(PXKDPSContext *)cur xDisplay];

  [super orderFront:sender];

  if (XWINDOW)
    XMapWindow(xDisplay, XWINDOW);
  [self flushWindow];
}

- (void)setContentViewSize:(NSSize)aSize
{
  NSRect r;

  r.origin = NSZeroPoint;
  r.size = aSize;
  if (content_view)
    [content_view setFrame: r];
}

- (void)_updateWindowGeometry
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];
  int xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  unsigned int borderWidth;
  Window rootWindow;
  int oldWidth = WIDTH;
  int oldHeight = HEIGHT;

  if (XWINDOW) {
    XFlush (xDisplay);
    if (!XGetGeometry (xDisplay, XWINDOW, &rootWindow, &X, &Y, &WIDTH, &HEIGHT,
		       &borderWidth, &DEPTH))
      return;

    frame.origin = [context userPointFromXPoint:
	    NSMakePoint (X, DisplayHeight(xDisplay, xScreen) - (Y + HEIGHT))];
    frame.size = [context userSizeFromXSize:NSMakeSize (WIDTH, HEIGHT)];

    NSDebugLog (@"window geometry user ((%f, %f), (%f, %f)), "
	   @"device ((%f, %f), (%f, %f))",
	  frame.origin.x, frame.origin.y,
	  frame.size.width, frame.size.height,
	  X, Y, WIDTH, HEIGHT);

    if (oldWidth != WIDTH || oldHeight != HEIGHT) {

      NSView *wv = [content_view superview];
      /* Inform our window and content view */
      if (content_view) {
        NSRect rect;

	rect.origin = NSZeroPoint;
	rect.size = frame.size;
	[wv setFrame: rect];
	// xxx Actually the content view should be resized
	// by the window view
	[content_view setFrame:rect];
      }

      [self createBackingStore];

      /* Tell the window view it needs display so that
	 it will fill the background of the window */
      [wv setNeedsDisplay:YES];
    }
  }
}

- (void)setContentSize:(NSSize)aSize
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];
  NSSize size;

  size = [context XSizeFromUserSize:aSize];
  NSDebugLog(@"content user size (%f, %f) device (%f, %f)",
	aSize.width, aSize.height, size.width, size.height);

  /* Resize the window */
  if (XWINDOW)
    XResizeWindow(xDisplay, XWINDOW, (int)size.width, (int)size.height);

  [self _updateWindowGeometry];

  if ([self isVisible])
    [self display];
}

- (void)setFrame:(NSRect)rect display:(BOOL)flag
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];
  int xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  NSPoint origin;
  NSSize size;

  origin = [context XPointFromUserPoint:
			    NSMakePoint(rect.origin.x,
					rect.origin.y + rect.size.height)];
  origin.y = DisplayHeight(xDisplay, xScreen) - origin.y;
  size = [context XSizeFromUserSize:rect.size];

#if DEBUGLOG
  printf ("%p (%s) user rect ((%f, %f), (%f, %f)), device ((%f, %f), (%f, %f))\n",
	self, [[self title] cString],
	rect.origin.x, rect.origin.y,
	rect.size.width, rect.size.height,
	origin.x, origin.y, size.width, size.height);
#endif

  /* Reposition and resize the window */
  XMoveResizeWindow (xDisplay, XWINDOW, (int)origin.x, (int)origin.y,
		     (int)size.width, (int)size.height);

  [self _updateWindowGeometry];

  if (flag)
    [self display];
}

- (void)setFrameOrigin:(NSPoint)aPoint
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];
  int xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  NSPoint origin;

  [super setFrameOrigin:aPoint];

  origin = [context XPointFromUserPoint:
			      NSMakePoint(frame.origin.x,
					  frame.origin.y + frame.size.height)];

  origin.y = DisplayHeight(xDisplay, xScreen) - origin.y;

  NSDebugLog (@"display a window at (%f, %f) (%f, %f)", origin.x, origin.y,
	      frame.size.width, frame.size.height);

  XMoveWindow (xDisplay, XWINDOW, (int)origin.x, (int)origin.y);
  [self _updateWindowGeometry];
}

- (void)sendEvent:(NSEvent *)theEvent
{
  [super sendEvent:theEvent];
  [NSWindow _flushWindows];
}

- (BOOL)_needsFlush				{ return needs_flush; }

- (void)_setNeedsFlush
{
//  NSLog (@"NSWindow _setNeedsFlush");
	needs_flush = YES;
	_needsFlushWindows = YES;
}

- (void)_setNeedsDisplay
{
//  NSLog (@"NSWindow setNeedsDisplay");
	needs_display = YES;
	[self _setNeedsFlush];
}

- (void)_collectFlushRectangles
{
PSMatrix* originMatrix;
PSMatrix* sizeMatrix;

	if (disable_flush_window || backing_type == NSBackingStoreNonretained)
		return;

	NSDebugLog (@"_collectFlushRectangles");
	[_flushRectangles removeAllObjects];

	originMatrix = [PSMatrix new];
	sizeMatrix = [PSMatrix new];

	[[content_view superview] 
					_collectInvalidatedRectanglesInArray:_flushRectangles
					originMatrix:originMatrix
					sizeMatrix:sizeMatrix];

	[originMatrix release];
	[sizeMatrix release];
}

- (void)update
{
NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];

	if(is_autodisplay && needs_display)					// if autodisplay is
		{												// enabled and window
		[self _collectFlushRectangles];					// is marked as needing
		[self displayIfNeeded];							// display
		[self flushWindowIfNeeded];
    	}

	[nc postNotificationName: NSWindowDidUpdateNotification object: self];
}

+ (BOOL)_flushWindows
{
  Display* xDisplay;
  NSMapEnumerator enumerator;
  void* xwindow;
  PXKWindow* window;
  PXKDPSContext* context;

  if (![self _needsFlushWindows])
    return NO;

//  NSLog (@"PXKWindow _flushWindows");

  context = (PXKDPSContext*)[NSDPSContext currentContext];
  DPSWaitContext ([context xDPSContext]);

  enumerator = NSEnumerateMapTable (windowsForNumbers);

  while (NSNextMapEnumeratorPair (&enumerator, &xwindow, (void**)&window)) {
    if (window->needs_display || window->needs_flush) {
      [window _collectFlushRectangles];
      [window displayIfNeeded];
      [window flushWindowIfNeeded];
    }
  }

  [self _setNeedsFlushWindows:NO];
  xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];

  return YES;
}

- (void)displayIfNeeded
{
  if (needs_display) {
    [[content_view superview] _displayNeededViews];
    needs_display = NO;
  }
}

- (void)flushWindow
{
  int x, y, width, height;
  int i, count;
  NSRect rect, xRect;
  PXKDPSContext* context;
  Display *xDisplay;
  int xScreen;

  if ([self isFlushWindowDisabled]
      || [self backingType] == NSBackingStoreNonretained)
    return;

//  NSLog (@"PXKWinndow flushWindow: rectangles = %d", [_flushRectangles count]);

  context = (PXKDPSContext*)[NSDPSContext currentContext];
  xDisplay = [context xDisplay];
  xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];

  for (i = 0, count = [_flushRectangles count]; i < count; i++) {
    rect = [[_flushRectangles objectAtIndex:i] rectValue];
    /* Convert the rectangle to X coordinates */
    xRect.origin = [context XPointFromUserPoint:
	  NSMakePoint(rect.origin.x,
		      frame.size.height - (rect.origin.y + rect.size.height))];
    xRect.size = [context XSizeFromUserSize:rect.size];
    x = floor (xRect.origin.x);
    y = floor (xRect.origin.y);
    width = ceil (xRect.size.width);
    height = ceil (xRect.size.height);
    NSDebugLog (@"copy PS rect ((%f, %f), (%f, %f))",
	    rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
    NSDebugLog (@"copy X rect ((%d, %d), (%d, %d))", x, y, width, height);
    XSetFunction(xDisplay, XGC, GXcopy);
    XSetPlaneMask(xDisplay, XGC, AllPlanes);
    XSetClipMask(xDisplay, XGC, None);
    XSetSubwindowMode(xDisplay, XGC, IncludeInferiors);
#if XlibSpecificationRelease > 5
    XClearArea(xDisplay, XWINDOW, x, y, width, height, False);
#endif
    XCopyArea (xDisplay, xPixmap, XWINDOW, XGC, x, y, width, height, x, y);
  }

  needs_flush = NO;
  [_flushRectangles removeAllObjects];
  [[content_view superview] _recursivelyResetNeedsDisplayInAllViews];
}

- (void)captureMouse:sender
{
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];
  int ret = XGrabPointer(xDisplay, [self xWindow], False,
			 PointerMotionMask | ButtonReleaseMask,
			 GrabModeAsync, GrabModeAsync, 
			 None, None, CurrentTime);

  if (ret != GrabSuccess)
    NSDebugLog(@"Failed to grab pointer\n");
  else
    {
      NSApplication *theApp = [NSApplication sharedApplication];
      NSDebugLog(@"Grabbed pointer\n");
      [(PXKApplication *)theApp setGrabXWindow: [self xWindow]];
    }
}

- (void)releaseMouse:sender
{
  NSApplication *theApp = [NSApplication sharedApplication];
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];
  XUngrabPointer(xDisplay, CurrentTime);
  [(PXKApplication *)theApp setGrabXWindow: 0];
}

- (void)createXWindow:(NSRect)rect
{
  XColor c1;
  NSColor *c;
  XWMHints *wm_hints;
  XGCValues values;
  unsigned long valuemask = (GCForeground | GCBackground);
  int xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];
  XSetWindowAttributes winattrs;
  unsigned long valuemask2;
  NSCursor *arrow;
  NSPoint origin;
  NSSize size;

  origin = [context XPointFromUserPoint:
			    NSMakePoint(rect.origin.x,
					rect.origin.y + rect.size.height)];
  size = [context XSizeFromUserSize:rect.size];
  origin.y = DisplayHeight(xDisplay, xScreen) - origin.y;

  if (!(wm_hints = XAllocWMHints())) {
    NSLog(@"PXKWindow: Failure allocating memory with XAllocWMHints()\n");
    exit(0);
  }

  // Window background color
  //c = [self backgroundColor];
  c = [NSColor whiteColor];
  if (([[c colorSpaceName] compare: NSCalibratedRGBColorSpace] 
       == NSOrderedSame) ||
      ([[c colorSpaceName] compare: NSDeviceRGBColorSpace]
       == NSOrderedSame))
    {
      // Create an XColor from the NSColor
      c1.red = (unsigned short)(65535 * [c redComponent]);
      c1.green = (unsigned short)(65535 * [c greenComponent]);
      c1.blue = (unsigned short)(65535 * [c blueComponent]);
    }
  else if (([[c colorSpaceName] compare: NSCalibratedWhiteColorSpace] 
	    == NSOrderedSame) ||
	   ([[c colorSpaceName] compare: NSDeviceWhiteColorSpace]
	    == NSOrderedSame))
    {
      // Create an XColor from the NSColor
      c1.red = (unsigned short)(65535 * [c whiteComponent]);
      c1.green = (unsigned short)(65535 * [c whiteComponent]);
      c1.blue = (unsigned short)(65535 * [c whiteComponent]);
    }
  else
    {
      // If not a colorspace we can easily convert then
      // then just make the background gray.
      c1.red = 32000;
      c1.green = 32000;
      c1.blue = 32000;
    }
  XAllocColor(xDisplay, XDefaultColormap(xDisplay, xScreen), &c1);

  valuemask2 = (CWBackPixel|CWBorderPixel);
  winattrs.border_pixel = BlackPixel(xDisplay, xScreen);
  winattrs.background_pixel = c1.pixel;
  XWINDOW = XCreateWindow(xDisplay, RootWindow(xDisplay, xScreen),
			  (int)origin.x, (int)origin.y,
			  (int)size.width, (int)size.height,
			  0,
			  CopyFromParent,
			  CopyFromParent,
			  CopyFromParent,
			  valuemask2,
			  &winattrs);

  // Create a GC for the content view
  values.foreground = BlackPixel(xDisplay, xScreen);
  values.background = WhitePixel(xDisplay, xScreen);
  values.function = GXcopy;
  valuemask = (GCForeground | GCBackground | GCFunction);
  XGC = XCreateGC(xDisplay, RootWindow(xDisplay, xScreen), valuemask, &values);

  /* Set the event mask */
  XSelectInput(xDisplay, XWINDOW,
	       ExposureMask | KeyPressMask | KeyReleaseMask |
	       ButtonPressMask | ButtonReleaseMask | 
	       ButtonMotionMask | StructureNotifyMask |
	       PointerMotionMask | EnterWindowMask |
	       LeaveWindowMask | FocusChangeMask |
	       PropertyChangeMask | ColormapChangeMask | KeymapStateMask |
	       VisibilityChangeMask);

  /* Give the window manager some hints */
  wm_hints->initial_state = NormalState;
  wm_hints->input = True;
  wm_hints->flags = StateHint | InputHint;
  XSetWMHints(xDisplay, XWINDOW, wm_hints);

  [self _updateWindowGeometry];

  // Pop the arrow as the default cursor
  arrow = [NSCursor arrowCursor];
  [arrow push];
}

- (void)setFrameFromXFrame:(NSRect)rect
{
  BOOL onlyMoved = NO;
  Drawable drawable;
  GContext gc;

  /* Determine if the window was only moved */
  if (rect.size.width == WIDTH && rect.size.height == HEIGHT
      && (rect.origin.x != X || rect.origin.y != Y))
    onlyMoved = YES;

  [self _updateWindowGeometry];

  if (onlyMoved)
    return;

  drawable = [self drawable];
  gc = XGContextFromGC ([self xGC]);

  PSWinitcontext (gc, drawable, 0, HEIGHT);

//  if ([self backingType] != NSBackingStoreNonretained)
//    /* The window has changed its size. Redisplay it. */
//    [self display];

  /* The window has changed its size. Redisplay it. */
  [[self contentView] setNeedsDisplay:YES];
}

- (void)setXUnmap:(BOOL)flag
{
  // Unmapped means it is no longer visible
  visible = flag;
}

- (Drawable)drawable
{
  if ([self backingType] != NSBackingStoreNonretained)
    return xPixmap;
  else
    return ([self isXExposed] ? XWINDOW : None);
}

- (NSPoint)convertBaseToScreen:(NSPoint)basePoint
{
  NSPoint screenPoint;

  screenPoint.x = frame.origin.x + basePoint.x;
  screenPoint.y = frame.origin.y + basePoint.y;
  return screenPoint;
}

- (NSPoint)convertScreenToBase:(NSPoint)screenPoint
{
  NSPoint basePoint;

  basePoint.x = screenPoint.x - frame.origin.x;
  basePoint.y = screenPoint.y - frame.origin.y;
  return basePoint;
}

- (void)createBackingStore
{
  Display* xDisplay;
  int xScreen;

  if ([self backingType] == NSBackingStoreNonretained)
    return;

  xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];
  xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];

  NSDebugLog (@"createBackingStore");
  /* Determine if the window was only moved */
  xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];

  if (xPixmap) {
    /* Destroy the old pixmap and create a new one */
    DPSWaitContext ([(PXKDPSContext *)[NSDPSContext currentContext]
					  xDPSContext]);
    XFreePixmap (xDisplay, xPixmap);
  }

  NSDebugLog (@"create backing store: width = %d, height = %d, depth = %d",
	      WIDTH, HEIGHT, DEPTH);
  xPixmap = XCreatePixmap (xDisplay, RootWindow(xDisplay, xScreen),
			   WIDTH, HEIGHT, DEPTH);
  if (!xPixmap) {
    NSLog (@"cannot create the pixmap for backing store!");
    return;
  }
  else
    [(PXKDPSContext *)[NSDPSContext currentContext] setXDrawable:self];

  /* Clear the pixmap to avoid garbages */
  XFillRectangle (xDisplay, xPixmap, XGC, 0, 0, WIDTH, HEIGHT);
}

- (void)addExposedRectangle:(XRectangle)rectangle
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];

  /* If the window has backing store just copy the exposed portion from the
     pixmap to window.

     If the window has no backing store the algorithm is different. Keep a list
     with the exposed rectangles here. In -processExposedRectangles compute the
     views that should be redisplayed. The code doesn't use the rectviewclip
     because it seems the DGS does not support it; use X clipping instead. */
  if ([self backingType] != NSBackingStoreNonretained) {
    Display *xDisplay = [context xDisplay];
    int xScreen;

    xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];

    // xxx is this necessary?
    /* DPSWaitContext ([(PXKDPSContext *)[NSDPSContext currentContext]
				      xDPSContext]); */
    NSDebugLog (@"copy exposed area ((%d, %d), (%d, %d))",
		rectangle.x, rectangle.y, rectangle.width, rectangle.height);
    XSetFunction(xDisplay, XGC, GXcopy);
    XSetPlaneMask(xDisplay, XGC, AllPlanes);
    XSetClipMask(xDisplay, XGC, None);
    XSetSubwindowMode(xDisplay, XGC, IncludeInferiors);
#if XlibSpecificationRelease > 5
    XClearArea(xDisplay, XWINDOW, rectangle.x, rectangle.y,
    	       rectangle.width, rectangle.height, False);
#endif
    XCopyArea (xDisplay, xPixmap, XWINDOW, XGC,
	       rectangle.x, rectangle.y, rectangle.width, rectangle.height,
	       rectangle.x, rectangle.y);
  }
  else {
    NSRect rect;

    /* Add this rectangle to the region. The region is used in
       -processExposedRectangles to set the clipping path. */
    XUnionRectWithRegion (&rectangle, region, region);

    /* Transform the rectangle's coordinates to PS coordinates and add this
       new rectangle to the list of exposed rectangles. */
    rect.origin = [context userPointFromXPoint:NSMakePoint(rectangle.x,
							   rectangle.y)];
    rect.size = [context userSizeFromXSize:NSMakeSize(rectangle.width,
						       rectangle.height)];
    [exposedRectangles addObject:[NSValue valueWithRect:rect]];
  }
}

- (void)processExposedRectangles
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [context xDisplay];

  if ([self backingType] != NSBackingStoreNonretained)
    return;

  xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext] xDisplay];

  /* Set the clipping path to the exposed rectangles so the further drawings
     will not affect the non-exposed region */
  XSetRegion (xDisplay, XGC, region);

  /* We should determine the views that need to be redisplayed. Until we
     fully support scalation and rotation of views redisplay everything. */
  [self display];

  /* Restore the exposed rectangles and the region */
  [exposedRectangles removeAllObjects];
  XDestroyRegion (region);
  region = XCreateRegion();
  XSetClipMask (xDisplay, XGC, None);
}

- (NSRect)xFrame
{
  return NSMakeRect (X, Y, WIDTH, HEIGHT);
}

- (Window)xWindow		{ return XWINDOW; }
- (GC)xGC			{ return XGC; }
- (void)setXExposed:(BOOL)flag	{ XEXPOSE = flag; }
- (BOOL)isXExposed		{ return XEXPOSE; }

- (void)_view:(NSView*)view needsFlushInRect:(NSRect)rect
{
  [self _setNeedsFlush];

  /* Convert the rectangle to the window coordinates */
  rect = [view convertRect:rect toView:nil];

  /* If the view is rotated then compute the bounding box of the rectangle in
     the window's coordinates */
  if ([view isRotatedFromBase])
    rect = [view _boundingRectFor:rect];

  /* TODO: Optimize adding rect to the already existing arrays */

  [_flushRectangles addObject:[NSValue valueWithRect:rect]];
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [PXKWindow class])
    return [super class];
  return [self class];
}

@end
