// This is -*- C++ -*-
// $Id: Permutation.h,v 1.2 1999/03/16 18:29:25 alstrup Exp $

/* 
 * Permutation.h
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _INC_PERMUTATION_H
#define _INC_PERMUTATION_H

#include "RCArray.h"
#include "Random.h"

using namespace std;

class Permutation {
public:
  Permutation();
  Permutation(size_t);
  Permutation(const Permutation&);

  Permutation& operator=(const Permutation&);
  
  size_t size() const { return perm_.size(); }
  const size_t* map() const { return perm_.data(); }
  // note: no bounds checking
  size_t map(size_t i) const { return (perm_.data())[i]; }

  void initialize(size_t N, size_t*);

  size_t cycle_count() const { calc_cycles(); return cyclerep_.size(); }
  size_t orbit_size(size_t) const;
  size_t order() const; // In the sense of group theory...

  void set_identity(size_t);
  void set_shift(size_t, int k);
  void set_reverse(size_t);
  void scramble_explicit(Random&);
  void scramble() { scramble_explicit(common_random); }

  void precompose(const Permutation&);
  void postcompose(const Permutation&);
  void invert();

  // WARNING: We assume that this points to an array of at least size()
  // X's.  X should be a type that can be copied by operator=().
  // The nice thing about using our cycle decomposition is that we can
  // permute the array in O(N) time without having to do any buffer copies.
  template<class X>
  void permute(X* x) const {
    calc_cycles();
    for(size_t i=0; i<cyclerep_.size(); ++i) {
      size_t j = (cyclerep_.data())[i];
      if (j != map(j)) {
	size_t start = j;
	X tmp = x[j];
      do {
	X tmp2 = x[map(j)];
	x[map(j)] = tmp;
	tmp = tmp2;
      } while ((j = map(j)) != start);
      }
    }
  }

  // some output that is useful for debugging (and not much else)
  void spew(ostream& out) const;
  void spew_cycles(ostream& out) const;

private:
  void calc_cycles() const;

  RCArray<size_t> perm_;
  RCArray<size_t> cyclerep_;
};

#endif // _INC_PERMUTATION_H

// $Id: Permutation.h,v 1.2 1999/03/16 18:29:25 alstrup Exp $
