// Exercises the Transform class with the Fourier expansion functions

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
You should have received a copy of the GNU General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include<complex.h>
#include<mMatrix.h>
#include<instance.h>
#include<transform.h>

typedef mVector<double,allocator<double> > dbl_vctr;

typedef mVector<double_complex,allocator<double_complex> > cmplx_vctr;

double (*np)(const double_complex&)=norm;

double test_fft(cmplx_vctr &f,cmplx_vctr &fft) {
// Computes the L2-norm of the error between the FFT and the discrete FT.
  int n=f.size();
  double err=0;
  double_complex ftmp,arg;
  for(int j=1;j<=n;j++) {   // accumulate the FFT the hard way
    ftmp=double_complex(0,0);
    for(int k=1;k<=n;k++) {
      arg=double_complex(0,2*M_PI*(j-1)*(k-1)/n);
      ftmp+=f(k)*exp(arg);
    }
    ftmp/=sqrt(n);
    err+=norm(ftmp-fft(j));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

double test_sin(dbl_vctr &f,dbl_vctr &fft) {
// Computes the L2-norm of the error between the FST and the discrete FT.
  int n=f.size()+1;
  double err=0;
  double ftmp,arg;
  for(int j=1;j<n;j++) {   // accumulate the FFT the hard way
    ftmp=0;
    for(int k=1;k<n;k++) {
      arg=M_PI*j*k/n;
      ftmp+=f(k)*sin(arg);
    }
    ftmp*=sqrt(2.0/n);
    err+=norm(ftmp-fft(j));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

double test_cos(dbl_vctr &f,dbl_vctr &fft) {
// Computes the L2-norm of the error between the FST and the discrete FT.
  int n=f.size()-1;
  double err=0;
  double ftmp;
  int s=1;
  for(int k=0;k<=n;k++,s*=-1) {   // accumulate the FFT the hard way
    ftmp=0.5*(f(1)+s*f(n+1));
    for(int j=1;j<n;j++) ftmp+=f(j+1)*cos(k*j*M_PI/n);
    ftmp*=sqrt(2.0/n);
    err+=norm(ftmp-fft(k+1));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

int main(int argc,char **argv) {
  size_t dim=atoi(argv[1]);
  cout << "mVector length: " << dim << "\n\n";
  srand(1);              // seed the random # generator with a known value
  double maxr=(double)0x0fffffff;
  
  Fourier<double_complex> ff;             // expansion functions
  ff.a()=double_complex(1.0,0.0);         // set endpoints
  ff.b()=double_complex(dim+1,0.0);
  ff.resize(dim,dim);                     // set mesh and expansion size
  Transform<double_complex,double_complex,Fourier<double_complex> > 
    tf(ff,dim,dim);                       // transform object
  tf.initMesh();                          // initialize mesh from ff
  tf.initOp(0.0);                         // initialize operators
  
  FourierSin<double> fs;                  // expansion functions
  fs.a()=0.0;                             // set endpoints
  fs.b()=dim+1;
  fs.resize(dim,dim);                     // set mesh and expansion size
  Transform<double,double,FourierSin<double> > 
    ts(fs,dim,dim);                       // transform object
  ts.initMesh();                          // initialize mesh from fs
  ts.initOp(0.0);                         // initialize operators
  
  FourierCos<double> fc;                  // expansion functions
  fc.a()=1;                               // set endpoints
  fc.b()=dim;
  fc.resize(dim,dim);                     // set mesh and expansion size
  Transform<double,double,FourierCos<double> > 
    tc(fc,dim,dim);                       // transform object
  tc.initMesh();                          // initialize mesh from fc
  tc.initOp(0.0);                         // initialize operators
 
  cmplx_vctr f(dim);                      // initialize a random complex vector
  for(int j=1;j<=f.size();j++) {
    f(j)=double_complex(rand()/maxr,rand()/maxr);
  }
  dbl_vctr r(dim);                        // initialize a random real vector
  for(int j=1;j<=r.size();j++) {
    r(j)=rand()/maxr;
  }

  cout << "Complex FFT: \n";
  tf.data=f;                              // initialize transform data
  tf.forward();                           // compute coefficients
  double err=sqrt(accumulate(f,0.0,np))-sqrt(accumulate(tf.coef,0.0,np));
  cout << "\tTest of discrete Parsival: " << err << "\n";
  cout << "\tError of FFT from discrete FT: " << test_fft(f,tf.coef) << "\n";
  tf.backward();                          // recompute data
  err=sqrt(accumulate(f-tf.data,0.0,np)/accumulate(f,0.0,np));
  cout << "\tError of inverse FFT: " << err << "\n\n";

  cout << "Real ST: \n";
  ts.data=r;                              // initialize transform data
  ts.forward();                           // compute coefficients
  err=sqrt(accumulate(r,0.0,np))-sqrt(accumulate(ts.coef,0.0,np));
  cout << "\tTest of discrete Parsival: " << err << "\n";
  cout << "\tError of ST from discrete T: " << test_sin(r,ts.coef) << "\n";
  ts.backward();                          // recompute data
  err=sqrt(accumulate(r-ts.data,0.0,np)/accumulate(r,0.0,np));
  cout << "\tError of inverse ST: " << err << "\n\n";

  cout << "Real CT: \n";
  tc.data=r;                              // initialize transform data
  tc.forward();                           // compute coefficients
  err=sqrt(accumulate(r,0.0,np))-sqrt(accumulate(tc.coef,0.0,np));
  cout << "\tNo discrete Parsival: " << err << "\n";
  cout << "\tError of CT from discrete T: " << test_cos(r,tc.coef) << "\n";
  tc.backward();                          // recompute data
  err=sqrt(accumulate(r-tc.data,0.0,np)/accumulate(r,0.0,np));
  cout << "\tError of inverse CT: " << err << "\n\n";

  cout << "If any numbers except CT discrete Parsival are nonzero,\n";
  cout << "then it indicates a problem which should be investigated.\n";

  return 0;
}
