
#ifndef _ode_h_
#define _ode_h_

template<class T,class Real>
class OIVP
{
// integrates the initial value problem dx/dt = f(t,x) for real scalar
// t and complex vector f
public:
// The following must be defined by the user prior to calling integrate:
  int dim;                          // dimension
  int plotpoint;                    // number of points to plot result
  Real step;                        // current mesh size
  Real tol;                         // numerical tolerance for error
  Real start,stop;                  // beginning and ending values of interval
  virtual mVector<T,allocator<T> >& // updates f with supplied tp and xp, and
    evaluate(Real tp,mVector<T,allocator<T> > &xp) {} // returns reference to f

// The remaining members may be defined at the user's discretion
  Real t;                           // current parameter value
  mVector<T,allocator<T> > x;       // current solution values
  mVector<T,allocator<T> > f;       // current derivative values
  mVector<T,allocator<T> > x0;      // starting and previous derivative values  
  mVector<T,allocator<T> > error;   // estimated error for each solution
  mVector<T,allocator<T> > scale;   // characteristic values for each dimension
  mMatrix<T,allocator<T> > plot;    // columns are results at specified points
protected:
  mVector<Real,allocator<Real> > midpoint; // storage for interior points
  mMatrix<T,allocator<T> > midval;  // storage for interior solution values

public:
  OIVP(const OIVP&);
  OIVP() : tol(1.0e-8), plotstep(0.0) {}
  virtual void precondition() {}    // called at beginning of integrate()
  virtual void postcondition() {}   // called at end of integrate()
  virtual void init() {}            // initialize structures for integration
  virtual void takestep() {}        // updates (t,x,f) with one step
  virtual void changestep() {}      // estimates new step size based on error
       // perform the integration from start to stop, storing output in plot
  virtual void integrate(); 
};

template<class T,class Real>
ostream&
operator<<(ostream &os,OIVP<T,Real>& a);

template<class T,class Real>
class RungeKutta : public OIVP<T,Real>
{
// Implements Runge-Kutta specialization of OIVP
private:
  mVector<Real,allocator<Real> > a,c,cs;   // coef. of Runge-Kutta step
  mMatrix<Real,allocator<Real> > b;
  mVector<T,allocator<T> > tmp;      // temporary storage for x and f
  
  const double cutoff=1.89e-4;       // (5/safety)^(1/growexp)
  const double safety=0.9;           // a safety factor
  const double growexp=-0.2;           // the step growth exponent
  const double shrinkexp=-0.25;           // the step shrink exponent
  const double growfact=5.0;           // the step growth factor
  const double shrinkfact=0.1;           // the step shrink factor
 

public:
  RungeKutta(const RungeKutta<T,Real> &a);
  RungeKutta(int d);
  virtual void init();
  virtual void takestep();          // updates (t,x,f) with one step
  virtual void changestep();        // estimates new step size based on error
      // evaluate() must be overloaded by the user
  virtual mVector<T,allocator<T> >& // updates f with supplied tp and xp, and
    evaluate(Real tp,mVector<T,allocator<T> > &xp) {} // returns reference to f
};

#endif
