/* This file implements a plotting utility class, Viewport */

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This file is part of the GNUSSL software package.  This package is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this distribution; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _Viewport_h_
#define _Viewport_h_

#include<utilities.h>

// Note that for 2-d plots, the horizontal (dependent) variable is x, and
// y is the vertical (independent) variable.  For 3-d plots, x is the column
// variable, y is the row variable and z is the vertical variable.  Since
// the plot is made in a right-handed coordinate system, x cross y equals z,
// this may mean that x and y are swapped in relation to what you expect.

// To change the plot range from the default of including all points, the
// axis reset flags must be unset and the Frame member of the Viewport
// instance must be changed.  Suppose that vp is the Viewport object, then
// vp.?reset=0, vp.f.?o is the beginning of the plot axis,
// and (vp.f.?o+vp.f.?m) is the end.  These are in the "natural" units of the
// data; an arbitrary transformation of natural to user defined units is
// defined in the class members "vp.?scale" and "vp.?off", e.g.,
// xlabel(j)=vp.f.xtics(j)*vp.xscale + vp.xoff .  The number of tic
// marks on each axis may be set by the variable "vp.opt.?n".

class Frame {      // Frame contains frame and tic mark info
public:
  size_t xn,yn,zn;                          // number of tic marks on each axis
  mVector<float,allocator<float> > xtics, ytics, ztics;  // tic mark storage 
  float xo,xm,yo,ym,zo,zm;                  // offset and width on each axis
  Frame();
};


class Quadril {    // An internal class for handling surface elements
public:
  size_t r,c;                            // row and col label of element 
  mMatrix<float,allocator<float> > z;    // z values of vertices
  float dist;                            // distance from viewPoint

  Quadril();
  Quadril(const Quadril &q);
  Quadril& operator=(const Quadril &q);
};

class Options {   // Lists all options for the Viewport plotting class
public:
  typedef mVector<float,allocator<float> > flt_vctr;
  typedef mMatrix<float,allocator<float> > flt_mtrx;

  ostream *osp;              // pointer to output stream
  char *title;               // title string
  char *labelFont;           // PostScript font for labels
  bool plotJoined;           // points connected by line
  bool points;               // do/not draw points
  bool shading;              // do/not shade between contours
  bool frame;                // draw axes
  bool tics;                 // draw ticmarks on the axes
  bool label;                // draw labels for ticmarks
  bool fft;                  // retile the axes assuming data is from an FFT
  bool autoContour;          // do/not use contour mVector
  bool xreset,yreset,zreset; // flags for resetting tic marks
  size_t numContour;         // number of contours to draw, maximum 50
  size_t numPlot;            // max number of 1-D plots on a graph
  size_t oneDMesh;           // maximum number of points on one dimen. plot
  size_t twoDMesh;           // maximum number of points on two dimen. plot
  size_t order;              // polynomial order of interpolating function
  size_t rpix,cpix;          // pixel width of contour plot bitmap
  float xscale,yscale,zscale;// scale factor for labeling tic marks
  float xoff,yoff,zoff;      // offset for labeling tic marks
  float vscale,hscale;       // plot size in inches
  float fontSize;            // font size (plot is 1x1 in these units)
  float alpha;               // ratio of viewpoint (to viewport)/(to midpoint)
  float axisWidth;           // thickness of axes (plot is 1x1 in these units)
  float lineWidth;           // thickness of lines
  float ticLength;           // length of ticmarks
  float dotSize;             // diameter of points
  float wire;                // default line color (0=black,1=white)
  float background;          // background color
  flt_vctr contour;     // contours to draw
  flt_vctr viewPoint;   // view point (direction) for 3-d frame plots
  flt_vctr stretch;     // stretch in each direction for 3-d frame plots
};



class Viewport : public Options {
public:
  Frame f;
  flt_mtrx *plot;
  flt_mtrx surf;
  Viewport();
  ~Viewport();

  // PLOTTING ROUTINES
  void pplot();        // Internal 2-d parametric plot
  void cplot();        // Internal contour plot
  void fplot();        // Internal 3-d frame plot
  
  // UTILITIES
  void clearPlot();
  void clearPlot(size_t j);
  void dashing(size_t j); // sets dashing characteristics for the j'th line plot

  // member templates for initializing plots
#ifdef _MEMBER_TEMPLATES_
  template<class T,class F>
    void
    initPlot(T tmin,T tmax,size_t length,F f,size_t dash);
  template<class T,class F,class Fl>
    void
    initPlot(T tmin,T tmax,size_t length,F f,Fl filter,size_t dash);
  template<class T,class F,class M>
    void
    initImplPlot(T tmin,T tmax,size_t length,F f,M m,size_t dash);
  template<class T,class F,class M,class Fl>
    void
    initImplPlot(T tmin,T tmax,size_t length,F f,M m,Fl filter,size_t dash);
  template<class T,class Fx,class Fy>
    void
    initParamPlot(T tmin,T tmax,size_t length,Fx fx,Fy fy,size_t dash);
  template<class T,class Fx,class Fy,class Fl>
    void
    initParamPlot(T tmin,T tmax,size_t length,Fx fx,Fy fy,Fl filter,size_t dash);
  template<class T,class F>
    void
    initSurfPlot(T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f);
  template<class T,class F,class Fl>
    void
    initSurfPlot(T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f,Fl filter);
#endif

private:
  flt_mtrx viewRotation;

  // UTILITIES
  void setRotation();
  void frameSet1D(); // Sets default frame for a 2-d parametric plot.
  void frameSet2D(); // Sets default frame for contour or 3-d frame plot.
  void p3dtop2d(flt_vctr &p3d,flt_vctr &p2d);
  float contain(size_t r,size_t c);
  bool testContour(flt_mtrx &ref,flt_vctr &cnt,size_t r,size_t c);
  float interpolate(flt_vctr &p);
  
  // POSTSCRIPT GENERATING ROUTINES
  void header(float xsize,float ysize,float xoff=0.0,float yoff=0.0);
  void trailer();
  void polygon(Quadril &q,float xel,float yel);
  void frame2d();
  void frame3d();
};



template<class T,class F>
void
initPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,size_t dash);
// plots (t,f(t)) stepping over [tmin,tmax] 
// t and f(t) must be castable to float

template<class T,class F,class Fl>
void
initPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,Fl filter,size_t dash);
// plots (t,filter(f(t))) stepping over [tmin,tmax] 

template<class T,class F,class M>
void
initImplPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,M m,size_t dash);
// plots (t,f(m(t))) stepping over [tmin,tmax] 
// t and f(m(t)) must be castable to float

template<class T,class F,class M,class Fl>
void
initImplPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,M m,Fl filter,size_t dash);
// plots (t,filter(f(m(t)))) stepping over [tmin,tmax] 

template<class T,class Fx,class Fy>
void
initParamPlot(Viewport& vp,T tmin,T tmax,size_t length,Fx fx,Fy fy,size_t dash);
// plots (fx(t),fy(t)) stepping over [tmin,tmax] 

template<class T,class Fx,class Fy,class Fl>
void
initParamPlot(Viewport& vp,T tmin,T tmax,size_t length,Fx fx,Fy fy,Fl filter,size_t dash);
// plots (filter(fx(t)),filter(fy(t))) stepping over [tmin,tmax] 

template<class T,class F>
void
initSurfPlot(Viewport& vp,T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f);
// plots f(x,y) stepping over [xmin,ymax] and [ymin,ymax]

template<class T,class F,class Fl>
void
initSurfPlot(Viewport& vp,T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f,Fl filter);
// plots filter(f(x,y)) stepping over [xmin,ymax] and [ymin,ymax]


// FUNCTION DEFINITIONS

Options::Options() : contour(50), viewPoint(3), stretch(3) 
{
    osp=&cout;
    title=0;
    plotJoined=1;
    points=0;  
    numContour=0;
    numPlot=10;
    oneDMesh=1000;
    twoDMesh=10000;
    autoContour=1;
    vscale=5;
    hscale=5;
    xreset=yreset=zreset=1;
    xscale=yscale=zscale=1;
    xoff=yoff=zoff=0;
    order=1;
    shading=1;
    frame=1;
    tics=1;
    label=1;
    labelFont="Times-Roman";
    rpix=250;
    cpix=250;
    fontSize=0.05;
    viewPoint(1)=1;
    viewPoint(2)=-1.7;
    viewPoint(3)=0.2;
    stretch(1)=3;
    stretch(2)=3;
    stretch(3)=0.75;
    alpha=0.2;
    axisWidth=0.003;
    lineWidth=0.001;
    ticLength=0.01;
    dotSize=0.004;
    wire=0;
    background=0.9;
}



Frame::Frame() : xtics(50), ytics(50), ztics(50) 
{
    xn=yn=zn=5;
    xo=xm=yo=ym=zo=zm=0;
}

Quadril::Quadril() : z(2,2) {}

Quadril::Quadril(const Quadril &q) : r(q.r), c(q.c), dist(q.dist), z(2,2) 
{
  z=q.z; 
}

Quadril& 
Quadril::operator=(const Quadril &q) 
{
    r=q.r;
    c=q.c;
    dist=q.dist;
    z=q.z;
    return(*this);
  }


// VIEWPORT MEMBER DEFINITIONS

Viewport::Viewport() : viewRotation(3,3) {  // allocate storage plots
  surf.array().reserve(twoDMesh);
  plot=new flt_mtrx[numPlot];
  for(size_t j=0;j<numPlot;j++) {
    plot[j].reserve(oneDMesh,2);
    plot[j].resize(0,0);
  }
}

Viewport::~Viewport() 
{ 
  delete [] plot; 
}

  // PLOTTING ROUTINES
 
void 
Viewport::pplot() 
{
// internal 2-d parametric plot
    frameSet1D();
    float vborder=0.1,hborder=0.2;
    if(tics==1) {
      vborder+=5*fontSize;
      hborder+=1.25*fontSize;
    }
    header(1.6,1.6,vborder,hborder);    // initialize PS device
    frame2d();                           // draw a frame
    for(size_t j=0;j<numPlot;j++) {
      if(plot[j].rsize()==0) continue;       // skip unallocated plots
      *osp << "newpath\n";                   // begin the line drawing
      *osp << "1 setlinejoin\n";             // rounded joints between lines
      *osp << lineWidth << " setlinewidth\n";
      dashing(j);                        // set the dashing characteristics
      if(plotJoined==1) {               // draw lines point to point
        *osp << (plot[j](1,1)-f.xo)/f.xm << " ";
        *osp << (plot[j](1,2)-f.yo)/f.ym << " moveto\n";
        for(size_t r=2;r<=plot[j].rsize();r++) {       // loop over the points
          *osp << (plot[j](r,1)-f.xo)/f.xm << " ";
          *osp << (plot[j](r,2)-f.yo)/f.ym << " lineto\n";
        }
      }       // end r-loop
      *osp << "stroke\n";                    // plot the graph
      if(points==1) {                   // draw a dot at each point
        for(size_t r=1;r<=plot[j].rsize();r++) {       // loop over the points
          *osp << (plot[j](r,1)-f.xo)/f.xm << " ";
          *osp << (plot[j](r,2)-f.yo)/f.ym << " ";
          *osp << dotSize << " 0 360 arc fill\n";
          *osp << "stroke\n";                    // plot the point
        }
      }       // end r-loop
    }       // end j-loop
    trailer();
}
 
void 
Viewport::cplot() 
{
  // Internal contour plot
  // Produces a bitmap of the interpolated values of surf.
    frameSet2D();
    flt_mtrx ref(rpix,cpix);
    float z;
    flt_vctr p(2);
    size_t rpos,cpos;
    for(size_t r=1; r<=ref.rsize(); r++) {       // step over the bitmap
      for(size_t c=1; c<=ref.csize(); c++) {
        if(fft==1) { // retile 1st quad to 3rd, 2nd to 4th, and vice versa
          size_t half=ref.rsize()/2;    // int arith. rounds down
          rpos=(r<=half)?r+half:r-half;
          half=ref.csize()/2;
          cpos=(c<=half)?c+half:c-half;
        } else {
          rpos=r;
          cpos=c;
        }
        p(1)=(rpos-1.0)/(ref.rsize()-1.0);  // calculate position of the pixel
        p(2)=(cpos-1.0)/(ref.csize()-1.0);
        z=interpolate(p);              // interpolate the value
        ref(r,c)=(z-f.zo)/f.zm;         // rescale zo to 0 and zm to 1
        if(ref(r,c)<0) ref(r,c)=0;             // clip to [0,1]
        else if(ref(r,c)>1) ref(r,c)=1; 
      }
    }
    flt_vctr cnt(numContour);   // short hand for contour vector
    if(numContour>0) {
      if(autoContour==1) cnt(1)=1.0/(numContour+1.0);
      else cnt(1)=(contour(1)-f.zo)/f.zm;   // rescale contour lines
      float tc;
      for(size_t j=2;j<=numContour;j++) {
        if(autoContour==1) cnt(j)=j/(numContour+1.0);
        else cnt(j)=(contour(j)-f.zo)/f.zm;
        for(size_t k=j;k>1;k--) {           // and sort them in increasing order
          if(cnt(k)<cnt(k-1)) {
            tc=cnt(k-1);
            cnt(k-1)=cnt(k);
            cnt(k)=tc;
          }
        }  // k-loop                     // end sort
      }  // j-loop                       // end contour setup
    }
  
    if(frame==1) {
      float vborder=0.1,hborder=0.2;
      if(tics==1) {
        vborder+=5*fontSize;
        hborder+=1.25*fontSize;
      }
      header(1.6,1.6,vborder,hborder);    // initialize PS device
    } else header(1.2,1.2);
  
    if(shading==1) {                           // output the bitmap
      osp->form("/picstr %d string def\n",cpix);
      osp->form("%d %d 8 \n[%d 0 0 -%d 0 %d] \n",cpix,rpix,cpix,rpix,rpix);
      *osp << "{currentfile picstr readhexstring pop} image\n";
      for(size_t r=1; r<=rpix; r++) {  // step over the bitmap (output in hex)
        for(size_t c=1; c<=cpix; c++)
          osp->form("%02X ",(size_t)rint(GREY*ref(r,c)));
        *osp << "\n";
      }
    }
  
    if(numContour>0) {
      *osp << "1 setlinecap\n" << lineWidth << " setlinewidth\n";
      for(size_t r=1; r<=ref.rsize(); r++) {      // step over the bitmap
        for(size_t c=1;c<=ref.csize(); c++) {
          if(testContour(ref,cnt,r,c)==1) {
            if(testContour(ref,cnt,r,c+1)==1) {    // right
              *osp << (c-1.0)/(ref.csize()-1.0) << "  ";
              *osp << 1.0 - (r-1.0)/(ref.rsize()-1.0) << " moveto\n";
              *osp << 1.0/(ref.csize()-1.0) << " 0 rlineto\n";
            }
            if(testContour(ref,cnt,r+1,c+1)==1) {  // right and down
              *osp << (c-1.0)/(ref.csize()-1.0) << "  ";
              *osp << 1.0 - (r-1.0)/(ref.rsize()-1.0) << " moveto\n";
              *osp << 1.0/(ref.csize()-1.0) << "  " << -1.0/(ref.rsize()-1.0);
              *osp << " rlineto\n";
            }
            if(testContour(ref,cnt,r+1,c)==1) {    // down
              *osp << (c-1.0)/(ref.csize()-1.0) << "  ";
              *osp << 1.0 - (r-1.0)/(ref.rsize()-1.0) << " moveto\n";
              *osp << "0  " << -1.0/(ref.rsize()-1.0) << " rlineto\n";
            }
            if(testContour(ref,cnt,r+1,c-1)==1) {   // down and left
              *osp << (c-1.0)/(ref.csize()-1.0) << "  ";
              *osp << 1.0 - (r-1.0)/(ref.rsize()-1.0) << " moveto\n";
              *osp << -1.0/(ref.csize()-1.0) << "  " << -1.0/(ref.rsize()-1.0);
              *osp << " rlineto\n";
            }
          *osp << "stroke\n";
          }        // end of drawing for this (r,c)
        }          // finished this c
      }            // finished this r
    }              // finished drawing contours
    frame2d();          // draw a frame
    trailer();                           // eps trailer
}
 
void 
Viewport::fplot() 
{
  // Internal 3-d frame plot
    setRotation();
    frameSet2D();
    mVector<Quadril,allocator<Quadril> > lst((surf.rsize()-1)*(surf.csize()-1));
    for(size_t r=1;r<surf.rsize();r++) {  // loop over the mesh, init. a Quadril
      for(size_t c=1;c<surf.csize();c++) {      // for each element
        size_t n=(r-1)*(surf.csize()-1)+c;      // vector label
        if(fft==1) { // retile 1st quad to 3rd, 2nd to 4th, and vice versa
          size_t half=surf.rsize()/2;    // int arith. rounds down
          lst(n).r=(r<=half)?r+half:r-half;
          half=surf.csize()/2;
          lst(n).c=(c<=half)?c+half:c-half;
        } else {
          lst(n).r=r;
          lst(n).c=c;
        }
                                         // normalize the function values
        lst(n).z(1,1)=((surf(r,c))-f.zo)/f.zm;
        lst(n).z(1,2)=((surf(r,c+1))-f.zo)/f.zm;
        lst(n).z(2,1)=((surf(r+1,c))-f.zo)/f.zm;
        lst(n).z(2,2)=((surf(r+1,c+1))-f.zo)/f.zm;
        for(size_t j=1;j<=2;j++)            // and clip to [0,1]
          for(size_t k=1;k<=2;k++) {
            if(lst(n).z(j,k)<0) lst(n).z(j,k)=0;
            else if(lst(n).z(j,k)>1) lst(n).z(j,k)=1;
          }
        flt_vctr mp(3);             // calc the midpoint of this element
        mp(1)=(lst(n).c-0.5)/surf.csize();
        mp(2)=(surf.rsize()-lst(n).r-0.5)/surf.rsize();
        mp(3)=0;
              // vp is an actual viewing position OUTSIDE of the 1x1x1 plotting
              // cube, 4 units along the viewPoint vector from the midpoint
              // of the cube, (0.5,0.5,0.5).
        flt_vctr vp(3);
        vp=viewPoint;
        vp*=(float)4.0;
        vp+=(float)0.5;
        float (*np)(const float&)=norm;
        lst(n).dist=accumulate(vp-mp,0.0,np); // calc distance mp to viewPoint
        for(size_t j=n;j>1;j--) {               // sort, farthest first
          if(lst(j).dist<lst(j-1).dist) break;
          swap(lst,lst,j,j-1);
        }
      }
    }
    header(1.2,1.2,0.6,0.5);              // initialize PS device
    *osp << "-90 rotate\n";                // to get upright image
    *osp << "2 setlinejoin\n";             // beveled joints between lines
    frame3d();                         // draw a frame
    *osp << lineWidth << " setlinewidth\n";
    float xel=1.0/(surf.csize()-1.0);
    float yel=1.0/(surf.rsize()-1.0);
    for(size_t j=1;j<=lst.size();j++) polygon(lst(j),xel,yel); //draw polygon
    *osp << "[" << 2*axisWidth << " " << 6*axisWidth << "] 0 setdash\n";
    frame3d();                      // redraw the frame dashed
    trailer();
}
  
  // UTILITIES

void 
Viewport::clearPlot() 
{ 
  for(size_t j=0;j<numPlot;j++) clearPlot(j); 
}

void 
Viewport::clearPlot(size_t j) 
{ 
  plot[j].resize(0,0); 
}

void 
Viewport::dashing(size_t j) 
{
  // sets dashing characteristics for the j'th line plot 
    if(j==0) return;
    *osp << "[ " << 30*lineWidth << " ";    // dashing
    for(int k=1;k<=j%5;k++) *osp << 20*lineWidth << " " << 5*lineWidth << " ";
    *osp << "] 0 setdash\n";
    if(j>=5) *osp << "0.5 setgray\n";       // gray lines
}

void 
Viewport::setRotation() 
{
  // Uses Euler angles to initialize the rotation matrix which takes viewPoint
  // into the z-axis; viewPoint is normalized along the way.
    flt_vctr &v=viewPoint;   // short hand for viewPoint
    float (*np)(const float&)=norm;
    float x=v(1),y=v(2),z=v(3),t=(float)sqrt(accumulate(v,0.0,np));
    flt_mtrx &r=viewRotation;
    if(t==0) {
#ifdef EX_HANDLE
      throw gError("setRotation(): viewPoint has zero length.");
#else
      cerr << "setRotation(): viewPoint has zero length.\n";
      exit(1);
#endif
    }
    v/=t;                             // normalize v
    float dp=sqrt(x*x+y*y);           // length of projection onto x-y plane
    if(dp<=VP_TINY) {                // if already parallel to z, rot. is ident.
      r*=(float)0;
      r(1,1)=1;
      r(2,2)=1;
      r(3,3)=1;
    } else {                          // R has been calculated explicitly
      r(1,1)=x*z;
      r(1,2)=y*z;
      r(1,3)=-dp*dp;
      r(2,1)=-y;
      r(2,2)=x;
      r(2,3)=0;
      r(3,1)=dp*x;
      r(3,2)=dp*y;
      r(3,3)=dp*z;
      r/=dp;
    }
}
 
void 
Viewport::frameSet1D() 
{
  // Sets up the default frame for a 2-d parametric plot.
    float step;
    if(xreset==1) {    // check for default frame
      float mx=-FLT_MAX,mn=FLT_MAX;     // search for max and min
      for(size_t j=0; j<numPlot; j++) {       // step over the separate curves
        if(plot[j].rsize()==0) continue;      // skip unused plots
        mx=max(mx,max(plot[j].col(1)));        // find max and min
        mn=min(mn,min(plot[j].col(1)));
      }
      if(mn==mx) {                        // check for a zero interval
        mn-=0.5*mn;                       // and create an arbitrary one
        mx+=0.5*mx;
      }
      f.xo=mn;
      f.xm=mx-mn;
    }
    step=f.xm/(f.xn-1.0);
    for(size_t j=1;j<=f.xn;j++) f.xtics(j)=f.xo+step*(j-1);
  
    if(yreset==1) {    // check for default frame
      float mx=-FLT_MAX,mn=FLT_MAX;     // search for max and min
      for(size_t j=0; j<numPlot; j++) {       // step over the separate curves
        if(plot[j].rsize()==0) continue;      // skip unused plots
        mx=max(mx,max(plot[j].col(2)));        // find max and min
        mn=min(mn,min(plot[j].col(2)));
      }
      if(mn==mx) { // check for a zero interval and create an arbitrary one
        mn=(float)rint(mn-1.0);
        mx=(float)rint(mx+1.0);
      } else {
        float round=pow(10,floor(log10(mx-mn))); // exponent of difference
        mn=(float)(round*floor(mn/round));       // round down to nearest whole
        mx=(float)(round*ceil(mx/round));        // round up
      }
      f.yo=mn;
      f.ym=mx-mn;
    }
    step=f.ym/(f.yn-1.0);
    for(size_t j=1;j<=f.yn;j++) f.ytics(j)=f.yo+step*(j-1);
}
 
void 
Viewport::frameSet2D() 
{
  // Sets up the default frame for either a contour plot or a 3-d frame plot.
    float step;
    if(xreset==1) {    // check for default frame
      f.xo=1;
      f.xm=surf.rsize()-1;
    }
    step=f.xm/(f.xn-1.0);
    for(size_t j=1;j<=f.xn;j++) f.xtics(j)=f.xo+step*(j-1);
  
    if(yreset==1) {    // check for default frame
      f.yo=1;
      f.ym=surf.csize()-1;
    }
    step=f.ym/(f.yn-1.0);
    for(size_t j=1;j<=f.yn;j++) f.ytics(j)=f.yo+step*(j-1);
    if(zreset==1) {    // check for default frame
      float mx=max(surf);         // calculate the max and min
      float mn=min(surf);
      if(mn==mx) { // check for a zero interval and create an arbitrary one
        mn=(float)rint(mn-1.0);
        mx=(float)rint(mx+1.0);
      } else {
        float round=pow(10,floor(log10(mx-mn))); // exponent of difference
        mn=(float)(round*floor(mn/round));       // round down to nearest whole
        mx=(float)(round*ceil(mx/round));        // round up
      }
      f.zo=mn;
      f.zm=mx-mn;
    }
    step=f.zm/(f.zn-1.0);
    for(size_t j=1;j<=f.zn;j++) f.ztics(j)=f.zo+step*(j-1);
}
 
void 
Viewport::p3dtop2d(flt_vctr &p3d,flt_vctr &p2d) 
{
    flt_vctr mp(3),pperp(3);
    mp(1)=mp(2)=mp(3)=0.5;        // midpoint of 3-d plot
//    pperp=(p3d-mp);               // p3d relative to mp
assign(p3d-mp,pperp);               // p3d relative to mp
    pperp*=stretch;     // stretch the vector for cosmetic reasons
  // take the projection onto the plane parallel to the viewPort but passing
  // through the midpoint. Note that viewPoint is unit normal to the viewPort.
    pperp-=sum(pperp*viewPoint)*viewPoint;
  // rescaling by similar triangles gives the projection on the viewPort itself.
    pperp*=alpha;
  // now rotate pperp into the x-y plane by multiplying by the rotation matrix
  // which rotates viewPoint into (0,0,1)
    dot(viewRotation,pperp,pperp);
    p2d(1)=pperp(1);              // and copy into p2d to return
    p2d(2)=pperp(2);
}
  
 
float 
Viewport::contain(size_t r,size_t c) 
{
  // returns surf(r,c).  If r and c lie outside the ranges
  // [1,surf.rsize()] and [1,surf.csize()] respectively, then it assumes the 
  // continued function is flat outside the grid.
    size_t rcont,ccont;
    if(r<1) rcont=1;         // contain the row index
    else if(r>surf.rsize()) rcont=surf.rsize();
    else rcont=r;
    if(c<1) ccont=1;         // contain the column index
    else if(c>surf.csize()) ccont=surf.csize();
    else ccont=c;
  
    return(surf(rcont,ccont));
}
  
 
bool 
Viewport::testContour(flt_mtrx &ref,flt_vctr &cnt,size_t r,size_t c) 
{
  // takes a point and a matrix of values and a vector of contour values and
  // decides if it is a contour point.
    if(r<1 || c<1 || r>ref.rsize() || c>ref.csize()) return 0;
    size_t lev,lev2;                      // temporary contour levels
         // set the contour level for this point
    for(lev=0;lev<numContour;lev++) if(ref(r,c)<cnt(lev+1)) break;
         // check nearest neighbors
    if(r!=1) {                   // upper
      for(lev2=0;lev2<numContour;lev2++)
        if(ref(r-1,c)<cnt(lev2+1)) break;  // decide the level of this point
      if(lev2<lev) return 1;
    }
    if(c!=1) {                   // left
      for(lev2=0;lev2<numContour;lev2++)
        if(ref(r,c-1)<cnt(lev2+1)) break;  // decide the level of this point
      if(lev2<lev) return 1;
    }
    if(r!=ref.rsize()) {          // lower
      for(lev2=0;lev2<numContour;lev2++)
        if(ref(r+1,c)<cnt(lev2+1)) break;  // decide the level of this point
      if(lev2<lev) return 1;
    }
    if(c!=ref.csize()) {          // right
      for(lev2=0;lev2<numContour;lev2++)
        if(ref(r,c+1)<cnt(lev2+1)) break;  // decide the level of this point
      if(lev2<lev) return 1;
    }
    return 0;
  }
  
 
float 
Viewport::interpolate(flt_vctr &p) 
{
  // interpolate a z-value from a matrix of values and a point which does not
  // lie on the matrix of values.
    float z;
              // first calculate the normalized position in the square
    size_t c=(size_t)(p(1)*(surf.csize()-1.0) + 1);     // rounds down
    float crem= p(1)*(surf.csize()-1.0) + 1 - c;
    size_t r=(size_t)((1.0-p(2))*(surf.rsize()-1.0) + 1);
    float rrem= (1.0-p(2))*(surf.rsize()-1.0) + 1 - r;
    switch(order) {            // find the interpolation order
      case 3: {          // cubic interpolation
        flt_mtrx f(4,4);      // matrix of function values
        f(1,1)=contain(r-1,c-1);
        f(1,2)=contain(r-1,c);
        f(1,3)=contain(r-1,c+2);
        f(1,4)=contain(r-1,c+2);
  
        f(2,1)=contain(r,c-1);
        f(2,2)=contain(r,c);
        f(2,3)=contain(r,c+2);
        f(2,4)=contain(r,c+2);
  
        f(3,1)=contain(r+1,c-1);
        f(3,2)=contain(r+1,c);
        f(3,3)=contain(r+1,c+2);
        f(3,4)=contain(r+1,c+2);
  
        f(4,1)=contain(r+2,c-1);
        f(4,2)=contain(r+2,c);
        f(4,3)=contain(r+2,c+2);
        f(4,4)=contain(r+2,c+2);
  
              // now label function values clockwise from upper-left:
        flt_vctr fv(16);
        size_t ind;
        for(size_t j=2;j<=3;j++)
          for(size_t k=2;k<=3;k++) {
            if(j==2 && k==2) ind=1;
            else if(j==2 && k==3) ind=2;
            else if(j==3 && k==3) ind=3;
            else ind=4;
            fv(ind)=f(j,k);      // function values
            fv(ind+4)=(f(j,k+1)-f(j,k-1))/2.0; // row-direction derivatives
            fv(ind+8)=(f(j+1,k)-f(j-1,k))/2.0; // column-direction derivatives
            fv(ind+12)=(f(j+1,k+1)-f(j+1,k-1)-f(j-1,k+1)+f(j-1,k-1))/4.0; // r&c
          }
             // define a transformation matrix (Num. Rec. In C, p107)
        static float wtp[16*16]=
            { 1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
            0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,
            -3,0,0,3,0,0,0,0,-2,0,0,-1,0,0,0,0,
            2,0,0,-2,0,0,0,0,1,0,0,1,0,0,0,0,
            0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,
            0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,
            0,0,0,0,-3,0,0,3,0,0,0,0,-2,0,0,-1,
            0,0,0,0,2,0,0,-2,0,0,0,0,1,0,0,1,
            -3,3,0,0,-2,-1,0,0,0,0,0,0,0,0,0,0,
            0,0,0,0,0,0,0,0,-3,3,0,0,-2,-1,0,0,
            9,-9,9,-9,6,3,-3,-6,6,-6,-3,3,4,2,1,2,
            -6,6,-6,6,-4,-2,2,4,-3,3,3,-3,-2,-1,-1,-2,
            2,-2,0,0,1,1,0,0,0,0,0,0,0,0,0,0,
            0,0,0,0,0,0,0,0,2,-2,0,0,1,1,0,0,
            -6,6,-6,6,-3,-3,3,3,-4,4,2,-2,-2,-2,-1,-1,
            4,-4,4,-4,2,2,-2,-2,2,-2,-2,2,1,1,1,1};
  
        flt_mtrx wt(16,16);   // make a mMatrix out of this data
        wt.container()=wtp;
        dot(wt,fv,fv);                 // matrix multiply
        float rcum=1,ccum=1;
        size_t l=1;
        z=0;
        for(size_t j=1;j<=4;j++) {
          rcum=1;
          for(size_t k=1;k<=4;k++) {
            z+=fv(l++)*rcum*ccum;
            rcum*=rrem;
          }
          ccum*=crem;
        }
        break;
      }
      default: {                   // default is linear interpolation
        flt_vctr f(4);      // function values clockwise from upper-left
        f(1)=contain(r,c);
        f(2)=contain(r,c+1);
        f(3)=contain(r+1,c+1);
        f(4)=contain(r+1,c);
        z=(1-rrem)*(1-crem)*f(1) + (1-rrem)*crem*f(2);
        z+= rrem*crem*f(3) + rrem*(1-crem)*f(4);
        break;
      }
    }
    return z;
}
  
  // POSTSCRIPT GENERATING ROUTINES
 
void 
Viewport::header(float xsize,float ysize,float xoff=0.0,float yoff=0.0)
{
  // Dumps an Encapsulated PostScript header to osp.  The bounding box is
  // specified by (0,0) in the lower left and (xsize,ysize) in the upper right.
  // The second two give the translation of the origin and create a border.
  // These units are in the scaled coordinate system.
    *osp << "%!PS-Adobe-2.0 EPSF-2.0\n";
    *osp << "%%Title: UNTITLED\n";
    *osp << "%%Creator: matgraph.t\n";
    struct timeval tv;             // get the date and echo it
    struct timezone tz;
    gettimeofday(&tv,&tz);
    *osp << "%%CreationDate: " << ctime(&(tv.tv_sec));
    *osp << "%%For: Viewport\n";
    *osp << "%%DocumentFonts: (atend)\n";
    *osp << "%%Pages: 0 0\n";
    *osp << "%%BoundingBox: 0.0 0.0 " << xsize*72*hscale;
    *osp << " " << ysize*72*vscale << "\n";
    *osp << "%%EndComments\n\n";
    *osp << 72*hscale << " " << 72*vscale << " scale\n";  // set scaling
    *osp << xoff << " " << yoff << " translate\n"; // move origin, create border
    *osp << "/horshow { dup stringwidth pop -0.5 mul " << -1.25*fontSize;
    *osp << " rmoveto show } def\n";
    *osp << "/horshow3d { 90 rotate dup stringwidth pop -0.5 mul ";
    *osp << -0.75*fontSize << " rmoveto show -90 rotate } def\n";
    *osp << "/vertshow { " << -0.25*fontSize;
    *osp << " 0 rmoveto dup stringwidth" << " pop neg " << -0.4*fontSize;
    *osp << " rmoveto show } def\n\n";
    *osp << "/vertshow3d { 90 rotate " << -0.25*fontSize;
    *osp << " 0 rmoveto dup stringwidth" << " pop neg " << -0.4*fontSize;
    *osp << " rmoveto show -90 rotate } def\n\n";
}
 
void 
Viewport::trailer() 
{
  // Dumps an Encapsulated PostScript trailer to osp
    *osp << "showpage\n";
    *osp << "%%Trailer\n";
    *osp << "%%DocumentFonts: " << labelFont << "\n" << flush;
}
 
void 
Viewport::polygon(Quadril &q,float xel,float yel) 
{
  // writes the postscript for a filled quadrilateral with data specified by q.
    flt_vctr p3d(3), p2d(2);
            // normalize viewPoint mVector
    float (*np)(const float&)=norm;
    viewPoint/=(float)sqrt(accumulate(viewPoint,0.0,np));
    p3d(1)=(q.c-1)*xel;              // init left-rear point
    p3d(2)=1-(q.r-1)*yel;
    p3d(3)=q.z(1,1);
    p3dtop2d(p3d,p2d);
    *osp << "newpath\n";
    *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path
    p3d(1)+=xel;                     // init right-rear point
    p3d(3)=q.z(1,2);
    p3dtop2d(p3d,p2d);
    *osp << p2d(1) << " " << p2d(2) << " lineto\n";
    p3d(2)-=yel;                     // init right-front point
    p3d(3)=q.z(2,2);
    p3dtop2d(p3d,p2d);
    *osp << p2d(1) << " " << p2d(2) << " lineto\n";
    p3d(1)-=xel;                     // init left-front point
    p3d(3)=q.z(2,1);
    p3dtop2d(p3d,p2d);
    *osp << p2d(1) << " " << p2d(2) << " lineto\n";
    *osp << "closepath\n";
    *osp << "gsave\n";
    *osp << background << " setgray\n";
    *osp << "fill\n";
    *osp << "grestore\n";
    *osp << "stroke\n";
}
 
void 
Viewport::frame2d() 
{
  // puts a frame and sets the clipping path to the box ((0,0),(1,1))
    *osp << "/" << labelFont << " findfont\n";     // set the font for labels
    *osp << fontSize << " scalefont\n";
    *osp << "setfont\n";
    *osp << "2 setlinecap\n";              // set for projecting caps on lines
    if(title!=0) {
      *osp << " 0 " << 1 + 2*fontSize << " moveto\n";
      *osp << "(" << title << ") show\n";
      *osp << "stroke\n";
    }
    if(frame==1) {
      *osp << "newpath\n";
      *osp << axisWidth << " setlinewidth\n";
      *osp << "0 1 moveto\n";
      *osp << "0 0 lineto\n";
      *osp << "1 0 lineto\n";
      *osp << "stroke\n";
      if(tics==1) {          // draw the given tickmarks
        for(size_t j=1;j<=f.xn;j++) {      // first x
          *osp << (f.xtics(j)-f.xo)/f.xm << " 0 moveto\n"; // orient on the axis
          *osp << "0 " << ticLength << " rlineto\n";   // draw the ticmark
          *osp << (f.xtics(j)-f.xo)/f.xm << " 0 moveto\n"; // orient on the axis
          if(label==1)
            osp->form("(%.3g) horshow\n",f.xtics(j)*xscale+xoff);
        }
        for(size_t j=1;j<=f.yn;j++) {      // then y
          *osp << "0 " << (f.ytics(j)-f.yo)/f.ym << " moveto\n";
          *osp << ticLength << " 0  rlineto\n";   // draw the ticmark
          *osp << "0 " << (f.ytics(j)-f.yo)/f.ym << " moveto\n";
          if(label==1)
            osp->form("(%.3g) vertshow\n",f.ytics(j)*yscale+yoff);
        }
      }                                  // end ticmark drawing
      *osp << "stroke\n";                  // plot the frame
    }                                    // end frame draw
    *osp << "newpath\n";                   // set the clipping path
    *osp << "0 0 moveto\n";
    *osp << "0 1 lineto\n";
    *osp << "1 1 lineto\n";
    *osp << "1 0 lineto\n";
    *osp << "closepath\n";
    *osp << "clip\n";
}
 
void 
Viewport::frame3d() 
{
  // puts a 3-d frame
    *osp << "/" << labelFont << " findfont\n";     // set the font for labels
    *osp << fontSize/2.0 << " scalefont\n";
    *osp << "setfont\n";
    *osp << "2 setlinecap\n";              // set for projecting caps on lines
    flt_vctr p3d(3), p2d(2);
    if(title!=0) {
      *osp << "newpath\n";
      p3d(1)=0;
      p3d(2)=1;
      p3d(3)=1.5;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path  (1,0,0)
      *osp << "(" << title << ") horshow3d\n";
      *osp << "stroke\n";
    }
    if(frame==1) {
      *osp << axisWidth << " setlinewidth\n";
      *osp << "newpath\n";
      p3d(1)=1;
      p3d(2)=0;
      p3d(3)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path  (1,0,0)
      p3d(1)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,0,0)->(0,0,0)
      p3d(2)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,0,0)->(0,1,0)
      *osp << "stroke\n";
      *osp << "newpath\n";
      *osp << 0.5*axisWidth << " setlinewidth\n";  // now non-axis parts
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // (0,0,0)->(0,1,0)
      p3d(1)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,1,0)->(1,1,0)
      p3d(2)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,1,0)->(1,0,0)
      *osp << "stroke\n";
      *osp << "newpath\n";
      *osp << axisWidth << " setlinewidth\n";      // z-axis
      p3d(1)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path (0,0,0)
      p3d(3)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,0,0)->(0,0,1)
      *osp << "stroke\n";
      *osp << "newpath\n";
      *osp << 0.5*axisWidth << " setlinewidth\n";  // complete the box
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // (0,0,0)->(0,0,1)
      p3d(2)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,0,1)->(0,1,1)
      p3d(1)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,1,1)->(1,1,1)
      p3d(2)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,1,1)->(1,0,1)
      p3d(1)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,0,1)->(0,0,1)
      p3d(2)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path (0,1,1)
      p3d(3)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (0,1,1)->(0,1,0)
      p3d(1)=1;
      p3d(3)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path (1,1,1)
      p3d(3)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,1,1)->(1,1,0)
      p3d(2)=0;
      p3d(3)=1;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " moveto\n";  // start path (1,0,1)
      p3d(3)=0;
      p3dtop2d(p3d,p2d);
      *osp << p2d(1) << " " << p2d(2) << " lineto\n";  // (1,0,1)->(1,0,0)
      *osp << "stroke\n";
      *osp << axisWidth << " setlinewidth\n";  // back to axis lines
      if(tics==1) {          // draw the given tickmarks
        *osp << "newpath\n";
        flt_vctr mp(3);                     // midpoint of 3-d frame
        float a;                                 // angle for drawing labels
        mp(1)=mp(2)=mp(3)=0.5;
        for(size_t j=1;j<=f.xn;j++) {      // first x
          p3d(1)=(f.xtics(j)-f.xo)/f.xm;
          p3d(2)=0;
          p3d(3)=0;
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // move to first tic
          p3d(2)=ticLength;
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " lineto\n";   // draw the ticmark
          if(label==1) {
            p3d(1)=-0.05 + (f.xtics(j)-f.xo)/f.xm;
            p3d(2)=0;
            p3d(3)=0;
            p3dtop2d(p3d,p2d);
            *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // draw the label
            osp->form("(%.3g) horshow3d\n",f.xtics(j)*xscale+xoff);
          }
        }
        for(size_t j=1;j<=f.yn;j++) {      // then y
          p3d(1)=1;
          p3d(2)=(f.ytics(j)-f.yo)/f.ym;
          p3d(3)=0;
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // move to first tic
          p3d(1)=1-ticLength;
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " lineto\n";   // draw the ticmark
          if(label==1) {
            p3d(1)=1;
            p3d(2)=0.05 + (f.ytics(j)-f.yo)/f.ym;
            p3d(3)=0;
            p3dtop2d(p3d,p2d);
            *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // draw the label
            osp->form("(%.3g) horshow3d\n",f.ytics(j)*yscale+yoff);
          }
        }
        for(size_t j=1;j<=f.zn;j++) {      // then z
          p3d(1)=0;
          p3d(2)=0;
          p3d(3)=(f.ztics(j)-f.zo)/f.zm;
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // move to first tic
          p3d(1)=ticLength/sqrt(2.0);
          p3d(2)=ticLength/sqrt(2.0);
          p3dtop2d(p3d,p2d);
          *osp << p2d(1) << " " << p2d(2) << " lineto\n";   // draw the ticmark
          if(label==1) {
            p3d(1)=0;
            p3d(2)=0;
            p3d(3)=0.05 + (f.ztics(j)-f.zo)/f.zm;
            p3dtop2d(p3d,p2d);
            *osp << p2d(1) << " " << p2d(2) << " moveto\n";   // draw the label
            osp->form("(%.3g) vertshow3d\n",f.ztics(j)*zscale+zoff);
          }
        }
        *osp << "stroke\n";                // plot the labels and ticmarks
      }                                  // end ticmark drawing
    }                                    // end frame draw
}
  
template<class T,class F>
static
void
initPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,size_t dash)
{
// plots (t,f(t)) stepping over [tmin,tmax] 
// t and f(t) must be castable to float

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initPlot(): too many steps.");
#else
    cerr << "initPlot(): too many steps.\n";
    exit(1);
#endif
  }
  vp.plot[dash].resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)tmin;
    vp.plot[dash](j,2)=(float)f(tmin);
  }
}

template<class T,class F,class Fl>
static
void
initPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,Fl filter,size_t dash)
{
// plots (t,filter(f(t))) stepping over [tmin,tmax] 

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initPlot(): too many steps.");
#else
    cerr << "initPlot(): too many steps.\n";
    exit(1);
#endif
  }
  vp.plot[dash].resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)tmin;
    vp.plot[dash](j,2)=(float)filter(f(tmin));
  }
}

template<class T,class F,class M>
static
void
initImplPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,M m,size_t dash)
{
// plots (t,f(m(t))) stepping over [tmin,tmax] 
// t and f(m(t)) must be castable to float

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initImplPlot(): too many steps.");
#else
    cerr << "initImplPlot(): too many steps.\n";
    exit(1);
#endif
  }
  vp.plot[dash].resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)tmin;
    vp.plot[dash](j,2)=(float)f(m(tmin));
  }
}

template<class T,class F,class M,class Fl>
static
void
initImplPlot(Viewport& vp,T tmin,T tmax,size_t length,F f,M m,Fl filter,size_t dash)
{
// plots (t,filter(f(m(t)))) stepping over [tmin,tmax] 

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initImplPlot(): too many steps.");
#else
    cerr << "initImplPlot(): too many steps.\n";
    exit(1);
#endif
  }
  vp.plot[dash].resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)tmin;
    vp.plot[dash](j,2)=(float)filter(f(m(tmin)));
  }
}

template<class T,class Fx,class Fy>
static
void
initParamPlot(Viewport& vp,T tmin,T tmax,size_t length,Fx fx,Fy fy,size_t dash)
{
// plots (fx(t),fy(t)) stepping over [tmin,tmax] 
// fx(t) and fy(t) must be castable to float

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initParamPlot(): too many steps.");
#else
    cerr << "initParamPlot(): too many steps.\n";
    exit(1);
#endif
  }
  (vp.plot[dash]).resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)fx(tmin);
    vp.plot[dash](j,2)=(float)fy(tmin);
  }
}

template<class T,class Fx,class Fy,class Fl>
static
void
initParamPlot(Viewport& vp,T tmin,T tmax,size_t length,Fx fx,Fy fy,Fl filter,size_t dash)
{
// plots (filter(fx(t)),filter(fy(t))) stepping over [tmin,tmax] 

  T step=(T)((tmax-tmin)/(length-1.0));
  if(length>vp.oneDMesh) {
#ifdef EX_HANDLE
    throw gError("initParamPlot(): too many steps.");
#else
    cerr << "initParamPlot(): too many steps.\n";
    exit(1);
#endif
  }
  vp.plot[dash].resize(length,2);        // reset plot size
  for(size_t j=1;j<=length;j++,tmin+=step) {  // store points
    vp.plot[dash](j,1)=(float)filter(fx(tmin));
    vp.plot[dash](j,2)=(float)filter(fy(tmin));
  }
}

template<class T,class F>
static
void
initSurfPlot(Viewport& vp,T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f)
{
// plots f(x,y) stepping over [xmin,ymax] and [ymin,ymax]

  T xstep=(T)((xmax-xmin)/(xlen-1.0));
  T ystep=(T)((ymax-ymin)/(ylen-1.0));
  if(xlen*ylen>vp.twoDMesh) {
#ifdef EX_HANDLE
    throw gError("initSurf(): too many steps.");
#else
    cerr << "initSurf(): too many steps.\n";
    exit(1);
#endif
  }
  vp.xscale=(float)((xmax-xmin)/(xlen-1.0));  // set axis scalings
  vp.yscale=(float)((ymax-ymin)/(ylen-1.0));
  vp.xoff=(float)((xlen*xmin-xmax)/(xlen-1.0));  // set axis offsets
  vp.yoff=(float)((ylen*ymin-ymax)/(ylen-1.0));
  vp.surf.resize(xlen,ylen);        // reset plot size
  for(size_t r=1;r<=xlen;r++,xmin+=xstep) {  // store points
    T y=ymin;
    for(size_t c=1;c<=ylen;c++,y+=ystep)
      vp.surf(r,c)=(float)f(xmin,y);
  }
}

template<class T,class F,class Fl>
static
void
initSurfPlot(Viewport& vp,T xmin,T xmax,size_t xlen,T ymin,T ymax,size_t ylen,F f,Fl filter)
{
// plots filter(f(x,y)) stepping over [xmin,ymax] and [ymin,ymax]

  T xstep=(T)((xmax-xmin)/(xlen-1.0));
  T ystep=(T)((ymax-ymin)/(ylen-1.0));
  if(xlen*ylen>vp.twoDMesh) {
#ifdef EX_HANDLE
    throw gError("initSurf(): too many steps.");
#else
    cerr << "initSurf(): too many steps.\n";
    exit(1);
#endif
  }
  vp.xscale=(float)((xmax-xmin)/(xlen-1.0));  // set axis scalings
  vp.yscale=(float)((ymax-ymin)/(ylen-1.0));
  vp.xoff=(float)((xlen*xmin-xmax)/(xlen-1.0));  // set axis offsets
  vp.yoff=(float)((ylen*ymin-ymax)/(ylen-1.0));
  vp.surf.resize(xlen,ylen);        // reset plot size
  for(size_t r=1;r<=xlen;r++,xmin+=xstep) {  // store points
    T y=ymin;
    for(size_t c=1;c<=ylen;c++,ymin+=ystep)
      vp.surf(r,c)=(float)filter(f(xmin,y));
  }
}

#endif

