/*
Copyright (C) 1995 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the 
terms of the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

template<class T,class S>
void 
mMatrix<T,S>::rc_dealloc() 
{
  if(row_) {
    for(size_type r= _START_ ;r _COMP_ r_;r++) 
      if(row_[r]) {
        delete row_[r]; 
        row_[r]=(rc_type*)0;
      }
    delete [] (row_ + _START_); 
    row_=(rc_type**)0;
  } 
  if(col_) {
    for(size_type c= _START_ ;c _COMP_ c_;c++) 
      if(col_[c]) {
        delete col_[c]; 
        col_[c]=(rc_type*)0;
      }
    delete [] (col_ + _START_); 
    col_=(rc_type**)0;
  } 
  if(diag_) {
    delete diag_;
    diag_=(rc_type*)0;
  }
}

template<class T,class S>
mMatrix<T,S>& 
mMatrix<T,S>::transpose()
{
  if(rsize()==csize()) {        // if square
    for(size_type ri= _START_ +1; ri _COMP_ rsize(); ri++) 
      for(size_type ci= _START_ ; ci < ri; ci++) {  // compute transpose
        T t=(*this)(ri,ci);         // swap elements above and below diagonal
        (*this)(ri,ci)=(*this)(ci,ri);
        (*this)(ci,ri)=t;
      }
  } else {                      // if not square
    mMatrix<T,allocator<T> > t(c_,r_);    // allocate temporary storage
    for(size_type ri= _START_ ; ri _COMP_ rsize(); ri++) 
      for(size_type ci= _START_ ; ci _COMP_ csize(); ci++) 
        t(ci,ri)=(*this)(ri,ci);          // transpose to temporary
    resize(c_,r_);              // reverse dimensions
    for(size_type ri= _START_ ; ri _COMP_ rsize(); ri++) 
      for(size_type ci= _START_ ; ci _COMP_ csize(); ci++) 
        (*this)(ri,ci)=t(ri,ci);          // copy temporary back to (*this)
  }
  return (*this); 
}

template<class T,class S,class C>
C&
swap(const mMatrix<T,S>& a,C& result,size_t r1,size_t r2)
{
  if((void*)&a!=(void*)&result) {
    for(size_t r=1;r<=a.rsize();r++) 
      for(size_t c=1;c<=a.csize();c++) result(r,c)=a(r,c);
  }
  for(size_t c=1;c<=a.csize();c++) {
    T t=a(r1,c);               // in case a and result are the same object
    result(r1,c)=a(r2,c);
    result(r2,c)=t;
  }
  return result;
}

template<class T,class S,class C>
C&
swap(const mMatrix<T,S>& a,C& result,size_t r1,size_t c1,size_t r2,size_t c2)
{
  swap(a.array(),result.array(),(r1-1)*a.csize()_c1,(r2-1)*a.csize()_c2);
  return result;
}

template<class T,class S,class T2,class S2>
static mMatrix<T2,S2>&
copy_sub(const mMatrix<T,S>& a,size_t r1,size_t c1,mMatrix<T2,S2>& result,\
size_t r2,size_t c2,size_t rs,size_t cs)
{
// copies a rs by cs submatrix of a to a submatrix of result, 
// starting at the alignment points a(r1,c1) and result(r2,c2).
  for(size_t r=0;r<rs;r++)
    for(size_t c=0;c<cs;c++) result(r+r2,c+c2) = a(r+r1,c+c1);
  return result; 
}

// IO operators

template<class T,class S> 
static ostream& 
operator<<(ostream& s,const mMatrix<T,S>& a)
{
  for(S::size_type r= _START_ ;r _COMP_ a.rsize();r++) s << a.row(r);
  s << _ldelim_;
  return s;
}

template<class T,class S> 
static istream& 
operator>>(istream& s,mMatrix<T,S>& a)
{
  return s >> a.array();
}

template<class T,class S,class C,class F> 
static C& 
transform(const mMatrix<T,S>& a,C& result,F f)
{
  transform(a.array(),result.array(),f);
  return result;
}

template<class T,class S,class T2,class S2,class C,class F> 
static C& 
transform(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b,C& result,F f)
{
  transform(a.array(),b.array(),result.array(),f);
  return result;
}

template<class T,class S,class C,class F> 
static C& 
transform(const T& b,const mMatrix<T,S>& a,C& result,F f)
{
  transform(b,a.array(),result.array(),f);
  return result;
}

template<class T,class S,class C,class F> 
static C& 
transform(const mMatrix<T,S>& a,const T& b,C& result,F f)
{
  transform(a.array(),b,result.array(),f);
  return result;
}

#ifndef _MEMBER_TEMPLATES_
template<class T,class S,class T2,class S2> 
static mMatrix<T2,S2>&
assign(const mMatrix<T,S>& a,mMatrix<T2,S2>& result)
// temporary until member templates are implemented
{
  assign(a.array(),result.array());
  return result;
}
#endif        _MEMBER_TEMPLATES_

template<class T,class S,class T2,class F> 
static T2 
accumulate(const mMatrix<T,S>& a,T2 init,F f)
{
  return accumulate(a.array(),init,f);
}

template<class T,class S,class T2,class S2,class T3,class F> 
static T3 
accumulate(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b,T3 init,F f)
{
  return accumulate(a.array(),b.array(),init,f);
}

template<class T,class S,class T2,class F> 
static T2 
accumulate(const T& b,const mMatrix<T,S>& a,T2 init,F f)
{
  return accumulate(b,a.array(),init,f);
}

template<class T,class S,class T2,class F> 
static T2 
accumulate(const mMatrix<T,S>& a,const T& b,T2 init,F f)
{
  return accumulate(a.array(),b,init,f);
}

template<class T,class S> 
static T 
max(const mMatrix<T,S>& a)
{
  return max(a.array());
}

template<class T,class S> 
static long double 
sup(const mMatrix<T,S>& a)
{
  return sup(a.array());
}

template<class T,class S> 
static T 
min(const mMatrix<T,S>& a)
{
  return min(a.array());
}

template<class T,class S> 
static long double 
inf(const mMatrix<T,S>& a)
{
  return inf(a.array());
}

template<class T,class S> 
static T 
sum(const mMatrix<T,S>& a)
{
  return sum(a.array());
}

template<class T,class S> 
static T 
product(const mMatrix<T,S>& a) 
{
  return product(a.array());
}


template<class T,class S,class T2,class S2> 
static bool 
operator==(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b)
{
  return a.array()==b.array();
}

/* template<class T,class S,class T2,class S2> 
static bool 
operator!=(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b)
{
  return a.array()!=b.array();
} */

// Note: dot() is not the inner product unless T and T2 are real types.
template<class T,class S,class T2,class S2,class T3,class S3>
static mMatrix<T3,S3>&
dot(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b,mMatrix<T3,S3>& result) 
{
  if(((void*)&result==(void*)&a) || ((void*)&result==(void*)&b)) {
    mMatrix<T,allocator<T> > res(a.rsize(),b.csize());
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++)
      for(unsigned c= _START_ ;c _COMP_ b.csize();c++)
        res(r,c)=sum(a.row(r)*b.col(c));
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++)
      for(unsigned c= _START_ ;c _COMP_ b.csize();c++)
        result(r,c)=res(r,c);
  } else {
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++)
      for(unsigned c= _START_ ;c _COMP_ b.csize();c++)
        result(r,c)=sum(a.row(r)*b.col(c));
  }
  return result;
}

template<class T,class S,class T2,class S2,class T3,class S3>
static mVector<T3,S3>&
dot(const mMatrix<T,S>& a,const mVector<T2,S2>& b,mVector<T3,S3>& result) 
{
  if((void*)&result==(void*)&b) {
    mVector<T,allocator<T> > res(a.rsize());
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++) res(r)=sum(a.row(r)*b);
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++) result(r)=res(r);
  } else {
    for(unsigned r= _START_ ;r _COMP_ a.rsize();r++) result(r)=sum(a.row(r)*b);
  }
  return result;
}

template<class T,class S,class T2,class S2,class T3,class S3>
static mVector<T3,S3>&
dot(const mVector<T,S>& a,const mMatrix<T2,S2>& b,mVector<T3,S3>& result) 
{
  if((void*)&result==(void*)&a) {
    mVector<T,allocator<T> > res(b.csize());
    for(unsigned c= _START_ ;c _COMP_ b.csize();c++) res(c)=sum(a*b.col(c));
    for(unsigned c= _START_ ;c _COMP_ b.csize();c++) result(c)=res(c);
  } else {
    for(unsigned c= _START_ ;c _COMP_ b.csize();c++) result(c)=sum(a*b.col(c));
  }
  return result;
}


// Define delayed execution classes

template<class C>
class rc_allocator
{
public:
  typedef C::value_type         value_type;
  typedef Delay_rc<C> 		pointer;
  typedef C::size_type          size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class C>
class Delay_rc
{
public:
  typedef C::value_type         value_type;
  typedef value_type&           reference;
  typedef const value_type&     const_reference;
  typedef C::size_type          size_type;
  typedef C::pointer            c_type;

  Delay_rc() {}
  Delay_rc(size_type n,const_reference t=value_type()) {}
  Delay_rc(const Delay_rc &d) : a_(d.a_), o_(d.o_), k_(d.k_) {}
  Delay_rc(const c_type& g,size_type o,size_type k) : a_(g), o_(o), k_(k) {}
 
  Delay_rc& operator=(const Delay_rc& d) { a_=d.a_; o_=d.o_; k_=d.k_; }
  Delay_rc& operator++() { return (*this); }
  Delay_rc& operator--() { return (*this); }
  reference operator[](size_type n) { return a_[o_+n*k_]; }
  value_type operator[](size_type n) const { return a_[o_+n*k_]; }
private:
  c_type a_;
  size_type o_,k_;
};

template<class T,class S,class F> 
static mMatrix<T, unary_allocator<S,F>  > 
apply(const mMatrix<T,S>& a,F f)
{
  mMatrix<T,unary_allocator<S,F> > r;
  r.resize(a.rsize(),a.csize());
  r.container()= Delay_unary<S,F>(a.container(),f);
  return r;
}

template<class T,class S,class F> 
static mMatrix<T, binary_allocator<scalar_allocator<T>,S,F> > 
apply(const T& b,const mMatrix<T,S>& a,F f)
{
  mMatrix<T,binary_allocator<scalar_allocator<T>,S,F> > r;
  r.resize(a.rsize(),a.csize());
  r.container()= Delay_binary<scalar_allocator<T>,S,F>(
	scalar_pointer(a.rsize()*a.csize(),b),a.container(),f);
  return r;
}

template<class T,class S,class F> 
static mMatrix<T, binary_allocator<S,scalar_allocator<T>,F> > 
apply(const mMatrix<T,S>& a,const T& b,F f)
{
  mMatrix<T,binary_allocator<S,scalar_allocator<T>,F> > r;
  r.resize(a.rsize(),a.csize());
  r.container()= Delay_binary<S,scalar_allocator<T>,F>(
	a.container(),scalar_pointer(a.rsize()*a.csize(),b),f);
  return r;
}

template<class T,class S,class T2,class S2,class F> 
static mMatrix<T, binary_allocator<S,S2,F> > 
apply(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b,F f)
{
  mMatrix<T,binary_allocator<S,S2,F> > r;
  r.resize(a.rsize(),a.csize());
  r.container()= Delay_binary<S,S2,F>(a.container(), b.container(),f);
  return r;
}


// MACRO DEFINITIONS

#define binary(_SYMBOL_,_SMBL_)						\
									\
template<class T,class S> 						\
static mMatrix<T,S>& 							\
operator _SYMBOL_ ## =(mMatrix<T,S>& a,T b)				\
{									\
  a.array() _SYMBOL_ ## = b;						\
  return a;								\
}									\
									\
template<class T,class S,class T2,class S2> 				\
static mMatrix<T,S>& 							\
operator _SYMBOL_ ## =(mMatrix<T,S>& a,const mMatrix<T2,S2>& b)		\
{									\
  a.array() _SYMBOL_ ## = b.array();					\
  return a;								\
}									\
									\
template<class T,class S> 						\
static mMatrix< T, _SMBL_ ## allocator<S,scalar_allocator<T> > >	\
operator _SYMBOL_(const mMatrix<T,S>& a,const T& b)			\
{									\
  mMatrix<T,_SMBL_ ## allocator<S,scalar_allocator<T> > > r;		\
  r.resize(a.rsize(),a.csize());					\
  r.container()= Delay_ ## _SMBL_<S,scalar_allocator<T> >(		\
	a.container(),scalar_pointer<T>(a.rsize()*a.csize(),b));	\
  return r;								\
}									\
									\
template<class T,class S> 						\
static mMatrix<T, _SMBL_ ## allocator<scalar_allocator<T>,S> > 		\
operator _SYMBOL_(const T& b,const mMatrix<T,S>& a)			\
{									\
  mMatrix<T,_SMBL_ ## allocator<scalar_allocator<T>,S> > r;		\
  r.resize(a.rsize(),a.csize());					\
  r.container()= Delay_ ## _SMBL_<scalar_allocator<T>,S>(		\
	scalar_pointer<T>(a.rsize()*a.csize(),b),a.container());	\
  return r;								\
}									\
									\
template<class T,class S,class T2,class S2>				\
static mMatrix<T, _SMBL_ ## allocator<S,S2> >				\
operator _SYMBOL_(const mMatrix<T,S>& a,const mMatrix<T2,S2>& b)	\
{									\
  mMatrix<T,_SMBL_ ## allocator<S,S2> > r;				\
  r.resize(a.rsize(),a.csize());					\
  r.container()= Delay_ ## _SMBL_<S,S2>(a.container(),b.container());	\
  return r;								\
}

#define unary(_SYMBOL_,_SMBL_)						\
									\
template<class T,class S> 						\
static mMatrix< T, _SMBL_ ## allocator<S> >				\
operator _SYMBOL_(const mMatrix<T,S>& a) 				\
{									\
  mMatrix<T,_SMBL_ ## allocator<S> > r;					\
  r.resize(a.rsize(),a.csize());					\
  r.container()= Delay_ ## _SMBL_<S>(a.container());			\
  return r;								\
}


// MACRO EXPANSIONS

binary(+,pl_)
binary(-,mi_)
binary(*,mu_)
binary(/,di_)
binary(%,pe_)
binary(&,an_)
binary(|,or_)
binary(<<,ll_)
binary(>>,gg_)
#undef binary

unary(+,u_pl_)
unary(-,u_mi_)
unary(~,u_ti_)
unary(!,u_ba_)
#undef unary

