# GNU Enterprise Application Server - Language interface: list of objects
#
# Copyright 2003-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: ObjectList.py 9953 2009-10-11 18:50:17Z reinhard $

from Object import Object
from gnue.common.datasources import GConditions

CACHE_STEP = 10

# ===========================================================================
# Class: ObjectList
# ===========================================================================

class ObjectList:

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------

  def __init__ (self, session, proxy, classname, cond, sort, properties,
      definition):

    self.__session   = session
    self.__proxy     = proxy
    self.__list      = []
    self.__geasList  = None

    self.classname   = classname
    self.conditions  = self.__normalizeCondition (cond)
    self.sortOrder   = self.__normalizeSorting (sort)
    self.properties  = self.__normalizeProperties (properties)
    self.definition  = definition

    self.__cacheStep = CACHE_STEP
    self.__buildList ()


  # ---------------------------------------------------------------------------
  # Return the number of elements in this sequence
  # ---------------------------------------------------------------------------

  def __len__ (self):

    return self.__geasList.count ()


  # ---------------------------------------------------------------------------
  # Truth value testing
  # ---------------------------------------------------------------------------

  def __nonzero__ (self):

    return len (self.__list) > 0


  # -------------------------------------------------------------------------
  # Return a slice or a specific element of this sequence
  # -------------------------------------------------------------------------

  def __getitem__ (self, index):

    try:
      if isinstance (index, slice):
        if index.start < 0:
          self.__fillup ()

        return self.__list [index.start:index.stop]
      else:
        if index < 0:
          self.__fillup ()

        return self.__list [index]

    except:
      # try to populate the sequence with missing entries
      if self.__populateList ():

        # if successful try to return entry/entries again
        return self.__getitem__ (index)
      
      else:
        raise IndexError
      

  # -------------------------------------------------------------------------
  # Create an initial sequence
  # -------------------------------------------------------------------------

  def __buildList (self):

    self.__geasList = self.__proxy.request (self.classname, self.conditions,
                                            self.sortOrder, self.properties)
    self.__list = []
    self.__populateList ()

      
  # -------------------------------------------------------------------------
  # Populate Cache
  # -------------------------------------------------------------------------

  def __populateList (self):

    rset = self.__geasList.fetch (len (self.__list), self.__cacheStep)
    for row in rset:
      obj = Object (self.__session, self.__proxy, self.classname, row [0],
                    self.definition)
      self.__list.append (obj)

    # We double the cache-size for the next call to fetch (). This boosts
    # performance for fetching larger lists
    self.__cacheStep *= 2

    # return number of updated rows
    return len (rset)
    

  # ---------------------------------------------------------------------------
  # Fetch all records from the list
  # ---------------------------------------------------------------------------

  def __fillup (self):

    if len (self.__list) < len (self):
      while self._populateList ():
        pass


  # ---------------------------------------------------------------------------
  # Normalize a condition given to a find
  # ---------------------------------------------------------------------------

  def __normalizeCondition (self, conditions):
    """
    This function returns a normalized form of a condition. This means all
    field names are fully qualified and all constants given as Object instances
    will be replaced by their gnue_id.

    @param conditions: a condition as sequence, dictionary or condition tree

    @return: the normalized condition in prefix notation
    """

    if conditions is None or not len (conditions):
      return None

    cTree = GConditions.buildCondition (conditions)

    # Replace all constant values represented by an Object instance with the
    # apropriate gnue_id
    for item in cTree.findChildrenOfType ('GCCConst', True, True):
      if isinstance (item.value, Object):
        item.value = item.value.gnue_id

    # Make sure all field names are fully qualified
    for item in cTree.findChildrenOfType ('GCCField', True, True):
      item.name = self.__session.qualifyMultiple (item.name)

    return cTree.prefixNotation ()



  # ---------------------------------------------------------------------------
  # Create a sequence of fully qualified property names
  # ---------------------------------------------------------------------------

  def __normalizeProperties (self, sequence):
    """
    This function returns a sequence of fully qualified names.

    @param sequence: sequence of names
    @return: sequence with fully qualified names
    """

    return [self.__session.qualifyMultiple (p) for p in sequence]


  # ---------------------------------------------------------------------------
  # Create a normalized sort order
  # ---------------------------------------------------------------------------

  def __normalizeSorting (self, sorting):
    """
    This function returns a normalized sorting sequence. Such a sequence has
    all fields fully qualified.

    @param sorting: sorting sequence to be normalized
    @return: sorting sequence with all fields fully qualified
    """

    result = []

    for item in sorting:
      if isinstance (item, dict):
        item ['name'] = self.__session.qualifyMultiple (item ['name'])

      elif isinstance (item, list):
        item [0] = self.__session.qualifyMultiple (item [0])

      elif isinstance (item, tuple):
        data = list (item)
        data [0] = self.__session.qualifyMultiple (data [0])
        item = tuple (data)

      else:
        item = self.__session.qualifyMultiple (item)

      result.append (item)

    return result
