/* cmd-my-revision-library.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "tla/libarch/my.h"
#include "tla/libarch/cmd-my-revision-library.h"



static t_uchar * usage = "[options] [dir]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n"
                                   cfg__std__release_id_string
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_errname, "e", "errname", 1, \
      "specify program name for errors") \
  OP (opt_delete, "d", "delete", 0, \
      "unspecify your revision library") \
  OP (opt_silent, "s", "silent", 0, \
      "suppress reassuring messages")

t_uchar arch_cmd_my_revision_library_help[] = ("print or change your revision library path\n"
                                               "With no argument, and without -d, print the path to your revision\n"
                                               "library.\n"
                                               "\n"
                                               "With an argument, record DIR as the path to your revision library\n"
                                               "in ~/.arch-params/=revision-library\n"
                                               "\n"
                                               "With the option -d (--delete) and no argument, ensure that\n"
                                               "you do not have a revision library path set in ~/.arch-params.\n"
                                               "\n"
                                               "If no revision library is set, the program exits with status 1,\n"
                                               "printing an error message unless the -s (--silent) option is given.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



enum operation
{
  op_print,
  op_set,
  op_delete,
};

int
arch_cmd_my_revision_library (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  char * dir;
  char * errname;
  enum operation op = op_print;
  int silent;

  errname = argv[0];
  silent = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_my_revision_library_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_errname:
          {
            if (!option->arg_string[0])
              errname = 0;
            else
              errname = str_save (0, option->arg_string);
            break;
          }

        case opt_delete:
          {
            op = op_delete;
            break;
          }

        case opt_silent:
          {
            silent = 1;
            break;
          }
        }
    }

  if (op == op_delete)
    {
      if (argc != 1)
        goto usage_error;
    }
  else if (argc == 2)
    {
      op = op_set;
      dir = argv[1];
    }
  else if (argc == 1)
    {
      op = op_print;
    }
  else
    goto usage_error;


  /****************************************************************
   * Do It
   */

  switch (op)
    {
    default:
      {
        panic ("my-revision-library: unreachable op case");
        break;
      }

    case op_delete:
      {
        arch_delete_my_default_library ();
        if (!silent)
          safe_printfmt (1, "default revision library removed\n");
        break;
      }

    case op_print:
      {
        dir = arch_my_default_library ();
        if (dir)
          safe_printfmt (1, "%s\n", dir);
        else
          {
            if (errname)
              safe_printfmt (2, "%s: no default revision library\n", errname);
            exit (1);
          }
        break;
      }

    case op_set:
      {
        arch_set_my_default_library (dir);
        break;
      }
    }

  return 0;
}



/* tag: Tom Lord Wed May 21 13:36:58 2003 (my-revision-library.c)
 */
