/* local-cache.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/mem/alloc-limits.h"
#include "hackerlab/char/str.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/vu/safe.h"
#include "tla/libfsutils/tmp-files.h"
#include "tla/libfsutils/rmrf.h"
#include "tla/libfsutils/ensure-dir.h"
#include "tla/libarch/build-revision.h"
#include "tla/libarch/pristines.h"
#include "tla/libarch/libraries.h"
#include "tla/libarch/chatter.h"
#include "tla/libarch/local-cache.h"



t_uchar *
arch_find_local_copy (t_uchar * tree_root, t_uchar * cache_dir, t_uchar * archive, t_uchar * revision)
{
  t_uchar * answer = 0;

  if (tree_root)
    answer = arch_find_pristine (0, tree_root, archive, revision, arch_any_pristine, arch_tree_and_sibling_pristine_search);

  if (!answer && cache_dir)
    answer = arch_find_pristine (0, cache_dir, archive, revision, arch_any_pristine, arch_cache_dir_pristine_search);

  if (!answer)
    answer = arch_library_find (archive, revision);

  return answer;
}


t_uchar *
arch_find_or_make_local_copy (int chatter_fd, t_uchar * tree_root, t_uchar * cache_dir, struct arch_archive * arch, t_uchar * archive, t_uchar * revision)
{
  t_uchar * answer = 0;
  t_uchar * parent_dir = 0;

  invariant (!!tree_root);
  invariant (!arch || !str_cmp (archive, arch->name));

  if (!cache_dir)
    {
      parent_dir = file_name_directory_file (0, tree_root);
      cache_dir = parent_dir;
    }

  answer = arch_find_local_copy (tree_root, cache_dir, archive, revision);

  if (!answer)
    {
      t_uchar * tree_dir = 0;
      t_uchar * tmp_path = 0;

      tree_dir = file_name_directory_file (0, tree_root);
      if (!tree_dir)
        tree_dir = str_save (0, ".");

      tmp_path = tmp_file_name (tree_dir, ",,new-pristine");

      rmrf_file (tmp_path);
      safe_mkdir (tmp_path, 0777);

      if (chatter_fd >= 0)
        safe_printfmt (1, "* build pristine tree for %s/%s\n", archive, revision);

      arch_build_revision (chatter_fd, tmp_path, arch, archive, revision, cache_dir);

      arch_install_pristine (tree_root, archive, revision, tmp_path);

      answer = arch_find_local_copy (tree_root, cache_dir, archive, revision);

      lim_free (0, tree_dir);
      lim_free (0, tmp_path);
    }

  if (parent_dir)
    lim_free (0, parent_dir);

  return answer;
}


t_uchar *
arch_find_or_make_tmp_local_copy  (int chatter_fd, t_uchar * tmp_dir, t_uchar * tree_root, t_uchar * cache_dir, struct arch_archive * arch, t_uchar * archive, t_uchar * revision)
{
  t_uchar * answer = 0;
  t_uchar * parent_dir = 0;

  if (!cache_dir)
    {
      parent_dir = file_name_directory_file (0, tree_root);
      cache_dir = parent_dir;
    }

  answer = arch_find_local_copy (tree_root, cache_dir, archive, revision);

  if (!answer)
    {
      t_uchar * tmp_stem = 0;
      t_uchar * tmp_path = 0;

      tmp_stem = str_alloc_cat_many (0, ",,", revision, "--", archive, str_end);
      tmp_path = tmp_file_name (tmp_dir, tmp_stem);

      rmrf_file (tmp_path);
      ensure_directory_exists (tmp_dir);
      safe_mkdir (tmp_path, 0777);

      if (chatter_fd >= 0)
        arch_chatter (chatter_fd, "* build reference tree for %s/%s\n", archive, revision);

      arch_build_revision (chatter_fd, tmp_path, arch, archive, revision, cache_dir);

      lim_free (0, tmp_stem);
      answer = tmp_path;
    }

  if (parent_dir)
    lim_free (0, parent_dir);

  return answer;
}




/* tag: Tom Lord Fri May 23 14:42:03 2003 (local-cache.c)
 */
