/* conversions.c -- table of termcap/terminfo equivalencies
   Copyright (C) 1991, 1995 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "tput.h"

/* `conversions' is binary-searched to retrieve the termcap equivalent
   of the given terminfo capability, along with information about what
   type of capability it is.  If you add entries, keep the table sorted
   by terminfo capability */

static struct conversion conversions[] =
{
  {"acsc", "ac", STR},
  {"am", "am", BOOL},
  {"bel", "bl", STR},
  {"blink", "mb", STR},
  {"bold", "md", STR},
  {"bw", "bw", BOOL},
  {"cbt", "bt", STR},
  {"chts", "HC", BOOL},
  {"civis", "vi", STR},
  {"clear", "cl", STR | PAD},
  {"cmdch", "CC", STR},
  {"cnorm", "ve", STR},
  {"cols", "co", NUM},
  {"cr", "cr", STR},
  {"csr", "cs", STR},
  {"cub", "LE", STR},
  {"cub1", "le", STR},
  {"cud", "DO", STR | PAD},
  {"cud1", "do", STR},
  {"cuf", "RI", STR},
  {"cuf1", "nd", STR},
  {"cup", "cm", STR},
  {"cuu", "UP", STR},
  {"cuu1", "up", STR},
  {"cvvis", "vs", STR},
  {"da", "da", BOOL},
  {"db", "db", BOOL},
  {"dch", "DC", STR},
  {"dch1", "dc", STR},
  {"dim", "mh", STR},
  {"dl", "DL", STR | PAD},
  {"dl1", "dl", STR},
  {"dsl", "ds", STR},
  {"ech", "ec", STR},
  {"ed", "cd", STR | PAD},
  {"el", "ce", STR},
  {"el1", "cb", STR},
  {"enacs", "eA", STR},
  {"eo", "eo", BOOL},
  {"eslok", "es", BOOL},
  {"ff", "ff", STR},
  {"flash", "vb", STR},
  {"fsl", "fs", STR},
  {"gn", "gn", BOOL},
  {"hc", "hc", BOOL},
  {"hd", "hd", STR},
  {"home", "ho", STR},
  {"hpa", "ch", STR},
  {"hs", "hs", BOOL},
  {"ht", "ta", STR},
  {"hts", "st", STR},
  {"hu", "hu", STR},
  {"hz", "hz", BOOL},
  {"ich", "IC", STR},
  {"ich1", "ic", STR},
  {"if", "if", STR},
  {"il", "AL", STR | PAD},
  {"il1", "al", STR},
  {"in", "in", BOOL},
  {"ind", "sf", STR},
  {"indn", "SF", STR | PAD},
  {"invis", "mk", STR},
  {"ip", "ip", STR},
  {"iprog", "iP", STR},
  {"is1", "i1", STR},
  {"is2", "is", STR},
  {"is3", "i3", STR},
  {"it", "it", NUM},
  {"kBEG", "&9", STR},
  {"kCAN", "&0", STR},
  {"kCMD", "*1", STR},
  {"kCPY", "*2", STR},
  {"kCRT", "*3", STR},
  {"kDC", "*4", STR},
  {"kDL", "*5", STR},
  {"kEND", "*7", STR},
  {"kEOL", "*8", STR},
  {"kEXT", "*9", STR},
  {"kFND", "*0", STR},
  {"kHLP", "#1", STR},
  {"kHOM", "#2", STR},
  {"kIC", "#3", STR},
  {"kLFT", "#4", STR},
  {"kMOV", "%b", STR},
  {"kMSG", "%a", STR},
  {"kNXT", "%c", STR},
  {"kOPT", "%d", STR},
  {"kPRT", "%f", STR},
  {"kPRV", "%e", STR},
  {"kRDO", "%g", STR},
  {"kRES", "%j", STR},
  {"kRIT", "%i", STR},
  {"kRPL", "%h", STR},
  {"kSAV", "!1", STR},
  {"kSPD", "!2", STR},
  {"kUND", "!3", STR},
  {"ka1", "K1", STR},
  {"ka3", "K3", STR},
  {"kb2", "K2", STR},
  {"kbeg", "@1", STR},
  {"kbs", "kb", STR},
  {"kc1", "K4", STR},
  {"kc3", "K5", STR},
  {"kcan", "@2", STR},
  {"kcbt", "kB", STR},
  {"kclo", "@3", STR},
  {"kclr", "kC", STR},
  {"kcmd", "@4", STR},
  {"kcpy", "@5", STR},
  {"kcrt", "@6", STR},
  {"kctab", "kt", STR},
  {"kcub1", "kl", STR},
  {"kcud1", "kd", STR},
  {"kcuf1", "kr", STR},
  {"kcuu1", "ku", STR},
  {"kdch1", "kD", STR},
  {"kdl1", "kL", STR},
  {"ked", "kS", STR},
  {"kel", "kE", STR},
  {"kend", "@7", STR},
  {"kent", "@8", STR},
  {"kext", "@9", STR},
  {"kf0", "k0", STR},
  {"kf1", "k1", STR},
  {"kf10", "k;", STR},
  {"kf11", "F1", STR},
  {"kf12", "F2", STR},
  {"kf13", "F3", STR},
  {"kf14", "F4", STR},
  {"kf15", "F5", STR},
  {"kf16", "F6", STR},
  {"kf17", "F7", STR},
  {"kf18", "F8", STR},
  {"kf19", "F9", STR},
  {"kf2", "k2", STR},
  {"kf20", "FA", STR},
  {"kf21", "FB", STR},
  {"kf22", "FC", STR},
  {"kf23", "FD", STR},
  {"kf24", "FE", STR},
  {"kf25", "FF", STR},
  {"kf26", "FG", STR},
  {"kf27", "FH", STR},
  {"kf28", "FI", STR},
  {"kf29", "FJ", STR},
  {"kf3", "k3", STR},
  {"kf30", "FK", STR},
  {"kf31", "FL", STR},
  {"kf32", "FM", STR},
  {"kf33", "FN", STR},
  {"kf34", "FO", STR},
  {"kf35", "FP", STR},
  {"kf36", "FQ", STR},
  {"kf37", "FR", STR},
  {"kf38", "FS", STR},
  {"kf39", "FT", STR},
  {"kf4", "k4", STR},
  {"kf40", "FU", STR},
  {"kf41", "FV", STR},
  {"kf42", "FW", STR},
  {"kf43", "FX", STR},
  {"kf44", "FY", STR},
  {"kf45", "FZ", STR},
  {"kf46", "Fa", STR},
  {"kf47", "Fb", STR},
  {"kf48", "Fc", STR},
  {"kf49", "Fd", STR},
  {"kf5", "k5", STR},
  {"kf50", "Fe", STR},
  {"kf51", "Ff", STR},
  {"kf52", "Fg", STR},
  {"kf53", "Fh", STR},
  {"kf54", "Fi", STR},
  {"kf55", "Fj", STR},
  {"kf56", "Fk", STR},
  {"kf57", "Fl", STR},
  {"kf58", "Fm", STR},
  {"kf59", "Fn", STR},
  {"kf6", "k6", STR},
  {"kf60", "Fo", STR},
  {"kf61", "Fp", STR},
  {"kf62", "Fq", STR},
  {"kf63", "Fr", STR},
  {"kf7", "k7", STR},
  {"kf8", "k8", STR},
  {"kf9", "k9", STR},
  {"kfnd", "@0", STR},
  {"khlp", "%1", STR},
  {"khome", "kh", STR},
  {"khts", "kT", STR},
  {"kich1", "kI", STR},
  {"kil1", "kA", STR},
  {"kind", "kF", STR},
  {"kll", "kH", STR},
  {"km", "km", BOOL},
  {"kmov", "%4", STR},
  {"kmrk", "%2", STR},
  {"kmsg", "%3", STR},
  {"knp", "kN", STR},
  {"knxt", "%5", STR},
  {"kopn", "%6", STR},
  {"kopt", "%7", STR},
  {"kpp", "kP", STR},
  {"kprt", "%9", STR},
  {"kprv", "%8", STR},
  {"krdo", "%0", STR},
  {"kref", "&1", STR},
  {"kres", "&5", STR},
  {"krfr", "&2", STR},
  {"kri", "kR", STR},
  {"krmir", "kM", STR},
  {"krpl", "&3", STR},
  {"krst", "&4", STR},
  {"ksav", "&6", STR},
  {"kslt", "*6", STR},
  {"kspd", "&7", STR},
  {"ktbc", "ka", STR},
  {"kund", "&8", STR},
  {"lf0", "l0", STR},
  {"lf1", "l1", STR},
  {"lf10", "la", STR},
  {"lf2", "l2", STR},
  {"lf3", "l3", STR},
  {"lf4", "l4", STR},
  {"lf5", "l5", STR},
  {"lf6", "l6", STR},
  {"lf7", "l7", STR},
  {"lf8", "l8", STR},
  {"lf9", "l9", STR},
  {"lh", "lh", NUM},
  {"lines", "li", NUM},
  {"ll", "ll", STR},
  {"lm", "lm", NUM},
  {"lw", "lw", NUM},
  {"mc0", "ps", STR},
  {"mc4", "pf", STR},
  {"mc5", "po", STR},
  {"mc5i", "5i", BOOL},
  {"mc5p", "pO", STR},
  {"mgc", "MC", STR},
  {"mir", "mi", BOOL},
  {"mrcup", "CM", STR},
  {"msgr", "ms", BOOL},
  {"nel", "nw", STR},
  {"nlab", "Nl", NUM},
  {"npc", "NP", BOOL},
  {"nrrmc", "NR", BOOL},
  {"nxon", "nx", BOOL},
  {"os", "os", BOOL},
  {"pad", "pc", STR},
  {"pb", "pb", NUM},
  {"pfkey", "pk", STR},
  {"pfloc", "pl", STR},
  {"pfx", "px", STR},
  {"pln", "pn", STR},
  {"prot", "mp", STR},
  {"rc", "rc", STR},
  {"rep", "rp", STR},
  {"rev", "mr", STR},
  {"rf", "rf", STR},
  {"rfi", "RF", STR},
  {"ri", "sr", STR},
  {"rin", "SR", STR | PAD},
  {"rmacs", "ae", STR},
  {"rmam", "RA", STR},
  {"rmcup", "te", STR},
  {"rmdc", "ed", STR},
  {"rmir", "ei", STR},
  {"rmkx", "ke", STR},
  {"rmln", "LF", STR},
  {"rmm", "mo", STR},
  {"rmp", "rP", STR},
  {"rmso", "se", STR},
  {"rmul", "ue", STR},
  {"rmxon", "RX", STR},
  {"rs1", "r1", STR},
  {"rs2", "r2", STR},
  {"rs3", "r3", STR},
  {"sc", "sc", STR},
  {"sgr", "sa", STR},
  {"sgr0", "me", STR},
  {"smacs", "as", STR},
  {"smam", "SA", STR},
  {"smcup", "ti", STR},
  {"smdc", "dm", STR},
  {"smgl", "ML", STR},
  {"smgr", "MR", STR},
  {"smir", "im", STR},
  {"smkx", "ks", STR},
  {"smln", "LO", STR},
  {"smm", "mm", STR},
  {"smso", "so", STR},
  {"smul", "us", STR},
  {"smxon", "SX", STR},
  {"tbc", "ct", STR},
  {"tsl", "ts", STR},
  {"uc", "uc", STR},
  {"ul", "ul", BOOL},
  {"vpa", "cv", STR},
  {"vt", "vt", NUM},
  {"wind", "wi", STR},
  {"wsl", "ws", NUM},
  {"xenl", "xn", BOOL},
  {"xhp", "xs", BOOL},
  {"xmc", "sg", NUM},
  {"xoffc", "XF", STR},
  {"xon", "xo", BOOL},
  {"xonc", "XN", STR},
  {"xsb", "xb", BOOL},
  {"xt", "xt", BOOL}
};

#define CAPABILITIES (sizeof (conversions) / sizeof (conversions[0]))

char *bsearch ();

/* Terminfo capability name comparison function for bsearch. */

static int
info_compare (conv1, conv2)
     struct conversion *conv1, *conv2;
{
  return strcmp (conv1->info, conv2->info);
}

/* Return a pointer to the conversion table entry for terminfo
   capability INFONAME.
   Return NULL if INFONAME is not a valid terminfo capability. */

struct conversion *
find_info (infoname, termcap_only)
     char *infoname;
     int termcap_only;
{
  struct conversion conv, *found;

  conv.info = infoname;
  found = 0;
  if (termcap_only == 0)
    found = (struct conversion *)
      bsearch ((char *) &conv, (char *) conversions, CAPABILITIES,
	       sizeof (struct conversion), info_compare);
  if (found)
    return found;
  /* We have to do linear search for termcap symbols.  Might be worth
     building perfect hash table for both */
  for (found = conversions; found < &conversions[CAPABILITIES]; found++)
    if (strcmp (infoname, found->cap) == 0)
      return found;
  return 0;
}
