/* tspsol.c (stand-alone symmetric TSP solver) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "glpk.h"
#include "glptsp.h"

/*----------------------------------------------------------------------
-- This program is a stand-alone solver intended for solving symmetric
-- Traveling Salesman Problem (TSP) using branch-and-cut method.
--
-- Should note that this program is only an illustrative example. It is
-- *not* a state-of-the-art code. Therefore only TSP instances of small
-- size (not more than 100 cities) can be solved using this code.
--
-- To run this program the following command line should be used:
--
--    tspsol tsp-file
--
-- where tsp-file is the name of an input text file, which contains TSP
-- instance data.
--
-- Detailed description of the input format recognized by this program
-- is given in the report: Gerhard Reinelt, "TSPLIB 95". This report as
-- well as TSPLIB, a library of sample TSP instances (and other related
-- problems), are freely available for research purposes at the address
-- <http://www.iwr.uni-heidelberg.de/groups/comopt/software/TSPLIB95/>.
-- See also the file 'sample.tsp' (included in the GLPK distribution in
-- the subdirectory 'sample').
--
-- The initial (root) subproblem includes variables from the initial
-- tour found by the greedy heuristic and degree constraints. However,
-- all variables are included in the branch-and-cut workspace, so they
-- are not generated during the search. For each subproblem additional
-- subtour elimination constraints are generated.
--
-- Every time when a basic solution of some subproblem is a valid tour,
-- whose length is better than the best known one, this program writes
-- the corresponding LP relaxation to the file 'MPS' using MPS format,
-- and its optimal solution to the file 'SOL' using plain text format.
----------------------------------------------------------------------*/

#define mps_file "MPS"
/* name of the output file, to which LP relaxation is sent */

#define out_file "SOL"
/* name of the output file, to which optimal solution is sent */

typedef struct PROB PROB;
typedef struct EDGE EDGE;
typedef struct DEGR DEGR;
typedef struct SUBT SUBT;

struct PROB
{     /* main data block for symmetric TSP */
      TSP *tsp;
      /* TSP problem instance */
      int nn;
      /* number of nodes */
      int ne;
      /* number of edges; ne = (nn * (nn - 1)) / 2 */
      DMP *edge_pool;
      /* memory pool for EDGE */
      DMP *degr_pool;
      /* memory pool for DEGR */
      DMP *subt_pool;
      /* memory pool for SUBT */
};

#define EDGE_DISC 'E' /* EDGE discriminant */
#define DEGR_DISC 'D' /* DEGR discriminant */
#define SUBT_DISC 'S' /* SUBT discriminant */

struct EDGE
{     /* edge (decision variable) */
      BCSVAR *var;
      /* pointer to the corresponding variable in the workspace */
      int beg, end;
      /* node numbers, 1 <= beg < end <= nn */
};

struct DEGR
{     /* degree constraint */
      BCSCON *con;
      /* pointer to the corresponding constraint in the workspace */
      int node;
      /* number of the corresponding node */
};

struct SUBT
{     /* subtour elimination constraint */
      BCSCON *con;
      /* pointer to the corresponding constraint in the workspace */
};

/*----------------------------------------------------------------------
-- greedy_tour - find salesman's tour using greedy heuristic.
--
-- This routine finds a salesman's tour using greedy heuristic. Vertex
-- numbers are stored in locations tour[1], ..., tour[nn] in the order,
-- in which they have to be visited, i.e. the tour built by the routine
-- is: tour[1] -> tour[2] -> ... -> tour[nn] -> tour[1], where nn is
-- number of nodes in the graph. The length of tour is returned by the
-- routine on exit. */

double greedy_tour(PROB *prob, int tour[])
{     int nn = prob->nn, i, j, k, d, dmin, *flag;
      double sum;
      /* flag[i] means the i-th node has been visited */
      flag = ucalloc(1+nn, sizeof(int));
      for (i = 1; i <= nn; i++) flag[i] = 0;
      tour[1] = 1, flag[1] = 1;
      for (i = 2; i <= nn; i++)
      {  k = 0, dmin = INT_MAX;
         for (j = 2; j <= nn; j++)
         {  if (!flag[j])
            {  d = tsp_distance(prob->tsp, tour[i-1], j);
               if (dmin > d) k = j, dmin = d;
            }
         }
         insist(k != 0);
         tour[i] = k, flag[k] = 1;
      }
      ufree(flag);
      /* compute the length of the tour */
      sum = 0.0;
      for (i = 1; i <= nn; i++)
      {  d = tsp_distance(prob->tsp, tour[i], tour[i < nn ? i+1 : 1]);
         sum += (double)d;
      }
      return sum;
}

/*----------------------------------------------------------------------
-- initialize - create decision variables and degree constraints.
--
-- This routine is called from the application procedure in order to
-- create decision variables (by one for each edge of the graph) and
-- degree constraints (by one for each node of the graph).
--
-- Each decision variable x[i,j] is a binary variable corresponding to
-- the edge [i,j], 1 <= i < j <= nn, where nn is number of nodes in the
-- graph. The variable x[i,j] shows whether the edge [i,j] is included
-- in the tour or not.
--
-- Although it is not necessary, the routine finds an initial tour
-- using greedy heuristic and marks variables, which correspond to the
-- edges of the initial tour, as static in order to include them in the
-- root subproblem. Other variables are marked as dynamic, so they will
-- be included in subproblems if it will be required by reduced costs.
--
-- Note that some other set of variables, which are initially included
-- in the root subproblem, could be used. In particular, all variables
-- could be marked as static and included in the root subproblem.
--
-- Each degree constraint has the form:
--
--    sum x[i,j] = 2, i = k or j = k,
--
-- for k = 1, ..., nn, where nn is number of nodes in the graph. The
-- k-th degree constraint requires that exactly two edges, which are
-- incident to the k-th node, should be included in the tour. */

void initialize(PROB *prob, BCS *bcs)
{     EDGE *edge;
      DEGR *degr;
      BCSVAR *var;
      BCSCON **con, *ccc[1+2];
      int nn = prob->nn, i, j, t, init, *tour, *next;
      double val[1+2];
      char name[31+1];
      /* create degree constraints (by one for each node of the graph)
         and add them to the workspace; constraint coefficients will be
         created a bit later */
      con = ucalloc(1+nn, sizeof(BCSCON *));
      for (i = 1; i <= nn; i++)
      {  sprintf(name, "V%d", i);
         con[i] = bcs_add_con(bcs, name, 0, LPX_FX, 2.0, 0.0, 0, NULL,
            NULL);
         /* create application extension for i-th degree constraint */
         degr = dmp_get_atom(prob->degr_pool);
         degr->con = con[i];
         degr->node = i;
         bcs_set_con_appl(bcs, con[i], DEGR_DISC, degr);
      }
      /* find an initial tour using greedy heuristic */
      tour = ucalloc(1+nn, sizeof(int));
      print("Initial tour length: %g", greedy_tour(prob, tour));
      /* determine the next node in the initial tour for each node of
         the graph */
      next = ucalloc(1+nn, sizeof(int));
      for (t = 1; t <= nn; t++) next[t] = 0;
      for (t = 1; t <= nn; t++)
      {  i = tour[t];
         insist(next[i] == 0);
         next[i] = tour[t < nn ? t+1 : 1];
      }
      /* create decision variables (by one for each edge of the graph)
         and add them to the workspace */
      for (i = 1; i <= nn; i++)
      {  for (j = i+1; j <= nn; j++)
         {  /* determine whether the edge [i,j] belongs to the initial
               tour or not */
            init = (next[i] == j || next[j] == i);
            /* asterisk after the variable name means the corresponding
               edge belongs to the initial tour */
            sprintf(name, "%d-%d%s", i, j, init ? "*" : "");
            /* each variable has exactly two non-zero coefficients in
               the corresponding degree constraints */
            ccc[1] = con[i], val[1] = 1.0;
            ccc[2] = con[j], val[2] = 1.0;
            /* add the variable x[i,j] to the workspace */
            var = bcs_add_var(bcs, name,
               BCS_INTEGER | (init ? BCS_STATIC : BCS_DYNAMIC),
               LPX_DB, 0.0, 1.0, (double)tsp_distance(prob->tsp, i, j),
               2, ccc, val);
            /* create application extension for the variable x[i,j] */
            edge = dmp_get_atom(prob->edge_pool);
            edge->var = var;
            edge->beg = i;
            edge->end = j;
            bcs_set_var_appl(bcs, var, EDGE_DISC, edge);
         }
      }
      /* free working arrays */
      ufree(con);
      ufree(tour);
      ufree(next);
      /* return to the application procedure */
      return;
}

/*----------------------------------------------------------------------
-- max_flow - find max flow using simplex method.
--
-- This routine finds max flow in a given undirected network by means
-- of the simplex method.
--
-- The undirected capacitated network is specified by the parameters
-- nn, ne, beg, end, and cap. The parameter nn is number of vertices
-- (nodes), nn > 0, and the parameter ne is number of edges, ne >= 0.
-- k-th edge is specified by the triple (beg[k], end[k], cap[k]) for
-- k = 1, ..., ne, where beg[k] and end[k] are numbers of the first and
-- the second nodes of the k-th edge (it should be beg[k] < end[k]),
-- cap[k] > 0 is a capacity of the k-th edge. Loops and multiple edges
-- are not allowed.
--
-- The parameter s is the number of a source node, and the parameter t
-- is the number of a sink node.
--
-- On exit the routine computes elementary flows along edges and stores
-- their values to locations x[1], ..., x[ne]. Positive value of x[k]
-- means that the elementary flow goes from the node beg[k] to the node
-- end[k], and negative value means that the flow goes in the opposite
-- direction. A value returned by the routine is the total maximum flow
-- carried through the network. */

double max_flow(int nn, int ne, int beg[], int end[], double cap[],
      int s, int t, double x[])
{     LPX *lp;
      int i, k, nz, *rn, *cn;
      double flow, *aa;
      /* check input data for correctness */
      insist(nn > 0);
      insist(ne >= 0);
      insist(1 <= s && s <= nn);
      insist(1 <= t && t <= nn);
      insist(s != t);
      for (k = 1; k <= ne; k++)
      {  insist(1 <= beg[k] && beg[k] < end[k] && end[k] <= nn);
         insist(cap[k] > 0.0);
      }
      /* create LP problem instance */
      lp = lpx_create_prob();
      /* create LP rows; i-th row is the condition of conservation of
         the flow in the i-th node, i = 1, ..., nn */
      lpx_add_rows(lp, nn);
      for (i = 1; i <= nn; i++)
         lpx_set_row_bnds(lp, i, LPX_FX, 0.0, 0.0);
      /* create LP columns; k-th column is the elementary flow carried
         along the k-th edge, k = 1, ..., ne; the last column with the
         number ne+1 is the total flow through the network, which goes
         along a dummy edge from the sink to the source */
      lpx_add_cols(lp, ne + 1);
      for (k = 1; k <= ne; k++)
      {  if (cap[k] <= 1e-8)
            lpx_set_col_bnds(lp, k, LPX_FX, 0.0, 0.0);
         else
            lpx_set_col_bnds(lp, k, LPX_DB, - cap[k], + cap[k]);
      }
      lpx_set_col_bnds(lp, ne + 1, LPX_FR, 0.0, 0.0);
      /* build the constraint matrix; structurally this matrix is an
         incidence matrix of the network, so each its column (including
         the last column for the dummy edge) has exactly two non-zero
         entries */
      rn = ucalloc(1 + 2 * (ne + 1), sizeof(int));
      cn = ucalloc(1 + 2 * (ne + 1), sizeof(int));
      aa = ucalloc(1 + 2 * (ne + 1), sizeof(double));
      nz = 0;
      for (k = 1; k <= ne; k++)
      {  /* x[k] > 0 means the elementary flow through the k-th edge
            goes from the node beg[k] into the node end[k] */
         nz++, rn[nz] = beg[k], cn[nz] = k, aa[nz] = -1.0;
         nz++, rn[nz] = end[k], cn[nz] = k, aa[nz] = +1.0;
      }
      /* the total flow through the network goes from the sink to the
         source along the dummy edge */
      nz++, rn[nz] = t, cn[nz] = ne + 1, aa[nz] = -1.0;
      nz++, rn[nz] = s, cn[nz] = ne + 1, aa[nz] = +1.0;
      /* check number of non-zero entries */
      insist(nz == 2 * (ne + 1));
      /* load the constraint matrix into the LP problem object */
      lpx_load_mat3(lp, nz, rn, cn, aa);
      ufree(rn);
      ufree(cn);
      ufree(aa);
      /* the objective function is the total flow through the network,
         which should be maximized */
      lpx_set_obj_dir(lp, LPX_MAX);
      lpx_set_col_coef(lp, ne + 1, 1.0);
      /* solve the LP problem */
      lpx_set_int_parm(lp, LPX_K_MSGLEV, 1);
      lpx_adv_basis(lp);
      insist(lpx_simplex(lp) == LPX_E_OK);
      insist(lpx_get_status(lp) == LPX_OPT);
      /* obtain the maximum flow through the network and the elementary
         flows through edges that correspond to the optimal solution */
      flow = lpx_get_obj_val(lp);
      for (k = 1; k <= ne; k++)
         lpx_get_col_info(lp, k, NULL, &x[k], NULL);
      /* delete LP problem instance */
      lpx_delete_prob(lp);
      /* return to the calling program */
      return flow;
}

/*----------------------------------------------------------------------
-- min_st_cut - find min (s,t)-cut for known max flow.
--
-- This routine finds min (s,t)-cut, which corresponds to a known max
-- flow in a given undirected network.
--
-- This routine should be called after the routine max_flow (see above)
-- with the same parameters, where the array x contains elementary flows
-- through edges of the network, which correspond to maximal flow found
-- by the routine max_flow.
--
-- The routine min_st_cut splits the set of nodes V of the network into
-- two non-empty subsets V(s) and V(t) = V \ V(s), where the source node
-- s belongs to V(s), the sink node t belongs to V(t), and all edges,
-- one node of which belongs to V(s) and other one belongs to V(t), are
-- saturated (i.e. x[k] = +cap[k] or x[k] = -cap[k]).
--
-- On exit the routine stores flags of the nodes v[i], i = 1, ..., nn,
-- to the locations cut[i], where cut[i] = 1 means v[i] belongs to V(s)
-- and cut[i] = 0 means v[i] belongs to V(t) = V \ V(s). The routine
-- also returns a value of the corresponding minimal (s,t)-cut, which is
-- the sum of capacities of all edges between V(s) and V(t). (Due to the
-- theorem of Ford and Fulkerson the value of the minimal cut should be
-- the same as the value of maximal flow.)
--
-- In order to determine the set V(s) the routine just finds all nodes,
-- which can be reached from the source node s via non-saturated edges.
-- The set V(t) is determined as the complement V \ V(s). */

double min_st_cut(int nn, int ne, int beg[], int end[], double cap[],
      int s, int t, double x[], int cut[])
{     int i, j, k, p, q, *head1, *next1, *head2, *next2, *list;
      double temp;
      /* head1[i] points to the first edge with beg[k] = i
         next1[k] points to the next edge with the same beg[k]
         head2[i] points to the first edge with end[k] = i
         next2[k] points to the next edge with the same end[k] */
      head1 = ucalloc(1+nn, sizeof(int));
      head2 = ucalloc(1+nn, sizeof(int));
      next1 = ucalloc(1+ne, sizeof(int));
      next2 = ucalloc(1+ne, sizeof(int));
      for (i = 1; i <= nn; i++) head1[i] = head2[i] = 0;
      for (k = 1; k <= ne; k++)
      {  i = beg[k], next1[k] = head1[i], head1[i] = k;
         j = end[k], next2[k] = head2[j], head2[j] = k;
      }
      /* on constructing the set V(s) list[1], ..., list[p-1] contain
         nodes, which can be reached from the source node and have been
         visited, and list[p], ..., list[q] contain nodes, which can be
         reached from the source node but havn't been visited yet */
      list = ucalloc(1+nn, sizeof(int));
      for (i = 1; i <= nn; i++) cut[i] = 0;
      p = q = 1, list[1] = s, cut[s] = 1;
      while (p <= q)
      {  /* pick the next node, which is reachable from the source node
            and has not visited yet, and visit it */
         i = list[p++];
         /* walk through edges with beg[k] = i */
         for (k = head1[i]; k != 0; k = next1[k])
         {  j = end[k];
            insist(beg[k] == i);
            /* from v[i] we can reach v[j], if the elementary flow from
               v[i] to v[j] is non-saturated */
            if (cut[j] == 0 && x[k] <= + (cap[k] - 1e-7))
               list[++q] = j, cut[j] = 1;
         }
         /* walk through edges with end[k] = i */
         for (k = head2[i]; k != 0; k = next2[k])
         {  j = beg[k];
            insist(end[k] == i);
            /* from v[i] we can reach v[j], if the elementary flow from
               v[i] to v[j] is non-saturated */
            if (cut[j] == 0 && x[k] >= - (cap[k] - 1e-7))
               list[++q] = j, cut[j] = 1;
         }
      }
      /* the sink can't belong to V(s) */
      insist(cut[t] == 0);
      /* free working arrays */
      ufree(head1);
      ufree(head2);
      ufree(next1);
      ufree(next2);
      ufree(list);
      /* compute value of the corresponding minimal (s,t)-cut */
      temp = 0.0;
      for (k = 1; k <= ne; k++)
      {  i = beg[k], j = end[k];
         if (cut[i] && !cut[j] || !cut[i] && cut[j]) temp += cap[k];
      }
      /* return to the calling program */
      return temp;
}

/*----------------------------------------------------------------------
-- stoer_wagner - find minimal cut using Stoer-Wagner algorithm.
--
-- This routine finds min cut in a given undirected network by means of
-- Stoer-Wagner algorithm.
--
-- The undirected capacitated network is specified by the parameters
-- nn, ne, beg, end, and cap. The parameter nn is number of vertices
-- (nodes), nn > 0, and the parameter ne is number of edges, ne >= 0.
-- k-th edge is specified by the triple (beg[k], end[k], cap[k]) for
-- k = 1, ..., ne, where beg[k] and end[k] are numbers of the first and
-- the second nodes of the k-th edge (it should be beg[k] < end[k]),
-- cap[k] > 0 is a capacity of the k-th edge. Loops and multiple edges
-- are not allowed.
--
-- Let V be the set of nodes of the network and let W be an arbitrary
-- non-empty proper subset of V. A cut associated with the subset W is
-- a subset of all the edges, one node of which belongs to W and other
-- node belongs to V \ W. The capacity of a cut (W, V \ W) is the sum
-- of the capacities of all edges, which belong to the cut. Minimal cut
-- is a cut, whose capacity is minimal.
--
-- On exit the routine stores flags of the nodes v[i], i = 1, ..., nn,
-- to the locations cut[i], where cut[i] = 1 means v[i] belongs to W
-- and cut[i] = 0 means v[i] belongs to V \ W, where W corresponds to
-- a minimal cut. The routine returns the capacity of the minimal cut.
--
-- The basic idea of Stoer-Wagner algorithm is the following. Let G be
-- a capacitated network and G(s,t) be a network, in which the nodes s
-- and t are merged into one new node, loops are deleted, but multuple
-- edges are retained. It is obvious that a minimum cut in G is the
-- minimum of two quantities: the minimum cut in G(s,t) and a minimum
-- cut that separates s and t. This allows to find a minimum cut in the
-- original network solving at most nn max flow problems.
--
-- M.Stoer, F.Wagner. A Simple Min Cut Algorithm. Algorithms, ESA'94
-- LNCS 855 (1994), pp. 141-47.
--
-- J.Cheriyan, R.Ravi. Approximation Algorithms for Network Problems.
-- Univ. of Waterloo (1998), p. 147. */

double stoer_wagner(int nn, int ne, int beg[], int end[], double cap[],
      int cut[])
{     int i, j, k, *head1, *next1, *head2, *next2, I, J, K, S, T,
         DEG, NV, NE, *BEG, *END, *HEAD, *NEXT, *NUMB, *ADJ, *CUT;
      double min_cut, flow, temp, *X, *CAP, *SUM;
      /* check input data for correctness */
      insist(nn > 0);
      insist(ne >= 0);
      for (k = 1; k <= ne; k++)
      {  insist(1 <= beg[k] && beg[k] < end[k] && end[k] <= nn);
         insist(cap[k] > 0.0);
      }
      /* head1[i] points to the first edge with beg[k] = i
         next1[k] points to the next edge with the same beg[k]
         head2[i] points to the first edge with end[k] = i
         next2[k] points to the next edge with the same end[k] */
      head1 = ucalloc(1+nn, sizeof(int));
      head2 = ucalloc(1+nn, sizeof(int));
      next1 = ucalloc(1+ne, sizeof(int));
      next2 = ucalloc(1+ne, sizeof(int));
      for (i = 1; i <= nn; i++) head1[i] = head2[i] = 0;
      for (k = 1; k <= ne; k++)
      {  i = beg[k], next1[k] = head1[i], head1[i] = k;
         j = end[k], next2[k] = head2[j], head2[j] = k;
      }
      /* an auxiliary network used in the algorithm is resulted from
         the original network by merging some nodes into one supernode;
         all variables and arrays related to this auxiliary network are
         denoted in caps */
      /* HEAD[I] points to the first node of the original network that
         belongs to the I-th supernode
         NEXT[i] points to the next node of the original network that
         belongs to the same supernode as the i-th node
         NUMB[i] is a supernode, which the i-th node belongs to */
      /* initially the auxiliary network is equivalent to the original
         network, i.e. each supernode consists of one node */
      NV = nn;
      HEAD = ucalloc(1+nn, sizeof(int));
      NEXT = ucalloc(1+nn, sizeof(int));
      NUMB = ucalloc(1+nn, sizeof(int));
      for (i = 1; i <= nn; i++) HEAD[i] = i, NEXT[i] = 0, NUMB[i] = i;
      /* number of edges in the auxiliary network is never greater than
         in the original one */
      BEG = ucalloc(1+ne, sizeof(int));
      END = ucalloc(1+ne, sizeof(int));
      CAP = ucalloc(1+ne, sizeof(double));
      X = ucalloc(1+ne, sizeof(double));
      /* allocate some auxiliary arrays */
      ADJ = ucalloc(1+nn, sizeof(int));
      SUM = ucalloc(1+nn, sizeof(double));
      CUT = ucalloc(1+nn, sizeof(int));
      /* currently no min cut is known so far */
      min_cut = DBL_MAX;
      /* main loop starts here */
      while (NV > 1)
      {  /* build the set of edges of the auxiliary network */
         NE = 0;
         /* multiple edges are not allowed in the max flow algorithm,
            so we can replace each multiple edge, which is the result
            of merging nodes into supernodes, by a single edge, whose
            capacity is the sum of capacities of particular edges;
            these summary capacities will be stored in the array SUM */
         for (I = 1; I <= NV; I++) SUM[I] = 0.0;
         for (I = 1; I <= NV; I++)
         {  /* DEG is number of single edges, which connects the I-th
               supernode and some J-th supernode, where I < J */
            DEG = 0;
            /* walk through nodes that belong to the I-th supernode */
            for (i = HEAD[I]; i != 0; i = NEXT[i])
            {  /* the i-th node belongs to the I-th supernode */
               /* walk through edges with beg[k] = i */
               for (k = head1[i]; k != 0; k = next1[k])
               {  j = end[k];
                  /* j-th node belongs to the J-th supernode */
                  J = NUMB[j];
                  /* ignore loops and edges with I > J */
                  if (I >= J) continue;
                  /* add an edge, which connects the I-th and the J-th
                     supernodes (if not added yet) */
                  if (SUM[J] == 0.0) ADJ[++DEG] = J;
                  /* sum up the capacity of the original edge */
                  insist(cap[k] > 0.0);
                  SUM[J] += cap[k];
               }
               /* walk through edges with end[k] = i */
               for (k = head2[i]; k != 0; k = next2[k])
               {  j = beg[k];
                  /* j-th node belongs to the J-th supernode */
                  J = NUMB[j];
                  /* ignore loops and edges with I > J */
                  if (I >= J) continue;
                  /* add an edge, which connects the I-th and the J-th
                     supernodes (if not added yet) */
                  if (SUM[J] == 0.0) ADJ[++DEG] = J;
                  /* sum up the capacity of the original edge */
                  insist(cap[k] > 0.0);
                  SUM[J] += cap[k];
               }
            }
            /* add single edges, which connect the I-th supernode to
               other supernodes, to the auxiliary network, and restore
               the array SUM for subsequent use */
            for (K = 1; K <= DEG; K++)
            {  NE++;
               insist(NE <= ne);
               J = ADJ[K];
               BEG[NE] = I, END[NE] = J, CAP[NE] = SUM[J];
               SUM[J] = 0.0;
            }
         }
         /* choose two arbitrary supernodes of the auxiliary network,
            one of which is the source and other is the sink */
         S = 1, T = NV;
         /* determine max flow from S to T */
         flow = max_flow(NV, NE, BEG, END, CAP, S, T, X);
         /* if the min cut, which separates the supernodes S and T, is
            less than the currently known, remember it */
         if (min_cut > flow)
         {  min_cut = flow;
            /* determine the min cut in the auxiliary network */
            temp = min_st_cut(NV, NE, BEG, END, CAP, S, T, X, CUT);
            /* check if Ford and Fulkerson are always right :+) */
            insist(fabs(flow - temp) <= 1e-6 * (1.0 + fabs(flow)));
            /* determine the min cut in the original network */
            for (i = 1; i <= nn; i++) cut[i] = CUT[NUMB[i]];
            /* if the min cut is close to zero (that obviously means
               the network has unconnected components), the search can
               be prematurely terminated */
            if (min_cut <= 1e-8) break;
         }
         /* now merge all nodes of the original network, which belong
            to the supernodes S and T, into one new supernode; this is
            attained by carrying all nodes from T to S (for the sake of
            convenience T should be the last supernode) */
         insist(T == NV);
         /* assign new references to nodes from T */
         for (i = HEAD[T]; i != 0; i = NEXT[i]) NUMB[i] = S;
         /* find the last entry in the node list of S */
         i = HEAD[S];
         insist(i != 0);
         while (NEXT[i] != 0) i = NEXT[i];
         /* and attach to it the node list of T */
         NEXT[i] = HEAD[T];
         /* decrease number of nodes of the auxiliary network */
         NV--;
      }
      /* free working arrays */
      ufree(HEAD);
      ufree(NEXT);
      ufree(NUMB);
      ufree(BEG);
      ufree(END);
      ufree(CAP);
      ufree(X);
      ufree(ADJ);
      ufree(SUM);
      ufree(CUT);
      ufree(head1);
      ufree(head2);
      ufree(next1);
      ufree(next2);
      /* return to the calling program */
      return min_cut;
}

/*----------------------------------------------------------------------
-- gen_subt_con - generate subtour elimination constraint.
--
-- This routine is called from the application procedure in order to
-- generate a violated subtour elimination constraint.
--
-- Constraints of this class has the form:
--
--    sum x[i,j] >= 2, i in W, j in V \ W,
--
-- for all W, where W is a proper nonempty subset of V, V is the set of
-- nodes of the given graph.
--
-- In order to find a violated constraint of this class this routine
-- finds a min cut in a capacitated network, which has the same sets of
-- nodes and edges as the original graph, and where capacities of edges
-- are values of variables x[i,j] in a basic solution of the current
-- subproblem. */

void gen_subt_con(PROB *prob, BCS *bcs)
{     EDGE *edge;
      SUBT *subt;
      BCSVAR *var, **vvv;
      BCSCON *con;
      int nn, ne, *beg, *end, *cut, i, j, k, len, disc, nz;
      double vx, min_cut, *cap, *val;
      /* the network has the same set of nodes as the original graph */
      nn = prob->nn;
      /* if some variable x[i,j] is zero in the basic solution of the
         current subproblem, the capacity of the corresponding edge in
         the network is zero, therefore such edge may not be included
         in the network; note that if a variable is not presented in
         the current subproblem, it is zero by the definition, so only
         variables presented in the subproblem should be considered */
      /* count number of edges with non-zero capacity */
      ne = 0;
      for (k = bcs_get_num_cols(bcs); k >= 1; k--)
      {  var = bcs_get_jth_var(bcs, k);
         bcs_get_var_info(bcs, var, NULL, &vx, NULL);
         if (vx >= 1e-6) ne++;
      }
      /* create the capacitated network */
      beg = ucalloc(1+ne, sizeof(int));
      end = ucalloc(1+ne, sizeof(int));
      cap = ucalloc(1+ne, sizeof(double));
      nz = 0;
      for (k = bcs_get_num_cols(bcs); k >= 1; k--)
      {  var = bcs_get_jth_var(bcs, k);
         bcs_get_var_info(bcs, var, NULL, &vx, NULL);
         if (vx >= 1e-6)
         {  bcs_get_var_appl(bcs, var, &disc, (void **)&edge);
            insist(disc == EDGE_DISC);
            nz++;
            beg[nz] = edge->beg;
            end[nz] = edge->end;
            cap[nz] = vx;
         }
      }
      insist(nz == ne);
      /* find minimal cut in the capacitated network */
      cut = ucalloc(1+nn, sizeof(int));
      min_cut = stoer_wagner(nn, ne, beg, end, cap, cut);
      /* if the capacity of min cut is less than 2, the corresponding
         subtour elimination constraint is violated */
      if (min_cut <= 2.0 - 1e-5)
      {  /* count number of edges, which get involved into the cut,
            i.e. one node of which belongs to W and other node belongs
            to V \ W */
         len = 0;
         for (var = bcs_get_next_var(bcs, NULL); var != NULL;
              var = bcs_get_next_var(bcs, var))
         {  bcs_get_var_appl(bcs, var, &disc, (void **)&edge);
            insist(disc == EDGE_DISC);
            i = edge->beg, j = edge->end;
            if (cut[i] && !cut[j] || !cut[i] && cut[j]) len++;
         }
         /* create violated subtour elimination constraint */
         vvv = ucalloc(1+len, sizeof(BCSVAR *));
         val = ucalloc(1+len, sizeof(double));
         nz = 0;
         for (var = bcs_get_next_var(bcs, NULL); var != NULL;
              var = bcs_get_next_var(bcs, var))
         {  bcs_get_var_appl(bcs, var, &disc, (void **)&edge);
            insist(disc == EDGE_DISC);
            i = edge->beg, j = edge->end;
            if (cut[i] && !cut[j] || !cut[i] && cut[j])
            {  nz++;
               vvv[nz] = var;
               val[nz] = 1.0;
            }
         }
         insist(nz == len);
         /* and add it to the workspace */
         con = bcs_add_con(bcs, NULL, 0, LPX_LO, 2.0, 0.0, len, vvv,
            val);
         ufree(vvv);
         ufree(val);
         /* create application extension for this constraint */
         subt = dmp_get_atom(prob->subt_pool);
         subt->con = con;
         bcs_set_con_appl(bcs, con, SUBT_DISC, subt);
      }
      /* free working arrays */
      ufree(beg);
      ufree(end);
      ufree(cap);
      ufree(cut);
      /* return to the application procedure */
      return;
}

/*----------------------------------------------------------------------
-- appl_proc - event-driven application procedure.
--
-- This routine is called from the branch-and-cut driver and performs
-- application specific actions. */

void appl_proc(void *info, BCS *bcs)
{     PROB *prob = info;
      /* determine the current event code */
      switch (bcs_get_event(bcs))
      {  case BCS_V_INIT:
            /* initializing */
            /* create decision variables and degree constraints */
            initialize(prob, bcs);
            break;
         case BCS_V_GENCON:
            /* constraints generation required */
            /* generate one subtour elimination constraint */
            gen_subt_con(prob, bcs);
            break;
         case BCS_V_BINGO:
            /* better integer feasible solution found */
            /* write LP relaxation and its optimal solution */
            {  LPX *lp = bcs_get_lp_object(bcs);
               lpx_set_int_parm(lp, LPX_K_MPSORIG, 1);
               lpx_write_mps(lp, mps_file);
               lpx_print_sol(lp, out_file);
            }
            break;
         case BCS_V_DELVAR:
            /* some variable is being deleted */
            /* delete the corresponding application extension */
            {  BCSVAR *var = bcs_which_var(bcs);
               int disc; void *link; EDGE *edge;
               bcs_get_var_appl(bcs, var, &disc, &link);
               switch (disc)
               {  case EDGE_DISC:
                     edge = link;
                     dmp_free_atom(prob->edge_pool, edge);
                     break;
                  default:
                     insist(disc != disc);
               }
            }
            break;
         case BCS_V_DELCON:
            /* some constraint is being deleted */
            /* delete the corresponding application extension */
            {  BCSCON *con = bcs_which_con(bcs);
               int disc; void *link; DEGR *degr; SUBT *subt;
               bcs_get_con_appl(bcs, con, &disc, &link);
               switch (disc)
               {  case DEGR_DISC:
                     degr = link;
                     dmp_free_atom(prob->degr_pool, degr);
                     break;
                  case SUBT_DISC:
                     subt = link;
                     dmp_free_atom(prob->subt_pool, subt);
                     break;
                  default:
                     insist(disc != disc);
               }
            }
            break;
         default:
            /* ignore other events */
            break;
      }
      /* return to the branch-and-cut driver */
      return;
}

/*----------------------------------------------------------------------
-- main - main program.
--
-- This main program reads TSP problem data from the file specified in
-- the command line and tries to solve it. */

int main(int argc, char *argv[])
{     PROB *prob;
      double start;
      if (argc != 2)
      {  print("Usage: %s tsp-file", argv[0]);
         exit(EXIT_FAILURE);
      }
      /* create main data block */
      prob = umalloc(sizeof(PROB));
      prob->tsp = tsp_read_data(argv[1]);
      if (prob->tsp == NULL)
      {  print("TSP data file processing error");
         exit(EXIT_FAILURE);
      }
      if (prob->tsp->type != TSP_TSP)
      {  print("Data file is not symmetric TSP");
         exit(EXIT_FAILURE);
      }
      prob->nn = prob->tsp->dimension;
      prob->ne = (prob->nn * (prob->nn - 1)) / 2;
      print("Number of nodes: %d", prob->nn);
      print("Number of edges: %d", prob->ne);
      prob->edge_pool = dmp_create_pool(sizeof(EDGE));
      prob->degr_pool = dmp_create_pool(sizeof(DEGR));
      prob->subt_pool = dmp_create_pool(sizeof(SUBT));
      /* call the branch-and-cut driver */
      start = utime();
      bcs_driver(prob->tsp->name, LPX_MIN, prob, appl_proc);
      /* display statistics */
      print("Time used:   %.1f secs", utime() - start);
      print("Memory used: %.1fM (%d bytes)",
         (double)lib_env_ptr()->mem_tpeak / (double)1000000,
         lib_env_ptr()->mem_tpeak);
      /* delete main data block */
      tsp_free_data(prob->tsp);
      insist(prob->edge_pool->count == 0);
      insist(prob->degr_pool->count == 0);
      insist(prob->subt_pool->count == 0);
      dmp_delete_pool(prob->edge_pool);
      dmp_delete_pool(prob->degr_pool);
      dmp_delete_pool(prob->subt_pool);
      ufree(prob);
      insist(lib_env_ptr()->mem_total == 0);
      insist(lib_env_ptr()->mem_count == 0);
      /* return to the control program */
      return 0;
}

/* eof */
