/* glpapi.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPAPI_H
#define _GLPAPI_H

#include "glpavl.h"
#include "glpmat.h"
#include "glpstr.h"

typedef struct LPI LPI;
typedef struct LPIROW LPIROW;
typedef struct LPICOL LPICOL;

struct LPI
{     /* LP/MIP problem instance */
      POOL *row_pool;
      /* memory pool for allocating LPIROW objects */
      POOL *col_pool;
      /* memory pool for allocating LPICOL objects */
      POOL *elem_pool;
      /* memory pool for allocating constraint matrix elements */
      POOL *str_pool;
      /* memory pool for allocating segmented character strings */
      int check;
      /* data consistency checking flag */
      STR *name;
      /* problem name (1 to 255 chars) or NULL if problem has no name */
      int m_max;
      /* current dimension of array of row pointers; if necessary, this
         dimension is automatically increased */
      int n_max;
      /* current dimension of array of column pointers; if necessary,
         this dimension is automatically increased */
      int m;
      /* number of rows */
      int n;
      /* number of columns */
      LPIROW **row; /* LPIROW *row[1+m_max]; */
      /* array of row pointers; row[0] is not used; row[i] points to
         the i-th row (i = 1,...,m); row[m+1], ..., row[m_max] are
         reserved for new rows that can be added to the problem */
      LPICOL **col; /* LPICOL *col[1+n_max]; */
      /* array of column pointers; col[0] is not used; col[j] points to
         the j-th column (j = 1,...,n); col[n+1], ..., col[n_max] are
         reserved for new columns that can be added to the problem */
      AVLTREE *row_tree;
      /* row name search tree */
      AVLTREE *col_tree;
      /* column name search tree */
      double c0;
      /* constant term of the objective function */
      int sense;
      /* optimization sense:
         '-' - minimization
         '+' - maximization */
      int basis;
      /* basis status:
         '?' - basis is undefined
         'N' - basis is neither primal nor dual feasible
         'P' - basis is primal feasible, but dual infeasible
         'D' - basis is primal infeasible, but dual feasible
         'O' - basis is primal and dual feasible (optimal) */
      int status;
      /* solution status: */
#define GLP_UNDEF    0  /* solution is undefined */
#define GLP_OPT      1  /* solution is optimal */
#define GLP_FEAS     2  /* solution is feasible */
#define GLP_INFEAS   3  /* solution is infeasible */
#define GLP_NOFEAS   4  /* problem has no feasible solution */
#define GLP_UNBND    5  /* problem has unbounded solution */
#define GLP_INTOPT   6  /* solution is integer optimal */
#define GLP_INTSOL   7  /* solution is integer feasible */
#define GLP_DISINT   8  /* solution is integer infeasible */
#define GLP_NOINT    9  /* problem has no integer solution */
      double objval;
      /* value of the objective function */
};

struct LPIROW
{     /* row (auxiliary variable) */
      int seqn;
      /* row sequential number (1 to m) */
      STR *name;
      /* row name (1 to 255 chars) or NULL if row has no name */
      int type;
      /* row (auxiliary variable) type:
         'F' - free variable:    -inf <  x[k] <  +inf
         'L' - lower bound:      l[k] <= x[k] <  +inf
         'U' - upper bound:      -inf <  x[k] <= u[k]
         'D' - double bound:     l[k] <= x[k] <= u[k]
         'S' - fixed variable:   l[k]  = x[k]  = u[k] */
      double lb;
      /* lower bound; if row has no lower bound, lb is zero */
      double ub;
      /* upper bound; if row has no upper bound, ub is zero; in case
         of fixed auxiliary variable lb is equal to ub */
      ELEM *ptr;
      /* pointer to the linked list of constraint coefficients placed
         in this row (this linked list is unordered) */
      int tagx;
      /* row (auxiliary variable) status:
         'B' - basic variable
         'L' - non-basic variable on its lower bound
         'U' - non-basic variable on its upper bound
         'F' - non-basic free variable
         'S' - non-basic fixed variable */
      double valx;
      /* row primal activity */
      double dx;
      /* row dual activity */
};

struct LPICOL
{     /* column (structural variable) */
      int seqn;
      /* column sequential number (1 to n) */
      STR *name;
      /* column name (1 to 255 chars) or NULL if column has no name */
      int kind;
      /* column (structural variable) kind:
         'C' - continous variable
         'I' - integer variable */
      int type;
      /* column (structural variable) type:
         'F' - free variable:    -inf <  x[k] <  +inf
         'L' - lower bound:      l[k] <= x[k] <  +inf
         'U' - upper bound:      -inf <  x[k] <= u[k]
         'D' - double bound:     l[k] <= x[k] <= u[k]
         'S' - fixed variable:   l[k]  = x[k]  = u[k] */
      double lb;
      /* lower bound; if column has no lower bound, lb is zero */
      double ub;
      /* upper bound; if column has no upper bound, ub is zero; in case
         of fixed structural variable lb is equal to ub */
      ELEM *ptr;
      /* pointer to the linked list of constraint coefficients placed
         in this column (this linked list is unordered) */
      double coef;
      /* coefficient of the objective function */
      int tagx;
      /* column (structural variable) status:
         'B' - basic variable
         'L' - non-basic variable on its lower bound
         'U' - non-basic variable on its upper bound
         'F' - non-basic free variable
         'S' - non-basic fixed variable */
      double valx;
      /* column primal activity */
      double dx;
      /* column dual activity */
};

#include "glpk.h"

#ifdef GLP_OLD_API

#define glp glp_workspace

typedef struct GLPWKSP GLPWKSP;
typedef struct GLPITEM GLPITEM;
typedef struct GLPCOEF GLPCOEF;

struct GLPWKSP
{     /* GLPK API workspace */
      /* --- memory management segment --- */
      POOL *str_pool;
      /* memory pool for segmented character strings */
      POOL *item_pool;
      /* memory pool for allocating GLPITEM objects */
      POOL *coef_pool;
      /* memory pool for allocating GLPCOEF objects */
      /* --- problem data segment --- */
      AVLTREE *row_tab;
      /* row names look-up table */
      GLPITEM *first_row;
      /* pointer to the first row */
      GLPITEM *last_row;
      /* pointer to the last row */
      GLPITEM *this_row;
      /* pointer to the current row */
      AVLTREE *col_tab;
      /* column names look-up table */
      GLPITEM *first_col;
      /* pointer to the first column */
      GLPITEM *last_col;
      /* pointer to the last column */
      GLPITEM *this_col;
      /* pointer to the current column */
      GLPCOEF *this_coef;
      /* pointer to the current constraint coefficient */
      /* --- control parameters segment --- */
      double c0;
      /* the constant term of the objective function */
      STR *fn_gener; /* used by the glp_read_lpm routine */
      /* this character string specifies name of the file to which the
         glp_read_lpm routine should output generated LP/MIP problem in
         plain text format; NULL means no output */
      int mip_branch; /* used by MIP solver */
      /* this parameter specifies what branching technique should be
         used by the solver:
         GLP_FIRST - branch on the first integer variable
         GLP_LAST  - branch on the last integer variable
         GLP_DRTOM - branch using heuristic by Driebeek and Tomlin */
      int mip_btrack; /* used by MIP solver */
      /* this parameter specifies what backtracking technique should be
         used by the solver:
         GLP_FIFO  - backtrack using FIFO heuristic
         GLP_LIFO  - backtrack using LIFO heuristic
         GLP_BESTP - backtrack using the best projection heuristic */
      char mps_bnd_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of bound vector that
         should be used in the problem; if this parameter is an empty
         string, the first bound vector is used */
      char mps_obj_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of row that should be
         used as the objective function; if this parameter is an empty
         string, the first row of N type (i.e. free row) is used as the
         objective function */
      int mps_one_entry; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine keeps fields 5 and 6 empty;
         otherwise the routine uses all fields */
      int mps_pedantic; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine never omits column and vector
         names; otherwise the routine omits those names every time when
         it's possible */
      char mps_rhs_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of right-hand side
         (RHS) vector that should be used in the problem; if this
         parameter is an empty string, the first RHS vector is used */
      char mps_rng_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of range vector that
         should be used in the problem; if this parameter is an empty
         string, the first range vector is used */
      int mps_skip_empty; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine writes all columns, including
         empty ones (i.e. those which have no constraint coefficients);
         otherwise the routine write only non-empty columns */
      int mps_use_names; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine tries to generate standard
         MPS names using orifinal row and column names as templates;
         otherwise the routine generates plain names using sequential
         numbers of rows and columns */
      int obj_dir; /* used by the solver */
      /* optimization direction flag:
         GLP_MIN - objective function should be minimized
         GLP_MAX - objective function should be maximized */
      GLPITEM *obj_row; /* used by the solver */
      /* pointer to the objective function row; NULL indicates that the
         objective function is identically equal to zero */
      int option; /* used by the solver */
      /* this parameter specifies what solution should be found by the
         solver:
         GLP_INI - only initial solution
         GLP_ANY - any feasible solution
         GLP_FIN - final solution */
      STR *problem; /* informative */
      /* problem name (1 to 255 chars) */
      int round; /* used by the solver */
      /* if this flag is set, the solver replaces values of those basis
         variables which are near to zero, by exact zeros; otherwise the
         solver remains values of all basis variables "as is" */
      int scale; /* used by the solver */
      /* if this flag is set, the solver performs automatic scaling the
         problem before solving; otherwise scaling is not used */
      int spx_form; /* used by the solver */
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         GLP_EFI    - EFI
         GLP_RFI_BG - RFI + Bartels & Golub updating technique
         GLP_RFI_FT - RFI + Forrest & Tomlin updating technique */
      int spx_relax; /* used by the solver */
      /* if this flag is set, the solver uses two-pass ratio test (for
         both primal and dual simplex) proposed by P.Harris; otherwise
         the solver uses the standard "textbook" ratio test */
      int spx_steep; /* used by the solver */
      /* if this flag is set, the solver uses the steepest edge pricing
         (for both primal and dual simplex) proposed by Goldfarb & Reid;
         otherwise the solver uses the standard "textbook" pricing */
      int spx_use_dual; /* used by the solver */
      /* if this flag is set, the solver uses the dual simplex method
         for searching for feasible solution; otherwise the solver uses
         the primal simplex method */
      int status; /* reported by the solver */
      /* solution status:
         GLP_UNDEF  - solution status is undefined
         GLP_OPT    - solution is optimal
         GLP_FEAS   - solution is feasible
         GLP_INFEAS - solution is infeasible
         GLP_NOFEAS - problem has no feasible solution
         GLP_UNBND  - problem has unbounded solution
         GLP_INTOPT - solution is integer optimal
         GLP_INTSOL - solution is integer feasible
         GLP_DISINT - solution is integer infeasible
         GLP_NOINT  - problem has no integer feasible solution */
      int sum_aij; /* used by the solver */
      /* if this flag is set, the solver replaces multiplets (i.e.
         elements of the constraint matrix that have identical row and
         column indices) by their sum before solving the problem;
         otherwise the solver considers multiplets as incorrect data */
      double tol_aij; /* used by the solver */
      /* this parameter specifies the absolute tolerance which is used
         by the solver to drop small elements of the original constraint
         matrix; if this tolerance is equal to zero, the solver uses the
         original matrix without any changes */
      double tol_bnd; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for checking primal feasibility of the current
         basis solution */
      double tol_dj; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for checking dual feasibility of the current
         basis solution */
      double tol_int; /* used by MIP solver */
      /* this parameter specifies the absolute tolerance which is used
         by MIP solver to see if the solution is integer feasible */
      double tol_obj; /* used by MIP solver */
      /* this parameter specifies the relative tolerance which is used
         by MIP solver to see if current value of the objective function
         is not better than the best integer feasible solution found */
      double tol_piv; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for pivoting */
};

struct GLPITEM
{     /* row (auxiliary variable) or column (structural variable) */
      STR *name;
      /* row/column name (1 to 255 chars) */
      int kind;
      /* row/column kind:
         0 - continuous variable
         1 - integer variable (can be set only for column) */
      int type;
      /* row/column type:
         GLP_FR - free variable:  -inf <  x[k] <  +inf
         GLP_LO - lower bound:    l[k] <= x[k] <  +inf
         GLP_UP - upper bound:    -inf <  x[k] <= u[k]
         GLP_DB - double bound:   l[k] <= x[k] <= u[k]
         GLP_FX - fixed variable: l[k]  = x[k]  = u[k] */
      double lb;
      /* lower bound; if row/column has no lower bound, lb is zero */
      double ub;
      /* upper bound; if row/column has no upper bound, ub is zero;
         in case of fixed variable (GLP_FX) lb is equal to ub */
      GLPCOEF *ptr;
      /* pointer to the linked list of constraint coefficients placed
         in this row/column (this linked list is unordered) */
      int seqn;
      /* this field is used by GLPK routines for different purposes */
      int tagx;
      /* row/column status:
         GLP_BS - basic variable
         GLP_NL - non-basic variable on lower bound
         GLP_NU - non basic variable on upper bound
         GLP_NF - non-basic free variable
         GLP_NS - non-basic fixed variable */
      double valx;
      /* row/column primal activity */
      double dx;
      /* row/column dual activity */
      GLPITEM *next;
      /* pointer to the next row/column (created row/column is always
         added to the end of this linked list) */
};

struct GLPCOEF
{     /* constraint coefficient */
      GLPITEM *row;
      /* pointer to the row */
      GLPITEM *col;
      /* pointer to the column */
      double val;
      /* coefficient value */
      GLPCOEF *next1;
      /* pointer to the next coefficient in the same row */
      GLPCOEF *next2;
      /* pointer to the next coefficient in the same column */
};

extern GLPWKSP *glp;
/* pointer to the active workspace */

#endif /* GLP_OLD_API */

#endif

/* eof */
