/* glpefi/efi_ftran.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpefi.h"

/*----------------------------------------------------------------------
-- efi_ftran - perform forward transformation (FTRAN) using EFI.
--
-- *Synopsis*
--
-- #include "glpefi.h"
-- double *efi_ftran(EFI *efi, double z[], int save);
--
-- *Description*
--
-- The efi_ftran routine performs forward transformation of the vector
-- z using EFI which efi points to.
--
-- In order to perform this operation the routine solves the system
-- B*x = z, where B is the basis matrix defined by EFI, x is vector of
-- unknowns (transformed vector that should be computed), z is vector of
-- right-hand sides (given vector that should be transformed). On entry
-- the array z should contain elements of the vector z in locations
-- z[1], z[2], ..., z[m], where m is the order of the matrix B. On exit
-- this array will contain the vector x in the same locations.
--
-- The parameter save is a flag. If this flag is set, it means that the
-- vector z is a column corresponding to that non-basis variable, which
-- has been chosen to enter the basis. And then the efi_ftran routine
-- saves the vector x (i.e. transformed column), which will be used by
-- the update_efi routine to update EFI for adjacent basis matrix. It is
-- assumed that the simplex method routine should perform at least one
-- call to the efi_ftran routine with the save parameter set.
--
-- *Returns*
--
-- The efi_ftran routine returns a pointer to the array z. */

double *efi_ftran(EFI *efi, double z[], int save)
{     /* B = B0*H, therefore inv(B) = inv(H)*inv(B0) */
      solve_lu(efi->lu, 0, z, efi->work);
      h_solve(efi->eta, z);
      if (save)
      {  /* save transformed column */
         int i;
         for (i = 1; i <= efi->m; i++) efi->col[i] = z[i];
         efi->flag = 1;
      }
      return z;
}

/* eof */
