/* glpk.h */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#ifndef _GLPK_H
#define _GLPK_H

#include "glplpx.h"

#ifndef _GLPAPI_H
typedef struct { int dummy; } LPI;
#endif

#define GLP_UNDEF    0  /* solution is undefined */
#define GLP_OPT      1  /* solution is optimal */
#define GLP_FEAS     2  /* solution is feasible */
#define GLP_INFEAS   3  /* solution is infeasible */
#define GLP_NOFEAS   4  /* problem has no feasible solution */
#define GLP_UNBND    5  /* problem has unbounded solution */
#define GLP_INTOPT   6  /* solution is integer optimal */
#define GLP_INTSOL   7  /* solution is integer feasible */
#define GLP_DISINT   8  /* solution is integer infeasible */
#define GLP_NOINT    9  /* problem has no integer solution */

extern void glp_check_data(LPI *lp, int check);
/* turn on/off data consistency checking flag */

extern int glp_check_name(char *name);
/* check symbolic name for correctness */

extern void glp_clear_mat_cols(LPI *lp, int flag[]);
/* nullify columns of constraint matrix */

extern void glp_clear_mat_rows(LPI *lp, int flag[]);
/* nullify rows of constraint matrix */

extern LPI *glp_create_prob(char *name);
/* create problem instance */

extern void glp_delete_cols(LPI *lp, int flag[]);
/* delete columns from problem instance */

extern void glp_delete_prob(LPI *lp);
/* delete problem instance */

extern void glp_delete_rows(LPI *lp, int flag[]);
/* delete rows from problem instance */

extern int glp_find_col(LPI *lp, char *name);
/* find column by name */

extern int glp_find_row(LPI *lp, char *name);
/* find row by name */

extern int glp_get_bstat(LPI *lp);
/* determine basis status */

extern void glp_get_col_bnds(LPI *lp, int j, int *type, double *lb,
      double *ub);
/* determine column bounds */

extern int glp_get_col_coef(LPI *lp, int j, int rn[], double aj[]);
/* get column of constraint matrix */

extern double glp_get_col_fctr(LPI *lp, int j);
/* query column scale factor */

extern int glp_get_col_kind(LPI *lp, int j);
/* get column kind */

extern char *glp_get_col_name(LPI *lp, int j);
/* get column name */

extern void glp_get_col_soln(LPI *lp, int j, int *tagx, double *valx,
      double *dx);
/* obtain column solution information */

extern int glp_get_num_bin(LPI *lp);
/* determine number of binary columns */

extern int glp_get_num_cols(LPI *lp);
/* determine number of columns */

extern int glp_get_num_int(LPI *lp);
/* determine number of integer columns */

extern int glp_get_num_nz(LPI *lp);
/* determine number of constraint coefficients */

extern int glp_get_num_rows(LPI *lp);
/* determine number of rows */

extern double glp_get_obj_coef(LPI *lp, int j);
/* get objective function coefficient */

extern int glp_get_obj_sense(LPI *lp);
/* get objective function sense */

extern double glp_get_obj_val(LPI *lp);
/* obtain objective function value */

extern char *glp_get_prob_name(LPI *lp);
/* obtain problem name */

extern void glp_get_row_bnds(LPI *lp, int i, int *type, double *lb,
      double *ub);
/* determine row bounds */

extern int glp_get_row_coef(LPI *lp, int i, int cn[], double ai[]);
/* get row of constraint matrix */

extern double glp_get_row_fctr(LPI *lp, int i);
/* query row scale factor */

extern char *glp_get_row_name(LPI *lp, int i);
/* get row name */

extern void glp_get_row_soln(LPI *lp, int i, int *tagx, double *valx,
      double *dx);
/* obtain row solution information */

extern int glp_get_status(LPI *lp);
/* determine solution status */

extern void glp_new_aij(LPI *lp, int i, int j, double aij);
/* create new constraint coefficient */

extern void glp_new_col(LPI *lp, char *name);
/* create new column */

extern void glp_new_row(LPI *lp, char *name);
/* create new row */

extern int glp_print_soln(LPI *lp, char *fname);
/* write problem solution using printable format */

extern void glp_put_col_soln(LPI *lp, int j, int tagx, double valx,
      double dx);
/* store column solution information */

extern void glp_put_row_soln(LPI *lp, int i, int tagx, double valx,
      double dx);
/* store row solution information */

extern void glp_put_soln_info(LPI *lp, int bstat, int status,
      double objv);
/* store main solution information */

extern LPI *glp_read_lpm1(char *infile, char *outfile);
/* read linear programming model written in GLPK/L */

extern LPI *glp_read_mps1(char *fname);
/* read problem data using MPS format */

extern void glp_scale_prob(LPI *lp, int how);
/* scale problem */

extern void glp_set_col_bnds(LPI *lp, int j, int type, double lb,
      double ub);
/* set column bounds */

extern void glp_set_col_fctr(LPI *lp, int j, double fctr);
/* set column scale factor */

extern void glp_set_col_kind(LPI *lp, int j, int kind);
/* set column kind */

extern void glp_set_col_name(LPI *lp, int j, char *name);
/* assign (change) column name */

extern void glp_set_obj_coef(LPI *lp, int j, double coef);
/* set objective function coefficient */

extern void glp_set_obj_sense(LPI *lp, int sense);
/* set objective function sense */

extern void glp_set_prob_name(LPI *lp, char *name);
/* assign problem name */

extern void glp_set_row_bnds(LPI *lp, int i, int type, double lb,
      double ub);
/* set row bounds */

extern void glp_set_row_fctr(LPI *lp, int i, double fctr);
/* set row scale factor */

extern void glp_set_row_name(LPI *lp, int i, char *name);
/* assign (change) row name */

extern LPX *glp_lpi_to_lpx(LPI *lpi, int all);
/* convert LP problem instance to low-level object */

extern void glp_load_lpx_soln(LPI *lpi, LPX *lpx, int round);
/* load LP solution from low-level object */

struct wmps
{     /* parameter block of the routine glp_write_mps() */
      int prob_info;
      /* if this flag is set, the routine writes several comment cards
         that contains some information about the problem; otherwise the
         routine writes no comment cards */
      int make_obj;
      /* this parameter controls output objective function row:
         0 - never output objective function row
         1 - always output objective function row
         2 - output objective function row if and only if the problem
             has no free rows */
      int use_names;
      /* if this flag is set, the routine tries to generate standard
         MPS names using orifinal row and column names as templates;
         otherwise the routine generates plain names using sequential
         numbers of rows and columns */
      int one_entry;
      /* if this flag is set, the routine keeps fields 5 and 6 empty;
         otherwise the routine uses all fields */
      int pedantic;
      /* if this flag is set, the routine never omits column and vector
         names; otherwise the routine omits those names every time when
         it's possible */
      int skip_empty;
      /* if this flag is set, the routine writes all columns, including
         empty ones (i.e. those which have no constraint coefficients);
         otherwise the routine write only non-empty columns */
};

extern void glp_init_wmps(struct wmps *parm);
/* initialize parameter block by default values */

extern int glp_write_mps(LPI *lp, char *fname, struct wmps *parm);
/* write problem data using MPS format */

struct spx
{     /* simplex method parameter block */
      int msg_lev;
      /* level of messages output by the solver:
         0 - no output
         1 - error messages only
         2 - normal output
         3 - full output (includes informational messages) */
      int scale;
      /* scaling option:
         0 - do not scale the problem
         1 - scale the problem before solving it */
      int initb;
      /* initial basis option:
         0 - use the standard basis, where all auxiliary variables are
             basic and all structural variables are non-basic
         1 - use an advanced initial basis
         2 - use the basis specified in the problem object */
      int price;
      /* pricing option:
         0 - use standard ("textbook") pricing
         1 - use projected steepest edge pricing */
      int relax;
      /* ratio test option:
         0 - use the standard ("textbook") ratio test
         1 - use two-pass ratio test proposed by P.Harris */
      double tol_bnd;
      /* relative tolerance used to check if the current basic solution
         is primal feasible */
      double tol_dj;
      /* absolute tolerance used to check if the current basic solution
         is dual feasible */
      double tol_piv;
      /* relative tolerance used to choose eligible pivotal elements of
         the simplex table in the ratio test */
      int round;
      /* solution rounding option:
         0 - report all computed values and reduced costs "as is"
         1 - round computed values of basic variables and reduced costs
             of non-basic variables toward their bounds or toward zero,
             if possible (i.e. if it is allowed by the tolerances) */
      int it_lim;
      /* simplex iterations limit; if this value is positive, it is
         decreased by one each time when one simplex iteration has been
         performed, and reaching zero value signals the solver to stop
         the search; negative value means no iterations limit */
      double tm_lim;
      /* searching time limit, in seconds; if this value is positive,
         it is decreased each time when one simplex iteration has been
         performed by the amount of time spent for the iteration, and
         reaching zero value signals the solver to stop the search;
         negative value means no time limit */
      int out_frq;
      /* output frequency, in iterations; this parameter specifies how
         frequently the solver sends information about the solution to
         the standard output */
};

extern void glp_init_spx(struct spx *parm);
/* initialize parameter block by default values */

extern int glp_simplex(LPI *lp, struct spx *parm);
/* solve LP problem using simplex method */

extern int glp_interior(LPI *lpi);
/* solve LP problem using interior point method */

struct bnb
{     /* branch-and-bound method parameter block */
      int branch;
      /* this parameter specifies what branching heuristic should be
         used by the solver: */
#define BB_FIRST  0  /* branch on the first variable */
#define BB_LAST   1  /* branch on the last variable */
#define BB_DRTOM  2  /* branch using heuristic by Driebeck and Tomlin */
      int btrack;
      /* this parameter specifies what backtracking heuristic should be
         used by the solver: */
#define BB_FIFO   0  /* backtrack using FIFO heuristic */
#define BB_LIFO   1  /* backtrack using LIFO heuristic */
#define BB_BESTP  2  /* backtrack using the best projection heuristic */
      double tol_int;
      /* absolute tolerance which is used to see if the solution is
         integer feasible */
      double tol_obj;
      /* relative tolerance which is used to check if current value of
         the objective function is not better than for the best integer
         feasible solution found */
      int scale;
      /* if this flag is set, the solver performs automatic scaling the
         problem; otherwise the scaling is not used */
      int steep;
      /* if this flag is set, the solver uses the steepest edge pricing
         proposed by Goldfarb & Reid; otherwise the standard "textbook"
         pricing is used */
      int relax;
      /* if this flag is set, the solver uses two-pass ratio test
         proposed by P.Harris; otherwise the standard "textbook" ratio
         test is used */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int iter_max;
      /* maximal allowed number of iterations (0 means no limit) */
      int round;
      /* if this flag is set, the solver replaces computed values of
         basic continuous variables which are close to zero by exact
         zeros; otherwise all computed values remain "as is" */
};

extern void glp_init_bnb(struct bnb *parm);
/* initialize parameter block by default values */

extern int glp_integer(LPI *lpi, struct bnb *parm);
/* solve MIP problem using branch-and-bound method */

#endif

/* eof */
