/* glpk.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPK_H
#define _GLPK_H

#ifndef _GLPAPI_H
typedef struct { int dummy; } LPI;
#endif

#define GLP_UNDEF    0  /* solution is undefined */
#define GLP_OPT      1  /* solution is optimal */
#define GLP_FEAS     2  /* solution is feasible */
#define GLP_INFEAS   3  /* solution is infeasible */
#define GLP_NOFEAS   4  /* problem has no feasible solution */
#define GLP_UNBND    5  /* problem has unbounded solution */
#define GLP_INTOPT   6  /* solution is integer optimal */
#define GLP_INTSOL   7  /* solution is integer feasible */
#define GLP_DISINT   8  /* solution is integer infeasible */
#define GLP_NOINT    9  /* problem has no integer solution */

extern void glp_check_data(LPI *lp, int check);
/* turn on/off data consistency checking flag */

extern int glp_check_name(char *name);
/* check symbolic name for correctness */

extern void glp_clear_mat_cols(LPI *lp, int flag[]);
/* nullify columns of constraint matrix */

extern void glp_clear_mat_rows(LPI *lp, int flag[]);
/* nullify rows of constraint matrix */

extern LPI *glp_create_prob(char *name);
/* create problem instance */

extern void glp_delete_cols(LPI *lp, int flag[]);
/* delete columns from problem instance */

extern void glp_delete_prob(LPI *lp);
/* delete problem instance */

extern void glp_delete_rows(LPI *lp, int flag[]);
/* delete rows from problem instance */

extern int glp_find_col(LPI *lp, char *name);
/* find column by name */

extern int glp_find_row(LPI *lp, char *name);
/* find row by name */

extern int glp_get_bstat(LPI *lp);
/* determine basis status */

extern void glp_get_col_bnds(LPI *lp, int j, int *type, double *lb,
      double *ub);
/* determine column bounds */

extern int glp_get_col_coef(LPI *lp, int j, int rn[], double aj[]);
/* get column of constraint matrix */

extern double glp_get_col_fctr(LPI *lp, int j);
/* query column scale factor */

extern int glp_get_col_kind(LPI *lp, int j);
/* get column kind */

extern char *glp_get_col_name(LPI *lp, int j);
/* get column name */

extern void glp_get_col_soln(LPI *lp, int j, int *tagx, double *valx,
      double *dx);
/* obtain column solution information */

extern int glp_get_num_bin(LPI *lp);
/* determine number of binary columns */

extern int glp_get_num_cols(LPI *lp);
/* determine number of columns */

extern int glp_get_num_int(LPI *lp);
/* determine number of integer columns */

extern int glp_get_num_nz(LPI *lp);
/* determine number of constraint coefficients */

extern int glp_get_num_rows(LPI *lp);
/* determine number of rows */

extern double glp_get_obj_coef(LPI *lp, int j);
/* get objective function coefficient */

extern int glp_get_obj_sense(LPI *lp);
/* get objective function sense */

extern double glp_get_obj_val(LPI *lp);
/* obtain objective function value */

extern char *glp_get_prob_name(LPI *lp);
/* obtain problem name */

extern void glp_get_row_bnds(LPI *lp, int i, int *type, double *lb,
      double *ub);
/* determine row bounds */

extern int glp_get_row_coef(LPI *lp, int i, int cn[], double ai[]);
/* get row of constraint matrix */

extern double glp_get_row_fctr(LPI *lp, int i);
/* query row scale factor */

extern char *glp_get_row_name(LPI *lp, int i);
/* get row name */

extern void glp_get_row_soln(LPI *lp, int i, int *tagx, double *valx,
      double *dx);
/* obtain row solution information */

extern int glp_get_status(LPI *lp);
/* determine solution status */

extern void glp_new_aij(LPI *lp, int i, int j, double aij);
/* create new constraint coefficient */

extern void glp_new_col(LPI *lp, char *name);
/* create new column */

extern void glp_new_row(LPI *lp, char *name);
/* create new row */

extern int glp_print_soln(LPI *lp, char *fname);
/* write problem solution using printable format */

extern void glp_put_col_soln(LPI *lp, int j, int tagx, double valx,
      double dx);
/* store column solution information */

extern void glp_put_row_soln(LPI *lp, int i, int tagx, double valx,
      double dx);
/* store row solution information */

extern void glp_put_soln_info(LPI *lp, int bstat, int status,
      double objv);
/* store main solution information */

extern LPI *glp_read_lpm1(char *infile, char *outfile);
/* read linear programming model written in GLPK/L */

extern LPI *glp_read_mps1(char *fname);
/* read problem data using MPS format */

extern void glp_scale_prob(LPI *lp, int how);
/* scale problem */

extern void glp_set_col_bnds(LPI *lp, int j, int type, double lb,
      double ub);
/* set column bounds */

extern void glp_set_col_fctr(LPI *lp, int j, double fctr);
/* set column scale factor */

extern void glp_set_col_kind(LPI *lp, int j, int kind);
/* set column kind */

extern void glp_set_col_name(LPI *lp, int j, char *name);
/* assign (change) column name */

extern void glp_set_obj_coef(LPI *lp, int j, double coef);
/* set objective function coefficient */

extern void glp_set_obj_sense(LPI *lp, int sense);
/* set objective function sense */

extern void glp_set_prob_name(LPI *lp, char *name);
/* assign problem name */

extern void glp_set_row_bnds(LPI *lp, int i, int type, double lb,
      double ub);
/* set row bounds */

extern void glp_set_row_fctr(LPI *lp, int i, double fctr);
/* set row scale factor */

extern void glp_set_row_name(LPI *lp, int i, char *name);
/* assign (change) row name */

struct wmps
{     /* parameter block of the routine glp_write_mps() */
      int prob_info;
      /* if this flag is set, the routine writes several comment cards
         that contains some information about the problem; otherwise the
         routine writes no comment cards */
      int make_obj;
      /* this parameter controls output objective function row:
         0 - never output objective function row
         1 - always output objective function row
         2 - output objective function row if and only if the problem
             has no free rows */
      int use_names;
      /* if this flag is set, the routine tries to generate standard
         MPS names using orifinal row and column names as templates;
         otherwise the routine generates plain names using sequential
         numbers of rows and columns */
      int one_entry;
      /* if this flag is set, the routine keeps fields 5 and 6 empty;
         otherwise the routine uses all fields */
      int pedantic;
      /* if this flag is set, the routine never omits column and vector
         names; otherwise the routine omits those names every time when
         it's possible */
      int skip_empty;
      /* if this flag is set, the routine writes all columns, including
         empty ones (i.e. those which have no constraint coefficients);
         otherwise the routine write only non-empty columns */
};

extern void glp_init_wmps(struct wmps *parm);
/* initialize parameter block by default values */

extern int glp_write_mps(LPI *lp, char *fname, struct wmps *parm);
/* write problem data using MPS format */

struct rsm1
{     /* revised simplex method parameter block */
      int what;
      /* this parameter specifies what basis solution should be found
         by the solver:
         0 - initial solution
         1 - primal feasible solution
         2 - optimal solution */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         0 - PFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique
         3 - AFI
         4 - UFI */
      int scale;
      /* if this flag is set, the solver performs automatic scaling the
         problem; otherwise the scaling is not used */
      int dual;
      /* if this flag is set, the solver uses the dual simplex method
         in order to find the primal feasible basis solution; otherwise
         the solver uses the primal simplex method */
      int steep;
      /* if this flag is set, the solver uses the steepest edge pricing
         (for both primal and dual simplex) proposed by Goldfarb & Reid;
         otherwise the solver uses the standard "textbook" pricing */
      int relax;
      /* if this flag is set, the solver uses two-pass ratio test (for
         both primal and dual simplex) proposed by P.Harris; otherwise
         the solver uses the standard "textbook" ratio test */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int iter_max;
      /* maximal allowed number of iterations (0 means no limit) */
      int round;
      /* if this flag is set, the solver replaces computed values of
         basic variables and reduced costs of non-basic variables which
         are close to zero by exact zeros; otherwise the solver remains
         all computed values and reduced costs "as is" */
};

extern void glp_init_rsm1(struct rsm1 *parm);
/* initialize parameter block by default values */

extern int glp_call_rsm1(LPI *lpi, struct rsm1 *parm);
/* solve LP problem using revised simplex method */

extern int glp_call_ipm1(LPI *lpi);
/* solve LP problem using interior point method */

struct bbm1
{     /* branch-and-bound method parameter block */
      int what;
      /* this parameter specifies what basis solution should be found
         by the solver:
         0 - initial relaxed solution
         1 - feasible integer solution
         2 - optimal integer solution */
      int branch;
      /* this parameter specifies what branching heuristic should be
         used by the solver: */
#define BB_FIRST  0  /* branch on the first variable */
#define BB_LAST   1  /* branch on the last variable */
#define BB_DRTOM  2  /* branch using heuristic by Driebeck and Tomlin */
      int btrack;
      /* this parameter specifies what backtracking heuristic should be
         used by the solver: */
#define BB_FIFO   0  /* backtrack using FIFO heuristic */
#define BB_LIFO   1  /* backtrack using LIFO heuristic */
#define BB_BESTP  2  /* backtrack using the best projection heuristic */
      double tol_int;
      /* absolute tolerance which is used to see if the solution is
         integer feasible */
      double tol_obj;
      /* relative tolerance which is used to check if current value of
         the objective function is not better than for the best integer
         feasible solution found */
      int meth;
      /* this parameter specifies what version of the simplex method
         should be used by the solver to find an optimal basis of LP
         relaxation:
         0 - rsm1_driver()
         1 - glp_simplex2() */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         0 - PFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique
         3 - AFI
         4 - UFI */
      int scale;
      /* if this flag is set, the solver performs automatic scaling the
         problem; otherwise the scaling is not used */
      int dual;
      /* if this flag is set, the solver uses the dual simplex method
         in order to find the primal feasible basis solution; otherwise
         the solver uses the primal simplex method */
      int steep;
      /* if this flag is set, the solver uses the steepest edge pricing
         proposed by Goldfarb & Reid; otherwise the standard "textbook"
         pricing is used */
      int relax;
      /* if this flag is set, the solver uses two-pass ratio test
         proposed by P.Harris; otherwise the standard "textbook" ratio
         test is used */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int iter_max;
      /* maximal allowed number of iterations (0 means no limit) */
      int round;
      /* if this flag is set, the solver replaces computed values of
         basic continuous variables which are close to zero by exact
         zeros; otherwise all computed values are remained "as is" */
};

extern void glp_init_bbm1(struct bbm1 *parm);
/* initialize parameter block by default values */

extern int glp_call_bbm1(LPI *lpi, struct bbm1 *parm);
/* solve MIP problem using branch-and-bound method */

struct spx1
{     /* comprehensive simplex driver parameter block */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         0 - PFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique
         3 - AFI
         4 - UFI */
      int scale;
      /* this parameter controls the scaling option:
         0 - do not scale the problem
         1 - scale the problem using scale factors specified in the
             problem object */
      int initb;
      /* this parameter controls the initial basis option:
         0 - use the standard basis, where all auxiliary variables are
             basic and all structural variables are non-basic
         1 - use the basis specified in the problem object */
      int feas;
      /* this parameter specifies by what method the solver should
         search for primal feasible solution if the initial basis is
         not primal feasible:
         0 - use the primal simplex method and implicit artificial
             variables technique
         1 - use the dual simplex method and artificial objective
             function technique */
      int dual;
      /* this parameter specifies what the solver should undertake if
         the initial basis happens to be dual feasible:
         0 - ignore dual feasibility, i.e. search for primal feasible
             solution (phase I) and then search for optimal solution
             (phase II) using the primal simplex method
         1 - search for optimal solution using the dual simplex method,
             i.e. skip the phase I */
      int steep;
      /* this parameter controls the pricing option (for both primal
         and dual simplex methods):
         0 - use the standard ("textbook") pricing
         1 - use the steepest edge pricing proposed by Goldfarb and
             Reid */
      int relax;
      /* this parameter controls the ratio test option (for both primal
         and dual simplex methods):
         0 - use the standard ("textbook") ratio test
         1 - use two-pass ratio test proposed by P.Harris */
      int round;
      /* this parameter controls the final solution rounding option:
         0 - keep all computed primal and dual values "as is"
         1 - replace computed primal and dual values, which are close
             to zero, by exact zeros (this is performed once after the
             final solution has been obtained) */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int max_iter;
      /* maximal allowable number of simplex iterations; if this
         parameter is zero, the solver performs no simplex iterations,
         but reports solution that corresponds to the initial basis */
      double max_time;
      /* maximal allowable time of computations, in seconds; if this
         parameter is zero, the solver performs no simplex iterations,
         but reports solution that corresponds to the initial basis */
      double obj_min, obj_max;
      /* allowable range of the objective function; the solver stops
         searching as soon as on the phase II the current value of the
         objective function becomes less than obj_min and continue
         decreasing or greater than obj_max and continue increasing;
         to avoid computing the objective function on each simplex
         iteration, the range (-DBL_MAX, +DBL_MAX) should be used */
      int quiet;
      /* this parameter controls displaying informative messages:
         0 - display all informative messages
         1 - don't display any informative messages */
      double delay;
      /* initial delay interval, in seconds; the solver doesn't display
         information about current solution the first delay seconds */
      double freq;
      /* output frequency, in seconds; the solver displays information
         about current solution approx. once per freq seconds */
      int iter_used;
      /* on entry this parameter is ignored; on exit this parameter is
         number of actually performed simplex iterations */
      double time_used;
      /* on entry this parameter is ignored; on exit this parameter is
         elapsed time (in seconds) actually spent by the solver */
};

extern void glp_init_spx1(struct spx1 *parm);
/* initialize parameter block by default values */

extern int glp_simplex1(LPI *lp, struct spx1 *_parm);
/* comprehensive driver to the simplex method */

struct piv1
{     /* pivoting simplex driver parameter block */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the routine:
         0 - PFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique
         3 - AFI
         4 - UFI */
      int scale;
      /* this parameter controls the scaling option:
         0 - do not scale the constraint matrix
         1 - scale the constraint matrix using scale factors specified
             in the problem object */
      double tol;
      /* Markowitz tolerance */
};

extern void glp_init_piv1(struct piv1 *parm);
/* initialize parameter block by default values */

extern int glp_pivot_in(LPI *lp, int flag[], struct piv1 *parm);
/* push auxiliary variables into the basis */

extern int glp_pivot_out(LPI *lp, int flag[], struct piv1 *parm);
/* pull structural variables from the basis */

struct spx2
{     /* primal simplex method parameter block */
      int scale;
      /* this parameter controls the scaling option:
         0 - do not scale the problem
         1 - scale the problem
         2 - scale the problem using scale factors specified in the
             problem object */
      int initb;
      /* this parameter controls the initial basis option:
         0 - use the standard basis, where all auxiliary variables are
             basic and all structural variables are non-basic
         1 - use an advanced initial basis
         2 - use the basis specified in the problem object */
      int round;
      /* this parameter controls the final solution rounding option:
         0 - keep all computed primal and dual values "as is"
         1 - replace computed primal and dual values, which are close
             to zero, by exact zeros (this is performed once after the
             final solution has been obtained) */
      double tol_bnd;
      /* tolerance used to check if the solution is primal feasible */
      double tol_dj;
      /* tolerance used to check if the solution is dual feasible */
      double tol_piv;
      /* tolerance used to choose pivot element of the simplex table */
};

extern void glp_init_spx2(struct spx2 *parm);
/* initialize parameter block by default values */

extern int glp_simplex2(LPI *lp, struct spx2 *parm);
/* solve LP problem using primal simplex method */

#endif

/* eof */
