/* glprsm/pivot_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- pivot_col - choose non-basic variable (primal).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int pivot_col(RSM *rsm, double c[], double cbar[], double gvec[],
--    double tol);
--
-- *Description*
--
-- The pivot_col routine chooses non-basic variable (xN)q (i.e. pivot
-- column of the simplex table) which should enter the basis on the next
-- iteration of the primal simplex method. Note that the routine assumes
-- that the objective function should be *minimized*.
--
-- The array c should contain the expanded vector of coefficients of
-- the objective function in locations c[1], ..., c[m+n]. This array is
-- not changed on exit.
--
-- The array cbar should contain reduced costs of non-basic variables in
-- locations cbar[1], ..., cbar[n]. This array is not changed on exit.
--
-- The array gvec should contain elements of the vector gamma in
-- locations gvec[1], ..., gvec[n]. This array is not changed on exit.
-- It is allowed to specify NULL instead the array gvec; in this case
-- the routine assumes that all elements of the vector gamma are equal
-- to one.
--
-- The parameter tol is a relative tolerance (see below).
--
-- The pivot_col routine considers only those non-basic variables which
-- changing in feasible direction can improve (decrease) the objective
-- function, i.e. for which
--
--    (xN)j in xF and |dj| >  eps, or
--    (xN)j in xL and  dj  < -eps, or
--    (xN)j in xU and  dj  > +eps
--
-- where
--
--    eps = tol * max(1, |(cN)j|)
--
-- (the absolute tolerance eps reflects that the reduced cost on
-- non-basic variable (xN)j is the difference dj = (cN)j - pi'*Nj).
--
-- The routine chooses the non-basic variable (xN)q which has the
-- largest (in absolute value) scaled reduced cost
--
--    d'q = dq / sqrt(gamma_q)
--
-- Thus, if the vector gamma is not used, the choice made by the routine
-- corresponds to the textbook pricing. Otherwise the choice corresponds
-- to the steepest edge pricing.
--
-- *Returns*
--
-- If the choice has been made, the pivot_col routine returns q which
-- is a number of the chosen non-basic variable (xN)q, 1 <= q <= n.
-- Otherwise, if the current basis solution is dual feasible and the
-- choice is impossible, the routine returns zero. */

int pivot_col(RSM *rsm, double c[], double cbar[], double gvec[],
      double tol)
{     int n = rsm->n, j, k, q, ret;
      double big, temp;
      q = 0, big = 0.0;
      for (j = 1; j <= n; j++)
      {  /* skip column if xN[j] can't change */
         if (rsm->tagn[j] == 'S') continue;
         /* skip column if xN[j] doesn't affect on the obj. func. */
         if (cbar[j] == 0.0) continue;
         k = rsm->indn[j]; /* x[k] = xN[j] */
         ret = check_rr(cbar[j] + c[k], c[k], tol);
         switch (rsm->tagn[j])
         {  case 'F':
               /* xN[j] can change in any direction */
               if (-1 <= ret && ret <= +1) continue;
               break;
            case 'L':
               /* xN[j] can increase */
               if (ret >= -1) continue;
               break;
            case 'U':
               /* xN[j] can decrease */
               if (ret <= +1) continue;
               break;
            default:
               insist(rsm->tagn[j] != rsm->tagn[j]);
         }
         /* xN[j] can improve (decrease) the objective function */
#if 0
         if (gvec == NULL)
            temp = fabs(cbar[j]);
         else
            temp = fabs(cbar[j]) / sqrt(gvec[j]);
#else /* 3.0.4 */
         temp = (cbar[j] * cbar[j]) / (gvec == NULL ? 1.0 : gvec[j]);
#endif
         if (big < temp) q = j, big = temp;
      }
      return q;
}

/* eof */
