/* glpapi/glp_print_soln.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- glp_print_soln - write problem solution using printable format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_print_soln(LPI *lp, char *fname);
--
-- *Description*
--
-- The routine glp_print_soln writes the problem solution associated
-- with the problem instance lp to the text file fname.
--
-- Information reported by the routine corresponds to the final simplex
-- table found by the solver. This information is intended mainly for
-- the visual analysis and has the meaning only if the problem has been
-- successfully solved.
--
-- *Returns*
--
-- The routine glp_print_soln returns one of the following codes:
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_print_soln(LPI *lp, char *fname)
{     FILE *fp;
      int what;
      print("glp_print_soln: writing problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_print_soln: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char *name;
         name = glp_get_prob_name(lp);
         if (name == NULL) name = "";
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         nr = glp_get_num_rows(lp);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc, nc_int, nc_bin;
         nc = glp_get_num_cols(lp);
         nc_int = glp_get_num_int(lp);
         nc_bin = glp_get_num_bin(lp);
         fprintf(fp, "%-12s%d", "Columns:", nc);
         if (nc_int)
            fprintf(fp, " (%d integer, %d binary)", nc_int, nc_bin);
         fprintf(fp, "\n");
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         nz = glp_get_num_nz(lp);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* problem status */
      {  int status;
         status = glp_get_status(lp);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == GLP_UNDEF  ? "UNDEFINED" :
            status == GLP_OPT    ? "OPTIMAL" :
            status == GLP_FEAS   ? "FEASIBLE" :
            status == GLP_INFEAS ? "INFEASIBLE (INTERMEDIATE)" :
            status == GLP_NOFEAS ? "INFEASIBLE (FINAL)" :
            status == GLP_UNBND  ? "UNBOUNDED" :
            status == GLP_INTOPT ? "INTEGER OPTIMAL" :
            status == GLP_INTSOL ? "INTEGER FEASIBLE" :
            status == GLP_DISINT ? "INTEGER INFEASIBLE (INTERMEDIATE)" :
            status == GLP_NOINT  ? "INTEGER INFEASIBLE (FINAL)" : "???")
            ;
      }
      /* objective function */
      {  int dir;
         double val;
         val = glp_get_obj_val(lp);
         dir = glp_get_obj_sense(lp);
         fprintf(fp, "%-12s%.6g %s\n", "Objective:", val,
            dir == '-'  ? "(MINimization)" :
            dir == '+'  ? "(MAXimization)" : "(???)");
      }
      /* main sheet */
      for (what = 1; what <= 2; what++)
      {  int mn, ij;
         fprintf(fp, "\n");
         fprintf(fp, "  No. %-12s   St   Activity     Lower bound   Upp"
            "er bound    Marginal\n",
            what == 1 ? "  Row name" : "Column name");
         fprintf(fp, "----- ------------   -- ------------- -----------"
            "-- ------------- -------------\n");
         mn = (what == 1 ? glp_get_num_rows(lp) : glp_get_num_cols(lp));
         for (ij = 1; ij <= mn; ij++)
         {  char *name;
            int kind, type, tagx;
            double lb, ub, valx, dx;
            if (what == 1)
            {  name = glp_get_row_name(lp, ij);
               if (name == NULL) name = "";
               kind = 0;
               glp_get_row_bnds(lp, ij, &type, &lb, &ub);
               glp_get_row_soln(lp, ij, &tagx, &valx, &dx);
            }
            else
            {  name = glp_get_col_name(lp, ij);
               if (name == NULL) name = "";
               kind = (glp_get_col_kind(lp, ij) == 'C' ? 0 : 1);
               glp_get_col_bnds(lp, ij, &type, &lb, &ub);
               glp_get_col_soln(lp, ij, &tagx, &valx, &dx);
            }
            /* row/column sequential number */
            fprintf(fp, "%5d ", ij);
            /* row column/name */
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%19s", name, "");
            /* row/column kind */
            fprintf(fp, "%s ", kind ? "*" : " ");
            /* row/column status */
            if (tagx == 'B')
            {  if (type == 'L' && valx < lb ||
                   type == 'D' && valx < lb ||
                   type == 'S' && valx < lb)
                  fprintf(fp, "B- ");
               else if (type == 'U' && valx > ub ||
                        type == 'D' && valx > ub ||
                        type == 'S' && valx > ub)
                  fprintf(fp, "B+ ");
               else if (kind && valx != floor(valx + 0.5))
                  fprintf(fp, "B* ");
               else
                  fprintf(fp, "B  ");
            }
            else if (tagx == 'L')
               fprintf(fp, "NL ");
            else if (tagx == 'U')
               fprintf(fp, "NU ");
            else if (tagx == 'F')
               fprintf(fp, "NF ");
            else if (tagx == 'S')
               fprintf(fp, "NS ");
            else
               fprintf(fp, "??");
            /* row/column primal activity */
            fprintf(fp, "%13.6g ", valx);
            /* row/column lower bound */
            if (type == 'L' || type == 'D' || type == 'S')
               fprintf(fp, "%13.6g ", lb);
            else
               fprintf(fp, "%13s ", "");
            /* row/column upper bound */
            if (type == 'U' || type == 'D')
               fprintf(fp, "%13.6g ", ub);
            else if (type == 'S')
               fprintf(fp, "%13s ", "=");
            else
               fprintf(fp, "%13s ", "");
            /* row/column dual activity */
            if (tagx != 'B')
            {  if (dx == 0.0)
                  fprintf(fp, "%13s", "< eps");
               else
                  fprintf(fp, "%13.6g", dx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  error("glp_print_soln: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
