/* glprsm/check_rsm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- check_rsm - check common block for correctness.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void check_rsm(RSM *rsm);
--
-- *Description*
--
-- The check_rsm routine checks the simplex method common block, which
-- rsm points to, for correctness. In case of error the routine displays
-- an appropriate message and terminates the program.
--
-- Note that the check_rsm routine doesn't check the constraint matrix,
-- because the corresponding operation is extremely inefficient. It can
-- be checked additionally by the check_mat routine. */

void check_rsm(RSM *rsm)
{     int m = rsm->m, n = rsm->n, k;
      if (m < 1)
         fault("check_rsm: invalid number of rows");
      if (n < 1)
         fault("check_rsm: invalid number of columns");
      for (k = 1; k <= m+n; k++)
      {  switch (rsm->type[k])
         {  case 'F':
               if (!(rsm->lb[k] == 0.0 && rsm->ub[k] == 0.0))
err1:             fault("check_rsm: invalid bounds of row/column");
               break;
            case 'L':
               if (rsm->ub[k] != 0.0) goto err1;
               break;
            case 'U':
               if (rsm->lb[k] != 0.0) goto err1;
               break;
            case 'D':
               break;
            case 'S':
               if (rsm->lb[k] != rsm->ub[k]) goto err1;
               break;
            default:
               fault("check_rsm: invalid type of row/column");
         }
      }
      if (!(rsm->A->m == m && rsm->A->n == m+n))
         fault("check_rsm: invalid dimension of constraint matrix");
      for (k = 1; k <= m+n; k++)
      {  if (rsm->posx[k] > 0)
         {  int i = +rsm->posx[k]; /* xB[i] = x[k] */
            if (!(1 <= i && i <= m && rsm->indb[i] == k))
               fault("check_rsm: invalid position of basic row/column");
         }
         else
         {  int j = -rsm->posx[k]; /* xN[j] = x[k] */
            if (!(1 <= j && j <= n && rsm->indn[j] == k))
               fault("check_rsm: invalid position of non-basic row/colu"
                  "mn");
            switch (rsm->type[k])
            {  case 'F':
                  if (rsm->tagn[j] != 'F')
err2:                fault("check_rsm: invalid tag of non-basic row/col"
                        "umn");
                  break;
               case 'L':
                  if (rsm->tagn[j] != 'L') goto err2;
                  break;
               case 'U':
                  if (rsm->tagn[j] != 'U') goto err2;
                  break;
               case 'D':
                  if (!(rsm->tagn[j] == 'L' || rsm->tagn[j] == 'U'))
                     goto err2;
                  break;
               case 'S':
                  if (rsm->tagn[j] != 'S') goto err2;
                  break;
               default:
                  insist(rsm->type[k] != rsm->type[k]);
            }
         }
      }
      return;
}

/* eof */
