/* glpapi/glp_new_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_new_col - create new column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_new_col(LPI *lp, char *name);
--
-- *Description*
--
-- The routine glp_new_col creates a new column and adds it to the end
-- of the column list (thus, sequential number of the existing columns
-- remain unchanged). Initially the new column is empty, corresponds to
-- a continuous non-negative variable, and has zero coefficient in the
-- objective function.
--
-- The parameter name specifies a symbolic name, which is assigned to
-- the new column. If this parameter is NULL, no name is assigned.
--
-- *Complexity*
--
-- If the parameter name is not NULL, the time complexity is O(log n'),
-- where n' is number of named columns in the column list. Otherwise,
-- the time complexity is O(1). */

void glp_new_col(LPI *lp, char *name)
{     LPICOL *col;
      /* create new column */
      col = get_atom(lp->col_pool);
      col->seqn = lp->n+1;
      col->name = NULL;
      col->kind = 'C';
      col->type = 'L';
      col->lb = 0.0;
      col->ub = 0.0;
      col->fctr = 1.0;
      col->ptr = NULL;
      col->coef = 0.0;
      col->tagx = 'L';
      col->valx = 0.0;
      col->dx = 0.0;
      /* add new column to the end of the column list */
      if (lp->n == lp->n_max)
      {  /* enlarge the column list */
         LPICOL **temp;
         int j;
         lp->n_max += lp->n_max;
         temp = ucalloc(1+lp->n_max, sizeof(LPICOL *));
         for (j = 1; j <= lp->n; j++) temp[j] = lp->col[j];
         ufree(lp->col);
         lp->col = temp;
      }
      lp->n++;
      lp->col[lp->n] = col;
      /* assign column name */
      if (name != NULL)
      {  AVLNODE *node;
         if (glp_check_name(name))
            fault("glp_new_col: invalid column name");
         lp->col[lp->n]->name = create_str(lp->str_pool);
         set_str(lp->col[lp->n]->name, name);
         if (find_by_key(lp->col_tree, lp->col[lp->n]->name) != NULL)
            fault("glp_new_col: duplicate column name `%s'", name);
         node = insert_by_key(lp->col_tree, lp->col[lp->n]->name);
         node->link = lp->col[lp->n];
      }
      return;
}

/* eof */
