/* glprsm/check_dvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- check_dvec - check accuracy of the vector delta.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double check_dvec(RSM *rsm, double dvec[]);
--
-- *Description*
--
-- The check_dvec routine is intended for checking accuracy of the
-- vector delta. It computes the absolute error
--
--    e = max |delta'[i] - delta[i]|
--
-- where delta' is exact vector computed by means of the exact_dvec
-- routine, delta is approximate vector given in the array dvec.
--
-- This operation is extremely inefficient and may be used only for
-- debugging purposes.
--
-- *Returns*
--
-- The check_dvec routine returns the absolute error e (see above). */

double check_dvec(RSM *rsm, double dvec[])
{     int m = rsm->m, i;
      double d, dmax = 0.0;
      for (i = 1; i <= m; i++)
      {  d = fabs(exact_dvec(rsm, i) - dvec[i]);
         if (dmax < d) dmax = d;
      }
      return dmax;
}

/* eof */
