/* glpapi/glp_read_mps1.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include <string.h>
#include "glpk.h"
#include "glpmps.h"

/*----------------------------------------------------------------------
-- glp_read_mps1 - read problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- LPI *glp_read_mps1(char *fname);
--
-- *Description*
--
-- The routine glp_read_mps1 reads LP problem data using MPS format from
-- a text file whose name is fname.
--
-- *Returns*
--
-- If the operation was successful, the routine returns a pointer to the
-- created problem object. Otherwise the routine returns NULL. */

LPI *glp_read_mps1(char *fname)
{     LPI *lp = NULL;
      MPS *mps;
      int m, n, obj, i, j;
      /* read MPS data block */
      mps = load_mps(fname);
      if (mps == NULL) goto fail;
      m = mps->n_row;
      n = mps->n_col;
      obj = 0;
      /* create problem object */
      lp = glp_create_prob(mps->name);
      /* process ROW section */
      for (i = 1; i <= m; i++)
      {  MPSROW *row;
         int type;
         row = mps->row[i];
         glp_new_row(lp, row->name);
         if (strcmp(row->type, "N") == 0)
         {  type = 'F';
            if (obj == 0) obj = i;
         }
         else if (strcmp(row->type, "L") == 0)
            type = 'U';
         else if (strcmp(row->type, "G") == 0)
            type = 'L';
         else if (strcmp(row->type, "E") == 0)
            type = 'S';
         else
         {  error("glp_read_mps1: row `%s' has unknown type `%s'",
               row->name, row->type);
            goto fail;
         }
         glp_set_row_bnds(lp, i, type, 0.0, 0.0);
      }
      if (obj == 0)
      {  error("glp_read_mps1: objective row not found");
         goto fail;
      }
      /* process COLUMN section */
      for (j = 1; j <= n; j++)
      {  MPSCOL *col;
         MPSCQE *cqe;
         col = mps->col[j];
         glp_new_col(lp, col->name);
         glp_set_col_kind(lp, j, col->flag ? 'I' : 'C');
         for (cqe = col->ptr; cqe != NULL; cqe = cqe->next)
         {  glp_new_aij(lp, cqe->ind, j, cqe->val);
            if (cqe->ind == obj) glp_set_obj_coef(lp, j, cqe->val);
         }
      }
      /* process RHS section */
      if (mps->n_rhs > 0)
      {  MPSCQE *cqe;
         for (cqe = mps->rhs[1]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type;
            double lb, ub;
            glp_get_row_bnds(lp, cqe->ind, &type, NULL, NULL);
            switch (type)
            {  case 'F':
                  /* if the current row is the objective function row,
                     specified right-hand side is considered as the
                     constant term of the objective function with
                     opposite sign; in other cases specified right-hand
                     side is ignored */
                  if (cqe->ind == obj)
                     glp_set_obj_coef(lp, 0, -cqe->val);
                  lb = ub = 0.0;
                  break;
               case 'L':
                  lb = cqe->val, ub = 0.0;
                  break;
               case 'U':
                  lb = 0.0, ub = cqe->val;
                  break;
               case 'S':
                  lb = ub = cqe->val;
                  break;
               default:
                  insist(type != type);
            }
            glp_set_row_bnds(lp, cqe->ind, type, lb, ub);
         }
      }
      /* process RANGES section */
      if (mps->n_rng > 0)
      {  MPSCQE *cqe;
         for (cqe = mps->rng[1]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type;
            double lb, ub;
            glp_get_row_bnds(lp, cqe->ind, &type, &lb, &ub);
            switch (type)
            {  case 'F':
                  error("glp_read_mps1: range vector entry refers to ro"
                     "w `%s' of N type", mps->row[cqe->ind]->name);
                  goto fail;
               case 'L':
                  ub = lb + fabs(cqe->val);
                  break;
               case 'U':
                  lb = ub - fabs(cqe->val);
                  break;
               case 'S':
                  if (cqe->val >= 0.0)
                     ub += fabs(cqe->val);
                  else
                     lb -= fabs(cqe->val);
                  break;
               default:
                  insist(type != type);
            }
            glp_set_row_bnds(lp, cqe->ind, lb == ub ? 'S' : 'D', lb, ub)
               ;
         }
      }
      /* process BOUNDS section */
      if (mps->n_bnd > 0)
      {  MPSBQE *bqe;
         for (bqe = mps->bnd[1]->ptr; bqe != NULL; bqe = bqe->next)
         {  int type;
            double lb, ub;
            glp_get_col_bnds(lp, bqe->ind, &type, &lb, &ub);
            if (type == 'F' || type == 'U') lb = -DBL_MAX;
            if (type == 'F' || type == 'L') ub = +DBL_MAX;
            if (strcmp(bqe->type, "LO") == 0)
               lb = bqe->val;
            else if (strcmp(bqe->type, "UP") == 0)
               ub = bqe->val;
            else if (strcmp(bqe->type, "FX") == 0)
               lb = ub = bqe->val;
            else if (strcmp(bqe->type, "FR") == 0)
               lb = -DBL_MAX, ub = +DBL_MAX;
            else if (strcmp(bqe->type, "MI") == 0)
               lb = -DBL_MAX;
            else if (strcmp(bqe->type, "PL") == 0)
               ub = +DBL_MAX;
            else if (strcmp(bqe->type, "UI") == 0)
            {  /* integer structural variable with upper bound */
               glp_set_col_kind(lp, bqe->ind, 'I');
               ub = bqe->val;
            }
            else if (strcmp(bqe->type, "BV") == 0)
            {  /* binary structural variable */
               glp_set_col_kind(lp, bqe->ind, 'I');
               lb = 0.0, ub = 1.0;
            }
            else
            {  error("glp_read_mps1: bound vector entry for column `%s'"
                  " has unknown type `%s'",
                  mps->col[bqe->ind]->name, bqe->type);
               goto fail;
            }
            if (lb == -DBL_MAX && ub == +DBL_MAX)
               type = 'F';
            else if (ub == +DBL_MAX)
               type = 'L';
            else if (lb == -DBL_MAX)
               type = 'U';
            else if (lb != ub)
               type = 'D';
            else
               type = 'S';
            glp_set_col_bnds(lp, bqe->ind, type, lb, ub);
         }
      }
      /* free MPS data block */
      free_mps(mps);
      /* return to the application program */
      return lp;
fail: /* the operation failed */
      if (lp != NULL) glp_delete_prob(lp);
      if (mps != NULL) free_mps(mps);
      return NULL;
}

/* eof */
