/* glpapi/glp_clear_mat_rows.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_clear_mat_rows - nullify rows of constraint matrix.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_clear_mat_rows(LPI *lp, int flag[]);
--
-- *Description*
--
-- The routine glp_clear_mat_rows deletes all elements from specified
-- rows of the constraint matrix.
--
-- The array flag should have at least 1+m locations, where m is number
-- of rows in the problem object. The location flag[0] is not used.
-- If flag[i] is non-zero, the routine deletes all elements of the i-th
-- row. If flag[i] is zero, the i-th row is not changed.
--
-- *Complexity*
--
-- The time complexity is O(nz), where nz is total number of elements
-- in the constraint matrix before operation. */

void glp_clear_mat_rows(LPI *lp, int flag[])
{     int i, j;
      for (i = 1; i <= lp->m; i++)
         if (flag[i]) lp->row[i]->ptr = NULL;
      for (j = 1; j <= lp->n; j++)
      {  ELEM *ptr = NULL, *e;
         while (lp->col[j]->ptr != NULL)
         {  e = lp->col[j]->ptr;
            lp->col[j]->ptr = e->col;
            if (flag[e->i])
               free_atom(lp->elem_pool, e);
            else
               e->col = ptr, ptr = e;
         }
         lp->col[j]->ptr = ptr;
      }
      return;
}

/* eof */
