/* glpapi/glp_simplex.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpk.h"
#include "glprsm.h"

/*----------------------------------------------------------------------
-- glp_simplex - solve problem by means of the revised simplex method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_simplex(void);
--
-- *Description*
--
-- The glp_simplex routine is a LP problem solver based on the two-phase
-- revised simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_simplex routine performs the following:
--
-- preparing internal data structures;
-- searching for feasible basis solution (pahse I);
-- searching for optimal basis solution (phase II);
-- storing the computed solution to the workspace.
--
-- Since large-scale problems may take a long time, the glp_simplex
-- routine reports some visual information about current status of the
-- search. This information is sent to stdout approximately once per
-- second and has the following format:
--
--    *nnn:   objval = xxx   infsum = yyy (ddd)
--
-- where nnn is an iteration count, xxx is the current value of the
-- objective function (which is unscaled and has correct sign), yyy is
-- the sum of infeasibilities (which is scaled and therefore it may be
-- used only for visual estimating), ddd is the current number of basis
-- fixed variables. If the asterisk (*) precedes to nnn, the solver is
-- searching for optimal solution (i.e. the feasible solution has been
-- found yet), otherwise the solver is searching for some feasible
-- solution.
--
-- Please note that this solver is not perfect. Although it has been
-- successfully tested on a wide set of LP problems, there are so called
-- hard problems, which can't be solved by this solver.
--
-- *Control parameters*
--
-- The behavior of the glp_simplex routine depends on a set of control
-- parameters which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note, for example, if the problem
--     has no feasible solution, the solver returns zero code);
-- 1 - it's not possible to start solving the problem due to incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_simplex(void)
{     LP *lp;
      LPSOL *sol;
      struct rsm1_cp cp;
      int m, n, option, form, ret;
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL) return 1;
      m = lp->m, n = lp->n;
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* set control parameters */
      glp_get_ipar("option", &option);
      switch (option)
      {  case GLP_INI:     cp.what = 0; break;
         case GLP_ANY:     cp.what = 1; break;
         case GLP_FIN:     cp.what = 2; break;
         default:          insist(option != option);
      }
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     cp.form = 0; break;
         case GLP_RFI_BG:  cp.form = 1; break;
         case GLP_RFI_FT:  cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("scale", &cp.scale);
      glp_get_ipar("spx_use_dual", &cp.dual);
      glp_get_ipar("spx_steep", &cp.steep);
      glp_get_ipar("spx_relax", &cp.relax);
      glp_get_rpar("tol_bnd", &cp.tol_bnd);
      glp_get_rpar("tol_dj", &cp.tol_dj);
      glp_get_rpar("tol_piv", &cp.tol_piv);
      cp.iter_max = 0;
      glp_get_ipar("round", &cp.round);
      /* solve the problem by means of the revised simplex method */
      ret = rsm1_driver(lp, sol, &cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* free working data structures */
      delete_lp(lp);
      delete_lpsol(sol);
      /* return to the application program */
      return ret == 0 ? 0 : 2;
}

/* eof */
