/* glpapi/glp_dump_ws.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- glp_dump_ws - dump GLPK API workspace.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_dump_ws(char *fname, int what);
--
-- *Description*
--
-- The glp_dump_ws routine dumps information from GLPK API workspace to
-- the text file, whose name is the character string fname, in readable
-- format. This operation is intended for application debugging.
--
-- The parameter what specifies what information should be dumped. It
-- can be a combination of the following options:
--
-- GLP_D_PARS  dump control parameters;
-- GLP_D_ROWS  dump rows (auxiliary variables);
-- GLP_D_RMAT  dump constraint coefficients in row-wise format;
-- GLP_D_COLS  dump columns (structural variables);
-- GLP_D_CMAT  dump constraint coefficients in column-wise format;
-- GLP_D_ALL   dump all information (assumes all options above).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

static void dump_parameters(FILE *fp)
{     int iii;
      double rrr;
      char sss[GLP_MAX_NAME+1];
      glp_get_rpar("c0", &rrr);
      fprintf(fp, "c0             = %.8g\n", rrr);
      glp_get_cpar("fn_gener", sss);
      fprintf(fp, "fn_gener       = %s\n", sss);
      glp_get_ipar("mip_branch", &iii);
      fprintf(fp, "mip_branch     = %s\n",
         iii == GLP_FIRST ? "FIRST" : iii == GLP_LAST ? "LAST" :
         iii == GLP_DRTOM ? "DRTOM" : "???");
      glp_get_ipar("mip_btrack", &iii);
      fprintf(fp, "mip_btrack     = %s\n",
         iii == GLP_FIFO  ? "FIFO"  : iii == GLP_LIFO ? "LIFO" :
         iii == GLP_BESTP ? "BESTP" : "???");
      glp_get_cpar("mps_bnd_name", sss);
      fprintf(fp, "mps_bnd_name   = %s\n", sss);
      glp_get_cpar("mps_obj_name", sss);
      fprintf(fp, "mps_obj_name   = %s\n", sss);
      glp_get_ipar("mps_one_entry", &iii);
      fprintf(fp, "mps_one_entry  = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("mps_pedantic", &iii);
      fprintf(fp, "mps_pedantic   = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_cpar("mps_rhs_name", sss);
      fprintf(fp, "mps_rhs_name   = %s\n", sss);
      glp_get_cpar("mps_rng_name", sss);
      fprintf(fp, "mps_rng_name   = %s\n", sss);
      glp_get_ipar("mps_skip_empty", &iii);
      fprintf(fp, "mps_skip_empty = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("mps_use_names", &iii);
      fprintf(fp, "mps_use_names  = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("nc", &iii);
      fprintf(fp, "nc             = %d\n", iii);
      glp_get_ipar("nc_bin", &iii);
      fprintf(fp, "nc_bin         = %d\n", iii);
      glp_get_ipar("nc_int", &iii);
      fprintf(fp, "nc_int         = %d\n", iii);
      glp_get_ipar("nr", &iii);
      fprintf(fp, "nr             = %d\n", iii);
      glp_get_ipar("nz", &iii);
      fprintf(fp, "nz             = %d\n", iii);
      glp_get_ipar("obj_dir", &iii);
      fprintf(fp, "obj_dir        = %s\n",
         iii == GLP_MIN ? "MIN" : iii == GLP_MAX ? "MAX" : "???");
      glp_get_cpar("obj_row", sss);
      fprintf(fp, "obj_row        = %s\n", sss);
      glp_get_ipar("option", &iii);
      fprintf(fp, "option         = %s\n",
         iii == GLP_INI ? "INI" : iii == GLP_ANY ? "ANY" :
         iii == GLP_FIN ? "FIN" : "???");
      glp_get_cpar("problem", sss);
      fprintf(fp, "problem        = %s\n", sss);
      glp_get_ipar("round", &iii);
      fprintf(fp, "round          = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("scale", &iii);
      fprintf(fp, "scale          = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_form", &iii);
      fprintf(fp, "spx_form       = %s\n",
         iii == GLP_EFI    ? "EFI"    :
         iii == GLP_RFI_BG ? "RFI_BG" :
         iii == GLP_RFI_FT ? "RFI_FT" : "???");
      glp_get_ipar("spx_relax", &iii);
      fprintf(fp, "spx_relax      = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_steep", &iii);
      fprintf(fp, "spx_steep      = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_use_dual", &iii);
      fprintf(fp, "spx_use_dual   = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("status", &iii);
      fprintf(fp, "status         = %s\n",
         iii == GLP_UNDEF  ? "UNDEF"  :
         iii == GLP_OPT    ? "OPT"    :
         iii == GLP_FEAS   ? "FEAS"   :
         iii == GLP_INFEAS ? "INFEAS" :
         iii == GLP_NOFEAS ? "NOFEAS" :
         iii == GLP_UNBND  ? "UNBND"  : "???");
      glp_get_ipar("sum_aij", &iii);
      fprintf(fp, "sum_aij        = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_rpar("tol_aij", &rrr);
      fprintf(fp, "tol_aij        = %.8g\n", rrr);
      glp_get_rpar("tol_bnd", &rrr);
      fprintf(fp, "tol_bnd        = %.8g\n", rrr);
      glp_get_rpar("tol_dj", &rrr);
      fprintf(fp, "tol_dj         = %.8g\n", rrr);
      glp_get_rpar("tol_int", &rrr);
      fprintf(fp, "tol_int        = %.8g\n", rrr);
      glp_get_rpar("tol_obj", &rrr);
      fprintf(fp, "tol_obj        = %.8g\n", rrr);
      glp_get_rpar("tol_piv", &rrr);
      fprintf(fp, "tol_piv        = %.8g\n", rrr);
      return;
}

/*--------------------------------------------------------------------*/

static void dump_items(FILE *fp, int what, int mat)
{     int ret;
      for (ret = glp_first_item(what); ret == 0;
           ret = glp_next_item(what))
      {  char name[GLP_MAX_NAME+1];
         int kind, type, tagx;
         double lb, ub, valx, dx;
         glp_get_name(what, name);
         glp_get_kind(what, &kind);
         glp_get_bounds(what, &type, &lb, &ub);
         glp_get_activity(what, &tagx, &valx, &dx);
         glp_get_name(what, name);
         if (strlen(name) <= 15)
            fprintf(fp, "%-15s ", name);
         else
            fprintf(fp, "%s\n%16s", name, "");
         /* row/column kind */
         fprintf(fp, "%s ",
            kind == GLP_NO ? " " : kind == GLP_YES ? "I" : "?");
         /* row/column type, lower bound, and upper bound */
         switch (type)
         {  case GLP_FR:
               fprintf(fp, "FR");
               break;
            case GLP_LO:
               fprintf(fp, "LO %15.8g", lb);
               break;
            case GLP_UP:
               fprintf(fp, "UP %15s %15.8g", "", ub);
               break;
            case GLP_DB:
               fprintf(fp, "DB %15.8g %15.8g", lb, ub);
               break;
            case GLP_FX:
               fprintf(fp, "FX %15.8g %15s", lb, "=");
               break;
            default:
               fprintf(fp, "?? %15.8g %15.8g", lb, ub);
               break;
         }
         fprintf(fp, "\n%18s", "");
         /* row/column status, primal value, and dual value */
         fprintf(fp, "%s %15.8g %15.8g",
            tagx == GLP_BS ? "BS" : tagx == GLP_NL ? "NL" :
            tagx == GLP_NU ? "NU" : tagx == GLP_NF ? "NF" :
            tagx == GLP_NS ? "NS" : "??", valx, dx);
         fprintf(fp, "\n");
         /* constraint coefficients */
         if (mat)
         {  int ret;
            for (ret = glp_first_coef(what); ret == 0;
                 ret = glp_next_coef(what))
            {  double coef;
               glp_get_name(what == GLP_ROW ? GLP_COL : GLP_ROW, name);
               glp_get_coef(&coef);
               fprintf(fp, "%15.8g (%s)\n", coef, name);
            }
         }
      }
      return;
}

/*--------------------------------------------------------------------*/

int glp_dump_ws(char *fname, int what)
{     FILE *fp;
      print("glp_dump_ws: dumping workspace to `%s'...", fname);
      /* open the output text file */
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_dump_ws: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* print header */
      fprintf(fp, "*** DUMP OF GLPK API WORKSPACE ***\n");
      /* dump control parameters */
      if (what & GLP_D_PARS)
      {  fprintf(fp, "\n\n*** Control Parameters ***\n\n");
         dump_parameters(fp);
      }
      /* dump rows and constraint coefficients for each row */
      if (what & GLP_D_ROWS)
      {  fprintf(fp, "\n\n*** Rows (Auxiliary Variables) ***\n\n");
         dump_items(fp, GLP_ROW, what & GLP_D_RMAT);
      }
      /* dump columns and constraint coefficients for each column */
      if (what & GLP_D_COLS)
      {  fprintf(fp, "\n\n*** Columns (Structural Variables) ***\n\n");
         dump_items(fp, GLP_COL, what & GLP_D_CMAT);
      }
      /* end of dump */
      fprintf(fp, "\n\n*** END OF DUMP ***\n");
      /* close the output text file */
      fflush(fp);
      if (ferror(fp))
      {  error("glp_dump_ws: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      /* return to the calling program */
      return 0;
fail: /* the operation failed */
      if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
