/* glpapi/glp_interior.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"
#include "glpipm.h"

/*----------------------------------------------------------------------
-- glp_interior - solve problem by means of the interior point method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_interior(void);
--
-- *Description*
--
-- The glp_interior routine is a *tentative* LP problem solver based on
-- the primal-dual interior point method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_interior routine performs the following actions:
--
-- preparing internal data structures;
-- transforming original LP problem to the standard form;
-- searching for optimal solution;
-- recovering found solution for the original LP problem;
-- storing computed solution back to the workspace.
--
-- On each iteration the glp_interior routine reports some information
-- about the current point. This information is sent to stdout and has
-- the following format:
--
--    nnn: F = fff; rpi = ppp; rdi = ddd; gap = ggg
--
-- where nnn is number of iteration (0 means the starting point), fff
-- is current value of the objective function (in case of maximization
-- this value has opposite sign), ppp is relative primal infeasibility,
-- ddd is relative dual infeasibility, ggg is relative primal-dual gap
-- (the difference between values of primal and dual objectives).
--
-- Note that the glp_interior solver is *tentative* and doesn't include
-- many important features (no dense column handling, no iterative
-- refinement of the solution, no optimal basis identification, etc.).
--
-- *Returns*
--
-- 0 - optimal solution found;
-- 1 - the solver can't start solving the problem because of incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_interior(void)
{     LP *lp;
      LPSOL *sol;
      int m, n, option, status, kase, ret;
      /* determine kind of the desired solution */
      glp_get_ipar("option", &option);
      if (option == GLP_INI)
      {  error("glp_interior: the interior point solver is not intended"
            " for computing initial basis solution");
         return 1;
      }
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL) return 1;
      m = lp->m, n = lp->n;
      /* if only primal feasible solution is needed, nullify objective
         function coefficients */
      if (option == GLP_ANY)
      {  int j;
         for (j = 0; j <= n; j++) lp->c[j] = 0.0;
      }
      /* warn about dense columns */
      if (m > 200)
      {  int j;
         for (j = 1; j <= n; j++)
         {  if (count_nz(lp->A, -j) > m / 10)
            {  print("*WARNING* THE PROBLEM HAS DENSE COLUMN(S)");
               break;
            }
         }
      }
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* solve LP problem using primal-dual interior point method */
      ret = ipm1_driver(lp, sol);
      if (ret == 0)
         status = (option == GLP_ANY ? GLP_FEAS : GLP_OPT);
      else
         status = GLP_UNDEF;
      /* store solution back to the workspace (incomplete) */
      glp_set_ipar("status", status);
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int k, seqn;
            glp_get_seqn(what, &seqn);
            k = (kase == 0 ? 0 : m) + seqn;
            /* all variables are superbasis */
            glp_set_activity(what, GLP_BS, sol->valx[k], sol->dx[k]);
         }
      }
      /* delete LP solution block */
      delete_lpsol(sol);
      /* delete LP data block */
      delete_lp(lp);
      /* returns to the application program */
      return (status != GLP_UNDEF ? 0 : 2);
}

/* eof */
